#' Create a FeatureTrack for a custom JBrowse 2 view
#'
#' Creates the necessary configuration string for an
#' indexed GFF3 file so that it can be used
#' in a JBrowse custom linear genome view.
#'
#' It is important to note that while only the GFF3 file is
#' passed as an argument, \code{tracks_variant} assumes that a GFF3
#' index of the same name is located with the file
#'
#' For example:
#'
#' \code{track_feature("data/features.gff")}
#'
#' Assumes that \code{data/features.gff.tbi} also exists.
#'
#' This is a JBrowse 2 convention, and the default naming output of tabix
#'
#' For more information on creating an index with tabix, visit
#' \url{https://www.htslib.org/}
#'
#' @param track_data the URL to the GFF3 file
#' @param assembly the config string generated by \code{assembly}
#'
#' @return a character vector of stringified FeatureTrack JSON configuration
#'
#' @export
#'
#' @examples
#' assembly <- assembly("https://jbrowse.org/genomes/hg19/fasta/hg19.fa.gz", bgzip = TRUE)
#'
#' track_feature("features.gff", assembly)
track_feature <- function(track_data, assembly) {
  check_gff(track_data)

  type <- "FeatureTrack"
  name <- get_name(track_data)
  assembly_name <- get_assembly_name(assembly)
  track_id <- stringr::str_c(assembly_name, "_", name)
  index <- stringr::str_c(track_data, ".tbi")

  as.character(
    stringr::str_glue(
      "{{ ",
      '"type": "{type}", ',
      '"name": "{name}", ',
      '"assemblyNames": ["{assembly_name}"], ',
      '"trackId": "{track_id}", ',
      '"adapter": {{ ',
      '"type": "Gff3TabixAdapter", ',
      '"gffGzLocation": {{ "uri": "{track_data}" }}, ',
      '"index": {{ "location": {{ "uri": "{index}"  }} }}',
      "}}",
      "}}"
    )
  )
}

check_gff <- function(track_data) {
  track_non_gz <- strip_gz(track_data)
  if (!stringr::str_ends(track_non_gz, ".gff") && !stringr::str_ends(track_non_gz, ".gff3")) {
    stop("feature data must be GFF3. Use .gff or .gff3 extension")
  }
}
