% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/0_loglik-LambertW-utils.R, R/loglik_LambertW.R, R/loglik_input.R, R/loglik_penalty.R
\name{loglik-LambertW-utils}
\alias{loglik-LambertW-utils}
\alias{loglik_LambertW}
\alias{loglik_input}
\alias{loglik_penalty}
\title{Log-Likelihood for Lambert W\eqn{\times} F RVs}
\usage{
loglik_LambertW(theta, y, distname, type, return.negative = FALSE,
  flattened.theta.names = names(theta))

loglik_input(beta, x, distname, dX = NULL, log.dX = function(x, beta)
  log(dX(x, beta)))

loglik_penalty(tau, y, type = c("h", "hh", "s"), is.non.negative = FALSE)
}
\arguments{
\item{theta}{list; a (possibly incomplete) list of parameters \code{alpha}, \code{beta}, \code{gamma},
\code{delta}. \code{\link{complete_theta}} fills in default values for missing entries.}

\item{y}{a numeric vector of real values (the observed data).}

\item{distname}{string; name of input distribution; see \code{\link{get_distnames}}.}

\item{type}{type of Lambert W \eqn{\times} F distribution: skewed \code{"s"};
heavy-tail \code{"h"}; or skewed heavy-tail \code{"hh"}.}

\item{return.negative}{logical; if \code{TRUE} it returns the
negative log-likelihood as a scalar (which is useful for numerical \emph{minization}
algorithms that do \emph{maximum} likelihood estimation); otherwise it returns a
list of input log-likelihood, penalty, and their sum = full likelihood. Default: \code{FALSE}.}

\item{flattened.theta.names}{vector of strings with names of flattened \code{theta};
this is necessary for optimization functions since they drop the names of a vector,
but all functions in this package use names to address specific elements of
(the flattened) \code{theta}.}

\item{beta}{vector; parameter \eqn{\boldsymbol \beta} of the input distribution.
See \code{\link{check_beta}} on how to specify \code{beta} for each distribution.}

\item{x}{a numeric vector of real values (the \emph{input} data).}

\item{dX}{optional; density function of \code{x}. Common distributions are already
built-in (see \code{distname}). If you want to supply your own density, you
\strong{must} supply a function of \code{x} and \code{beta} and set \code{distname = "user"}.}

\item{log.dX}{optional; a function that returns the logarithm of the density function
of \code{x}. Often -- in particular for exponential families --
the \eqn{\log} of \eqn{f_X(x)} has a simpler form (and is thus faster to evaluate).}

\item{tau}{named vector \eqn{\tau} which defines the variable transformation. Must have at least
\code{'mu_x'} and \code{'sigma_x'} element; see \code{\link{complete_tau}} for details.}

\item{is.non.negative}{logical; tell the penalty function if the data is from
a non-negative distribution;
by default it sets it to \code{TRUE} if the distribution is not a location but a scale family.
 If \code{TRUE}, the penalty can be computed; otherwise, it returns \code{NA}.}
}
\value{
\code{loglik_input} and \code{loglik_penalty} return a scalar;
\code{loglik_LambertW} returns a list with 3 values:

\item{loglik.input}{ loglikelihood of \code{beta} given the transformed data,}
\item{loglik.penalty}{ penalty for transforming the data,}
\item{loglik.LambertW}{ total log-likelihood of \code{theta} given the observed data;
if the former two values exist this is simply their sum.}
}
\description{
Evaluates the log-likelihood for \eqn{\theta} given observed data. See Details.

\code{loglik_LambertW} computes the log-likelihood of \eqn{\theta}
for a Lambert W \eqn{\times} F distribution given observations \code{y}.

\code{loglik_input} computes the log-likelihood of various distributions for
the parameter \eqn{\boldsymbol \beta} given the data \code{x}. This can be
used independently of the Lambert W framework to compute
the log-likelihood of parameters for common distributions.

\code{loglik_penalty} computes the penalty for transforming the data back to
the input. Note that this penalty is independent of the distribution
specified by \code{distname}; it only depends on \eqn{\tau}.

If \code{type = "s"} then the penalty term exists if the distribution
is non-negative (see \code{get_distname_family}) and \code{gamma >= 0};
otherwise, it returns \code{NA}.
}
\details{
For heavy-tail Lambert W\eqn{\times} F distributions (\code{type = "h"} or \code{type =
"hh"}) the log-likelihood decomposes into an input log-likelihood plus a penalty term
for transforming the data.
For skewed Lambert W \eqn{\times} F distributions this decomposition only exists for non-negative input RVs
 (e.g., \code{"exp"}onential, \code{"gamma"}, \code{"f"}, \ldots{}). If negative values are possible
 (\code{"normal"}, \code{"t"}, \code{"unif"}, \code{"cauchy"}, \ldots{}) then \code{loglik_input} and \code{loglik_penalty}
 return \code{NA}, but the value of the output log-likelihood will still be returned correctly
as \code{loglik.LambertW}.

See References for details on the decomposition of the log-likelihood into
a log-likelihood on the input parameters plus a penalty term for transforming the data.
}
\examples{
set.seed(1)
yy <- rLambertW(n = 1000, distname = "normal",
                theta = list(beta = c(0, 1), delta = 0.2))
loglik_penalty(tau = theta2tau(list(beta = c(1, 1), delta = c(0.2, 0.2)),
                               distname = "normal"),
               y = yy, type = "hh")
# For a type = 's' Lambert W x F distribution with location family input
# such a decomposition doesn't exist; thus NA.
loglik_penalty(tau = theta2tau(list(beta = c(1, 1), gamma = 0.03),
                               distname = "normal"),
               is.non.negative = FALSE,
               y = yy, type = "s")
# For scale-family input it does exist
loglik_penalty(tau = theta2tau(list(beta = 1, gamma = 0.01),
                               distname = "exp"),
               is.non.negative = TRUE,
               y = yy, type = "s")

# evaluating the Gaussian log-likelihood
loglik_input(beta = c(0, 1), x = yy, distname = "normal") # built-in version
# or pass your own log pdf function
loglik_input(beta = c(0, 1), x = yy, distname = "user",
             log.dX = function(xx, beta = beta) {
                dnorm(xx, mean = beta[1], sd = beta[2], log = TRUE)
             })
\dontrun{
# you must specify distname = 'user'; otherwise it does not work
loglik_input(beta = c(0, 1), x = yy, distname = "mydist",
             log.dX = function(xx, beta = beta) {
                dnorm(xx, mean = beta[1], sd = beta[2], log = TRUE)
                })
}

### loglik_LambertW returns all three values
loglik_LambertW(theta = list(beta = c(1, 1), delta = c(0.09, 0.07)),
                y = yy, type = "hh", distname ="normal")

# can also take a flattend vector; must provide names though for delta
loglik_LambertW(theta = flatten_theta(list(beta = c(1, 1),
                                          delta = c(delta_l = 0.09,
                                                    delta_r = 0.07))),
                y = yy, type = "hh", distname ="normal")
}
\references{
Goerg, G.M. (2014). \dQuote{The Lambert Way to Gaussianize heavy-tailed data
with the inverse of Tukey's h transformation as a special case.}
 The Scientific World Journal: Probability and Statistics with
Applications in Finance and Economics. Available at
 \url{http://www.hindawi.com/journals/tswj/aa/909231/}.
}
\keyword{distribution}
\keyword{univar}

