# Purpose: Fitting function for fitting a single multivariate normal
# distribution in the presence of missingness. 
# Updated: 20/07/18

#-------------------------------------------------------------------------------

#' Estimation for a Single Component Multivariate Normal with No Missingness.
#'
#' @param data Numeric data matrix.
#' @param init_mean Optional initial mean vector.
#' @param fix_mean Fix the means to their starting value? Must initialize
#' @return List containing the new `Mean`, `Covariance`, and log-likelihood
#'   `Objective`.

fit.mvn.no_miss <- function(
  data,
  init_mean,
  fix_mean
  ) {
  
  n <- nrow(data)
  d <- ncol(data)
  orig_col_names <- colnames(data)
  
  # Mean.
  if(fix_mean){
    new_mean <- init_mean
  } else {
    new_mean <- apply(data, 2, mean)
  }
  
  # Covariance.
  new_cov <- matCov(data, data)
  new_cov_inv <- matInv(new_cov)
  rownames(new_cov_inv) = colnames(new_cov_inv) <- orig_col_names
  
  # Objective (log-likelihood).
  n <- nrow(data)
  mean_mat <- matrix(data = new_mean, nrow = n, ncol = d, byrow = TRUE)
  resid <- data - mean_mat
  objective <- -n * log(det(new_cov)) - tr(MMP(matInv(new_cov), matIP(resid, resid)))
  
  # Output
  out <- list(
    "Mean" = new_mean, 
    "Covariance" = new_cov, 
    "Objective" = objective
  )
  return(out)
}

#-------------------------------------------------------------------------------

#' Paratition a Data.frame by Missingness cases. 
#' 
#' @param data Data.frame.
#' @return List containing:
#' \itemize{
#'   \item The original row and column names: `orig_row_names`, `orig_col_names`.
#'   \item The original row and column numbers: `n_row` and `n_col`.
#'   \item The complete cases `data_comp`.
#'   \item The incomplete cases `data_incomp`.
#'   \item The empty cases `data_empty`.
#'   \item Counts of complete `n0`, incomplete `n1`, and empty `n2` cases.
#'   \item Initial order of the observations `init_order`. 
#' }

PartitionData <- function(data){
  
  d <- ncol(data)
  idx <- seq(1:nrow(data))
  is_comp <- complete.cases(data)
  is_incomp <- !is_comp
  
  # Complete cases
  data_comp <- data[is_comp, , drop = FALSE]
  idx_comp <- idx[is_comp]
  
  # Incomplete cases
  data_incomp <- data[is_incomp, , drop = FALSE]
  idx_incomp <- idx[is_incomp]
  
  # Empty cases
  is_empty <- apply(data_incomp, 1, function(x){
    sum(is.na(x)) == d
  })
  data_empty <- data_incomp[is_empty, , drop = FALSE]
  idx_empty <- idx_incomp[is_empty]
  
  # Remove empty cases
  data_incomp <- data_incomp[!is_empty, ]
  idx_incomp <- idx_incomp[!is_empty]
  
  # Output
  out <- list()
  out$orig_row_names <- rownames(data)
  out$orig_col_names <- colnames(data)
  
  out$n_row <- nrow(data)
  out$n_col <- ncol(data)
  
  out$n0 <- nrow(data_comp)
  out$n1 <- nrow(data_incomp)
  out$n2 <- nrow(data_empty)
  
  out$data_comp <- data_comp
  out$data_incomp <- data_incomp
  out$data_empty <- data_empty
  
  out$init_order <- c(idx_comp, idx_incomp, idx_empty)
  return(out)
}


#-------------------------------------------------------------------------------

#' Parameter Initialization for Single Component Multivariate Normal with Missingness. 
#' 
#' @param data_comp Complete cases. 
#' @param init_mean Optional initial mean vector.
#' @param init_cov Optional initial covariance matrix.
#' @return List containing the initialized `mean` and `cov`. 

fit.mvn.miss.init <- function(
  data_comp,
  init_mean,
  init_cov
) {
  
  # Output structure. 
  theta0 <- list()
  n <- nrow(data_comp)
  
  # Case 1: Both init_means and init_cov provided.
  if (!is.null(init_mean) & !is.null(init_cov)) {
    
    theta0$mean <- init_mean
    theta0$cov <- init_cov
    
    # Case 2: At least one of init_means or S0 is null
  } else {
    
    # Check for complete obs
    if (n == 0) {
      stop("If there are no complete cases, initial values are required for all parameters.")
    }
    
    # Initialize mean if null. 
    if (is.null(init_mean)) {
      theta0$mean <- apply(data_comp, 2, mean)
    } else {
      theta0$mean <- init_mean
    }
    
    # Initialize covariance if null.
    if (is.null(init_cov)) {
      theta0$cov <- matCov(data_comp, data_comp)
    } else {
      theta0$cov <- init_cov
    }
    
  } # End case 2. 
  
  return(theta0)
}


#-------------------------------------------------------------------------------

#' Parameter Update for Single Component Multivariate Normal with Missingness. 
#' 
#' @param split_data Data partitioned by missingness.
#' @param theta List containing the current `mean` and `cov`. 
#' @param fix_mean Fix the mean to its starting value? Must initialize. 
#' @return List containing:
#' \itemize{
#'   \item The updated `mean` and `cov`.
#'   \item The initial `old_obj` and final `new_obj` EM objective. 
#'   \item The increase in the EM objective `delta`. 
#' }

fit.mvn.miss.update <- function(
  split_data,
  theta,
  fix_mean
) {
  
  # Unpack
  ## Data
  data_comp <- split_data$data_comp
  n0 <- split_data$n0
  d <- split_data$n_col
  
  data_incomp <- split_data$data_incomp
  n1 <- split_data$n1
  n <- n0 + n1
  
  ## Parameters
  old_mean <- theta$mean
  old_cov <- theta$cov
  
  # Initial objective.
  resid_cov <- array(0, c(d, d))
  
  ## Complete cases.
  if(n0 > 0){
    mean_mat <- matrix(data = old_mean, nrow = n0, ncol = d, byrow = TRUE)
    resid_mat <- data_comp - mean_mat
    resid_cov <- resid_cov + matIP(resid_mat, resid_mat)
  }
  
  ## Incomplete cases.
  if(n1 > 0){
    resid_cov <- resid_cov + ExpResidOP(
      data_incomp,
      old_mean,
      old_mean,
      old_cov
    )
  }
 
  ## Initial EM objective. 
  old_obj <- (-n) * log(det(old_cov)) - tr(MMP(matInv(old_cov), resid_cov))
  
  # Update mean:
  if (fix_mean) {
    new_mean <- old_mean
  } else {
    new_mean <- 0
    
    ## Complete cases.
    if (n0 > 0) {
      new_mean <- new_mean + apply(data_comp, 2, sum)
    }
    
    ## Incomplete cases.
    if (n1 > 0) {
      working_response <- WorkResp(data_incomp, old_mean, old_cov)
      new_mean <- new_mean + apply(working_response, 2, sum)
    }
    
    ## Normalize
    new_mean <- (new_mean / n)
  }
  
  # Update covariance
  new_cov <- array(0, c(d, d))
  
  ## Complete observations
  if (n0 > 0) {
    mean_mat <- matrix(data = new_mean, nrow = n0, ncol = d, byrow = TRUE)
    resid_mat <- data_comp - mean_mat
    resid_cov <- matIP(resid_mat, resid_mat)
    new_cov <- new_cov + resid_cov
  }
  
  ## Incomplete observations
  if (n1 > 0) {
    new_cov <- new_cov + ExpResidOP(
      data_incomp,
      new_mean,
      old_mean,
      old_cov
    )
  }
  
  ## Normalize
  new_cov <- (new_cov / n)
  rownames(new_cov) <- colnames(new_cov) <- split_data$orig_col_names
  
  ## Final EM objective.
  new_obj <- (-n) * log(det(new_cov)) - (d * n)
  
  # Increment
  delta <- new_obj - old_obj
  
  # Output
  out <- list()
  out$mean <- new_mean
  out$cov <- new_cov
  out$new_obj <- new_obj
  out$old_obj <- old_obj
  out$delta <- delta
  return(out)
}


#-------------------------------------------------------------------------------

#' Maximization Iteration.
#'
#' @param theta0 List containing the initial values of `mean` and `cov`.
#' @param Update Function to iteratively update theta0.
#' @param maxit Maximum number of EM iterations.
#' @param eps Minimum acceptable increment in the EM objective.
#' @param report Report fitting progress?
#' @return List containing the updated `mean` and `cov`, the final EM objective
#'   `new_obj`. 

Maximization <- function(
  theta0,
  Update,
  maxit,
  eps,
  report 
) {
  
  # Update until max iterations or tolderance are reached. 
  for (i in 1:maxit) {
    
    # Update:
    theta1 <- Update(theta0)
   
    # Accept if increment in EM objective is positive.
    if (theta1$delta > 0) {
      theta0 <- theta1
      if (report) {
        cat("Objective increment: ", signif(theta1$d, digits = 3), "\n")
      }
    }
    
    # Terminate if increment is below tolerance.
    if (theta1$delta < eps) {
      # If EM failes to perform any updates, e.g. the iteration is initialized
      # at the MLE, keep initial objective.
      if (i == 1) {
        theta0$new_obj <- theta1$old_obj
      }
      break
    }
  }
  
  # Fitting report
  if (report) {
    if (i < maxit) {
      cat(paste0(i - 1, " update(s) performed before reaching tolerance limit.\n\n"))
    } else {
      cat(paste0(i, " update(s) performed without reaching tolerance limit.\n\n"))
    }
  }
  
  # Output.
  return(theta0)
}


#-------------------------------------------------------------------------------

#' Imputation for a Single Component Multivariate Normal with Missingness. 
#' 
#' @param split_data Data partitioned by missingness.
#' @param theta List containing the `mean` and `cov`.
#' @return Data.matrix, in the same order as the original data, with missing values
#'   imputed to their expectations. 

fit.mvn.miss.impute <- function(
  split_data,
  theta
) {
  
  d <- split_data$n_col 
  out <- matrix(NA, nrow = 0, ncol = d)
  
  # Complete data.
  n0 <- split_data$n0
  if(n0 > 0){
    out <- rbind(out, split_data$data_comp)
  }
  
  # Impute data.
  n1 <- split_data$n1
  if(n1 > 0){
    data_imp <- WorkResp(split_data$data_incomp, theta$mean, theta$cov)
    out <- rbind(out, data_imp)
  }
  
  # Empty data.
  n2 <- split_data$n2
  if(n2 > 0){
    data_empty <- matrix(data = theta$mean, nrow = n2, ncol = d, byrow = TRUE)
    out <- rbind(out, data_empty)
  }
  
  # Restore initial order. 
  init_order <- split_data$init_order
  out <- out[order(init_order), ]
  
  # Output
  rownames(out) <- split_data$orig_row_names
  colnames(out) <- split_data$orig_col_names
  return(out)
}


#-------------------------------------------------------------------------------

#' Estimation for a Single Component Multivariate Normal with Missingness.
#'
#' @param data Numeric data matrix.
#' @param init_mean Optional initial mean vector.
#' @param fix_mean Fix the means to their starting value? Must initialize.
#' @param init_cov Optional initial covariance matrix.
#' @param maxit Maximum number of EM iterations.
#' @param eps Minimum acceptable increment in the EM objective.
#' @param report Report fitting progress?
#' @return List containing the final `Mean`, `Covariance`, EM `Objective` and a
#'   `Completed` data matrix.

fit.mvn.miss <- function(
  data,
  init_mean,
  fix_mean,
  init_cov,
  maxit,
  eps,
  report 
) {
  
  # Partitioned data
  split_data <- PartitionData(data)
  
  # Initialize
  theta0 <- fit.mvn.miss.init(
    split_data$data_comp,
    init_mean,
    init_cov
  )
  
  # Maximzation
  Update <- function(theta){fit.mvn.miss.update(split_data, theta, fix_mean)}
  theta1 <- Maximization(theta0, Update, maxit, eps, report)
  
  # Imputation
  imputed <- fit.mvn.miss.impute(split_data, theta1)
  
  # Output
  out <- list()
  out$Mean <- theta1$mean
  out$Covariance <- theta1$cov
  out$Objective <- theta1$new_obj
  out$Completed <- imputed
  return(out)
}


#-------------------------------------------------------------------------------
# Main Function
#-------------------------------------------------------------------------------

#' Fit Multivariate Normal Distribution
#'
#' Given a matrix of n x d-dimensional random vectors, possibly containing
#' missing elements, estimates the mean and covariance of the best fitting
#' multivariate normal distribution.
#'
#' @param data Numeric data matrix.
#' @param init_mean Optional initial mean vector.
#' @param fix_mean Fix the mean to its starting value? Must initialize. 
#' @param init_cov Optional initial covariance matrix.
#' @param maxit Maximum number of EM iterations.
#' @param eps Minimum acceptable increment in the EM objective.
#' @param report Report fitting progress?
#' @importFrom stats complete.cases
#' @return List containing the final `Mean`, `Covariance`, and EM `Objective`. 
#'   If missing data are present, a `Completed` data matrix is also returned. 

fit.mvn <- function(
  data, 
  init_mean = NULL, 
  fix_mean = FALSE, 
  init_cov = NULL, 
  maxit = 100, 
  eps = 1e-6, 
  report = TRUE
  ) {
  
  # Check for missingness
  mis <- sum(is.na(data)) > 0

  if (!mis) {
    ## If missingness is absent:
    out <- fit.mvn.no_miss(
      data,
      init_mean,
      fix_mean
    )
  } else {
    ## If missingness is present:
    out <- fit.mvn.miss(
      data,
      init_mean,
      fix_mean,
      init_cov,
      maxit,
      eps,
      report
    )
  }
  return(out)
}
