#' Multiplicative bias-corrected OLS estimator
#'
#' Computes the multiplicative bias correction (BCM) for an OLS
#' regression when the primary regressor is measured by an ML/AI method.
#'
#' @param Y Numeric vector (or one-column matrix) of responses.
#' @param Xhat Numeric matrix of regressors; **the first column** must be
#'   the ML-generated regressor whose bias we’re correcting, and remaining
#'   columns are any additional “true” controls.
#' @param fpr Numeric scalar.  Estimated false-positive rate of the
#'   generated regressor (proportion of ML positives that are actually negatives).
#' @param m Integer.  Size of the **validation/labeled** subsample used to
#'   estimate `fpr` — i.e.\ the number of observations where you observe both
#'   the ML prediction (`Xhat`) and the true regressor.
#' @param intercept logical, `TRUE` by default.
#'
#' @return An object of class `mlbc_fit` (and subclass `mlbc_bcm`) with two components:
#'   * `coef`: Numeric vector of bias-corrected regression coefficients.
#'   * `vcov`: Variance-covariance matrix for those coefficients.
#'
#' @examples
#' # generate data
#' Nunl     <- 10000
#' Xtrue_unl<- rbinom(Nunl, 1, 0.2)
#' Xhat_unl <- ifelse(runif(Nunl) < 0.1, 1, Xtrue_unl)
#' Y_unl     <- 5 + 2*Xtrue_unl + rnorm(Nunl)
#' #estimate the false-positive rate
#' nval     <- 100
#' Xtrue_val<- rbinom(nval, 1, 0.2)
#' Xhat_val <- ifelse(runif(nval) < 0.1, 1, Xtrue_val)
#' Y_val     <- 5 + 2*Xtrue_val + rnorm(nval)
#' fpr_hat  <- mean(Xhat_val==1 & Xtrue_val==0)
# #correct with an intercept
#' fit_bcm <- ols_bcm(Y_unl,
#'                   Xhat = matrix(Xhat_unl, ncol=1),
#'                   fpr = fpr_hat,
#'                   m   = nval,
#'                   intercept = TRUE)
#' summary(fit_bcm)
#' @references Battaglia, Christensen, Hansen, and Sacher (2025). "Inference for Regression with Variables Generated by AI or Machine Learning".
#' @seealso \link{ols_bca} for the additive correction.
#' @export

ols_bcm <- function(Y, Xhat, fpr, m, intercept = TRUE) {

  if (intercept) {
    Xhat    <- cbind(Intercept = 1, Xhat)
    gen_idx <- 2L
  } else {
    gen_idx <- 1L
  }


  orig <- ols(Y, Xhat, se = TRUE)
  b0   <- as.numeric(orig$coef)
  V0   <- orig$vcov
  sXX  <- orig$sXX
  d    <- length(b0)


  A       <- matrix(0, d, d)
  A[gen_idx, gen_idx] <- 1
  Gamma   <- solve(sXX, A)
  I       <- diag(d)
  Minv    <- solve(I - fpr * Gamma)


  b_raw   <- Minv %*% b0


  V1      <- Minv %*% V0 %*% t(Minv)
  V2      <- (fpr * (1 - fpr) / m) *
    (Gamma %*% (V0 + tcrossprod(b_raw)) %*% t(Gamma))
  V_raw   <- V1 + V2


  if (intercept) {
    perm   <- c(gen_idx, setdiff(seq_len(d), gen_idx))
    b      <- as.numeric(b_raw[perm])
    V      <- V_raw[perm, perm]
    names(b) <- colnames(Xhat)[perm]
  } else {
    b      <- as.numeric(b_raw)
    V      <- V_raw
    names(b) <- colnames(Xhat)
  }

  out <- list(coef = b, vcov = V)
  class(out) <- c("mlbc_fit", "mlbc_bcm")
  out
}
#' Additive bias‐corrected OLS estimator
#'
#' Computes the additive bias correction (BCA) for an OLS
#' regression when the primary regressor is measured by an ML/AI method.
#'
#' @param Y Numeric vector of responses.
#' @param Xhat Numeric matrix of regressors excluding the intercept.
#'             The first column **must** be the ML-generated variable to correct.
#' @param fpr Numeric.  Estimated false-positive rate of the generated regressor.
#' @param m Integer.  Size of the validation (labeled) sample used to estimate `fpr`.
#' @param intercept Logical; if `TRUE`, an intercept column of 1’s is prepended.
#'
#' @return An object of class `mlbc_fit` and subclass `mlbc_bca`, a list with elements
#'   * `coef`: Numeric vector of bias-corrected coefficients (intercept first, if requested).
#'   * `vcov`: Variance–covariance matrix of those coefficients.
#'
#' @examples
#' # unlabeled:
#' Nunl      <- 1e4
#' Xtrue_unl <- rbinom(Nunl, 1, 0.2)
#' Xhat_unl  <- ifelse(runif(Nunl) < 0.1, 1, Xtrue_unl)
#' Y_unl     <- 5 + 2 * Xtrue_unl + rnorm(Nunl)
#'
#' # small labeled sample to get fpr:
#' nval      <- 100
#' Xtrue_val <- rbinom(nval, 1, 0.2)
#' Xhat_val  <- ifelse(runif(nval) < 0.1, 1, Xtrue_val)
#' Y_val     <- 5 + 2 * Xtrue_val + rnorm(nval)
#' fpr_hat   <- mean(Xhat_val == 1 & Xtrue_val == 0)
#'
#' # now do additive correction, with intercept
#' fit_bca <- ols_bca(
#'   Y        = Y_unl,
#'   Xhat     = matrix(Xhat_unl, ncol = 1, dimnames = list(NULL, "Xhat")),
#'   fpr      = fpr_hat,
#'   m        = nval,
#'   intercept= TRUE
#' )
#' print(fit_bca)
#'
#' @references Battaglia, Christensen, Hansen, and Sacher (2025). "Inference for Regression with Variables Generated by AI or Machine Learning".
#' @seealso \link{ols_bcm} for the multiplicative correction.
#' @export

ols_bca <- function(Y, Xhat, fpr, m, intercept = TRUE) {

  if (intercept) {
    Xhat    <- cbind(Intercept = 1, Xhat)
    gen_idx <- 2L
  } else {
    gen_idx <- 1L
  }


  orig <- ols(Y, Xhat, se = TRUE)
  b0   <- as.numeric(orig$coef)
  V0   <- orig$vcov
  sXX  <- orig$sXX
  d    <- length(b0)


  A       <- matrix(0, d, d)
  A[gen_idx, gen_idx] <- 1
  Gamma   <- solve(sXX, A)


  b_raw   <- b0 + fpr * (Gamma %*% b0)


  I       <- diag(d)
  Minv    <- solve(I + fpr * Gamma)
  V1      <- Minv %*% V0 %*% t(Minv)
  V2      <- (fpr * (1 - fpr) / m) *
    (Gamma %*% (V0 + tcrossprod(b0)) %*% t(Gamma))
  V_raw   <- V1 + V2


  if (intercept) {
    perm <- c(gen_idx, setdiff(seq_len(d), gen_idx))
    b    <- as.numeric(b_raw[perm])
    V    <- V_raw[perm, perm]
    names(b) <- colnames(Xhat)[perm]
  } else {
    b    <- as.numeric(b_raw)
    V    <- V_raw
    names(b) <- colnames(Xhat)
  }

  out <- list(coef = b, vcov = V)
  class(out) <- c("mlbc_fit", "mlbc_bca")
  out
}
