% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions.R
\name{prop_stronger}
\alias{prop_stronger}
\title{Estimate proportion of true effect sizes above or below a threshold}
\usage{
prop_stronger(q, M, t2, se.M = NA, se.t2 = NA, CI.level = 0.95,
  tail = NA, dat = NULL, R = 2000, bootstrap = "ifneeded",
  yi.name = "yi", vi.name = "vi")
}
\arguments{
\item{q}{True effect size that is the threshold for "scientific importance"}

\item{M}{Pooled point estimate from meta-analysis}

\item{t2}{Estimated heterogeneity (tau^2) from meta-analysis}

\item{se.M}{Estimated standard error of pooled point estimate from meta-analysis}

\item{se.t2}{Estimated standard error of tau^2 from meta-analysis}

\item{CI.level}{Confidence level as a proportion}

\item{tail}{\code{above} for the proportion of effects above \code{q}; \code{below} for
the proportion of effects below \code{q}.}

\item{dat}{Dataset of point estimates (with names equal to the passed \code{yi.name}) and their variances
(with names equal to the passed \code{vi.name}). Only required when bootstrapping.}

\item{R}{Number of bootstrap iterates. Only required when bootstrapping.}

\item{bootstrap}{If equal to \code{ifneeded}, bootstraps if estimated proportion is less than 0.15 or more than
0.85. If equal to \code{never}, instead does not return inference in the above edge cases. Only required when bootstrapping.}

\item{yi.name}{Name of the variable in \code{dat} containing the study-level point estimates. Only required when bootstrapping.}

\item{vi.name}{Name of the variable in \code{dat} containing the study-level variances. Only required when bootstrapping.}
}
\value{
Returns a dataframe containing the point estimate for the proportion, its estimated standard error, and confidence
interval limits.
}
\description{
Estimates the proportion of true (i.e., population parameter) effect sizes in a meta-analysis
that are above or below a specified threshold of scientific importance.
}
\details{
When the estimated proportion is less than 0.15 or more than 0.85, it is best to bootstrap the confidence interval
using the bias-corrected and accelerated (BCa) method (Mathur & VanderWeele, 2018); this is the default behavior of \code{prop_stronger}.
Sometimes BCa confidence interval estimation fails, in which case \code{prop_stronger} instead uses the percentile method,
issuing a warning if this is the case. We use a modified "safe" version of the \code{boot} package code for bootstrapping
such that if any bootstrap iterates fail (usually because of model estimation problems), the error message is printed but the
bootstrap iterate is simply discarded so that confidence interval estimation can proceed.
}
\examples{
# calculate effect sizes for example dataset
d = metafor::escalc(measure="RR", ai=tpos, bi=tneg,
                   ci=cpos, di=cneg, data=metafor::dat.bcg)

# fit random-effects model
# note that metafor package returns on the log scale
m = metafor::rma.uni(yi= d$yi, vi=d$vi, knha=TRUE,
measure="RR", method="REML" )

# pooled point estimate (RR scale)
exp(m$b)

# estimate the proportion of effects stronger than RR = 0.80
# no bootstrapping will be needed
prop_stronger( q = log(0.8),
               M = as.numeric(m$b),
               t2 = m$tau2,
               se.M = as.numeric(m$vb),
               se.t2 = m$se.tau2,
               CI.level = 0.95,
              tail = "below",
              bootstrap = "ifneeded")

\dontrun{
# now try a more extreme threshold, q, such that the function will use bootstrapping
# now we will need to pass the final 4 arguments as well
prop_stronger( q = log(0.9),
               M = as.numeric(m$b),
              t2 = m$tau2,
              se.M = as.numeric(m$vb),
              se.t2 = m$se.tau2,
              CI.level = 0.95,
              tail = "below",

              # below arguments control bootstrapping
              # only 100 iterates for demo purposes (should be higher in practice)
              dat = d,
              R = 100,
              bootstrap = "ifneeded",
              yi.name = "yi",
              vi.name = "vi" )
}
}
\references{
Mathur MB & VanderWeele TJ. New metrics for meta-analyses of heterogeneous effects. Statistics in Medicine (2018).
}
