% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MiRKATS.R
\name{MiRKATS}
\alias{MiRKATS}
\title{Microiome Regression-based Kernel Association Test for Survival}
\usage{
MiRKATS(
  obstime,
  delta,
  X = NULL,
  Ks,
  beta = NULL,
  perm = FALSE,
  omnibus = "permutation",
  nperm = 999,
  returnKRV = FALSE,
  returnR2 = FALSE
)
}
\arguments{
\item{obstime}{A numeric vector of follow-up (survival/censoring) times.}

\item{delta}{Event indicator: a vector of 0/1, where 1 indicates that the event was observed for a subject (so "obstime" is 
survival time), and 0 indicates that the subject was censored.}

\item{X}{A vector or matrix of numeric covariates, if applicable (default = NULL).}

\item{Ks}{A list of or a single numeric n by n kernel matrices or matrix (where n is the sample size).}

\item{beta}{A vector of coefficients associated with covariates. If beta is NULL and covariates are present, coxph is used to 
calculate coefficients (default = NULL).}

\item{perm}{Logical, indicating whether permutation should be used instead of analytic p-value calculation (default=FALSE). 
Not recommended for sample sizes of 100 or more.}

\item{omnibus}{A string equal to either "Cauchy" or "permutation" (or nonambiguous abbreviations thereof), specifying whether 
to use the Cauchy combination test or residual permutation to generate the omnibus p-value.}

\item{nperm}{Integer, number of permutations used to calculate p-value if perm==TRUE (default=1000) and to calculate omnibus p-value if omnibus=="permutation".}

\item{returnKRV}{A logical indicating whether to return the KRV statistic. Defaults to FALSE.}

\item{returnR2}{A logical indicating whether to return the R-squared coefficient. Defaults to FALSE.}
}
\value{
Return value depends on the number of kernel matrices inputted. If more than one kernel matrix is given, MiRKATS returns two
items; a vector of the labeled individual p-values for each kernel matrix, as well as an omnibus p-value from the Optimal-MiRKATS
omnibus test. If only one kernel matrix is given, then only its p-value will be given, as no omnibus test will be needed.
   \item{p_values}{individual p-values for each inputted kernel matrix}
   \item{omnibus_p}{overall omnibus p-value}
    \item{KRV}{A vector of kernel RV statistics (a measure of effect size), one for each candidate kernel matrix. Only returned if returnKRV = TRUE}
    \item{R2}{A vector of R-squared statistics, one for each candidate kernel matrix. Only returned if returnR2 = TRUE}
}
\description{
Community level test for association between microbiome composition and survival outcomes (right-censored time-to-event data) 
using kernel matrices to compare similarity between microbiome profiles with similarity in survival times.
}
\details{
obstime, delta, and X should all have n rows, and the kernel or distance matrix should be a single n by n matrix. 
If a distance matrix is entered (distance=TRUE), a kernel matrix will be constructed from the distance matrix.

Update in v1.1.0: MiRKATS also utilizes the OMiRKATS omnibus test if more than one kernel matrix is provided by the user. 
The OMiRKATS omnibus test calculates an overall p-value for the test via permutation. 

Missing data is not permitted. Please remove individuals with missing data on y, X or in the kernel or distance matrix prior 
to using the function.

The Efron approximation is used for tied survival times.
}
\examples{

###################################
# Generate data
library(GUniFrac)

# Throat microbiome data
data(throat.tree)
data(throat.otu.tab)

unifracs = GUniFrac(throat.otu.tab, throat.tree, alpha = c(1))$unifracs
Ds = list(w = unifracs[,,"d_1"], uw = unifracs[,,"d_UW"], 
         BC= as.matrix(vegdist(throat.otu.tab, method="bray"))) 
Ks = lapply(Ds, FUN = function(d) D2K(d))

# Covariates and outcomes
covar <- matrix(rnorm(120), nrow=60)
S <- rexp(60, 3)   # survival time 
C <- rexp(60, 1)   # censoring time 
D <- (S<=C)        # event indicator
U <- pmin(S, C)    # observed follow-up time

MiRKATS(obstime = U, delta = D, X = covar, Ks = Ks, beta = NULL)


}
\references{
Plantinga, A., Zhan, X., Zhao, N., Chen, J., Jenq, R., and Wu, M.C. MiRKAT-S: a distance-based test of association between
microbiome composition and survival times. Microbiome, 2017:5-17. doi: 10.1186/s40168-017-0239-9

Zhao, N., Chen, J.,Carroll, I. M., Ringel-Kulka, T., Epstein, M.P., Zhou, H., Zhou, J. J., Ringel, Y., Li, H. and Wu, M.C. (2015)).
Microbiome Regression-based Kernel Association Test (MiRKAT). American Journal of Human Genetics, 96(5):797-807

Chen, J., Chen, W., Zhao, N., Wu, M~C.and Schaid, D~J. (2016) Small Sample Kernel Association Tests for Human Genetic and
Microbiome Association Studies. 40:5-19. doi: 10.1002/gepi.21934

Efron, B. (1977) "The efficiency of Cox's likelihood function for censored data." Journal of the American statistical 
Association 72(359):557-565.

Davies R.B. (1980) Algorithm AS 155: The Distribution of a Linear Combination of chi-2 Random Variables, Journal of the Royal 
Statistical Society Series C, 29:323-333
}
\author{
Nehemiah Wilson, Anna Plantinga
}
