% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ms_outlook_email.R
\docType{class}
\name{ms_outlook_email}
\alias{ms_outlook_email}
\title{Outlook mail message}
\format{
An R6 object of class \code{ms_outlook_email}, inheriting from \code{ms_outlook_object}, which in turn inherits from \code{ms_object}.
}
\description{
Class representing an Outlook mail message. The one class represents both sent and unsent (draft) emails.
}
\section{Fields}{

\itemize{
\item \code{token}: The token used to authenticate with the Graph host.
\item \code{tenant}: The Azure Active Directory tenant for the email account.
\item \code{type}: always "email" for an Outlook mail message.
\item \code{properties}: The item properties (metadata).
}
}

\section{Methods}{

\itemize{
\item \code{new(...)}: Initialize a new object. Do not call this directly; see 'Initialization' below.
\item \code{delete(confirm=TRUE)}: Delete this email. By default, ask for confirmation first.
\item \code{update(...)}: Update the email's properties (metadata) in Microsoft Graph.
\item \code{do_operation(...)}: Carry out an arbitrary operation on the email.
\item \code{sync_fields()}: Synchronise the R object with the email metadata in Microsoft Graph.
\item \code{set_body(body=NULL, content_type=NULL)}: Update the email body. See 'Editing an email' below.
\item \code{set_subject(subject)}: Update the email subject line.
\item \code{set_recipients(to=NULL, cc=NULL, bcc=NULL)}: Set the recipients for the email, overwriting any existing recipients.
\item \code{add_recipients(to=NULL, cc=NULL, bcc=NULL)}: Adds recipients for the email, leaving existing ones unchanged.
\item \code{set_reply_to(reply_to=NULL)}: Sets the reply-to field for the email.
\item \code{add_attachment(object, ...)}: Adds an attachment to the email. See 'Attachments' below.
\item \code{add_image(object)}: Adds an inline image to the email.
\item \code{get_attachment(attachment_name=NULL, attachment_id=NULL)}: Gets an attachment, either by name or ID. Note that attachments don't need to have unique names; if multiple attachments share the same name, the method throws an error.
\item \code{list_attachments(filter=NULL, n=Inf)}: Lists the current attachments for the email.
\item \code{remove_attachment(attachment_name=NULL, attachment_id=NULL, confirm=TRUE)}: Removes an attachment from the email. By default, ask for confirmation first.
\item \code{download_attachment(attachment_name=NULL, attachment_id=NULL, ...)}: Downloads an attachment. This is only supported for file attachments (not URLs).
\item \code{send()}: Sends an email.  See 'Sending, replying and forwarding'.
\item \code{create_reply(comment="", send_now=FALSE)}: Replies to the sender of an email.
\item \code{create_reply_all(comment="", send_now=FALSE)}: Replies to the sender and all recipients of an email.
\item \code{create_forward(comment="", to=NULL, cc=NULL, bcc=NULL, send_now=FALSE)}: Forwards the email to other recipients.
\item \verb{copy(dest),move(dest)}: Copies or moves the email to the destination folder.
\item \code{get_message_headers}: Retrieves the Internet message headers for an email, as a named character vector.
}
}

\section{Initialization}{

Creating new objects of this class should be done via the the appropriate methods for the \code{\link{ms_outlook}} or \code{\link{ms_outlook_folder}} classes. Calling the \code{new()} method for this class only constructs the R object; it does not call the Microsoft Graph API to retrieve or create the actual folder.
}

\section{Editing an email}{

This class exposes several methods for updating the properties of an email. They should work both for unsent (draft) emails and sent ones, although they make most sense in the context of editing drafts.

\code{set_body(body, content_type)} updates the message body of the email. This has 2 arguments: \code{body} which is the body text itself, and \code{content_type} which should be either "text" or "html". For both arguments, you can set the value to NULL to leave the current property unchanged. The \code{body} argument can also be a message object from either the blastula or emayili packages, much like when creating a new email.

\code{set_subject(subject)} sets the subject line of the email.

\code{set_recipients(to, cc, bcc)} sets or clears the recipients of the email. The \code{to}, \code{cc} and \code{bcc} arguments should be lists of either email addresses as character strings, or objects of class \code{az_user} representing a user account in Azure Active Directory. The default behaviour is to overwrite any existing recipients; to avoid this, pass \code{NA} as the value for the relevant argument. Alternatively, you can use the \code{add_recipients()} method.

\code{add_recipients(to, cc, bcc)} is like \code{set_recipients()} but leaves existing recipients unchanged.

\code{set_reply_to(reply_to)} sets or clears the reply-to address for the email. Leave the \code{reply_to} argument at its default NULL value to clear this property.
}

\section{Attachments}{

This class exposes the following methods for working with attachments.

\code{add_attachment(object, type, expiry, password, scope)} adds an attachment to the email. The arguments are as follows:
\itemize{
\item \code{object}: A character string containing a filename or URL, or an object of class \code{\link{ms_drive_item}} representing a file in OneDrive or SharePoint. In the latter case, a shareable link to the drive item will be attached to the email, with the link details given by the other arguments.
\item \verb{type, expiry, password, scope}: The specifics for the shareable link to attach to the email, if \code{object} is a drive item. See the \code{create_share_link()} method of the \code{\link{ms_drive_item}} class; the default is to create a read-only link valid for 7 days.
}

\code{add_image(object)} adds an image as an \emph{inline} attachment, ie, as part of the message body. The \code{object} argument should be a filename, and the message content type will be set to "html" if it is not already. Currently Microsoft365R does minimal formatting of the image; consider using a package like blastula for more control over the layout of inline images.

\code{list_attachments()} lists the attachments for the email, including inline images. This will be a list of objects of class \code{\link{ms_outlook_attachment}} containing the metadata for the attachments.

\code{get_attachment(attachment_name, attachment_id)}: Retrieves the metadata for an attachment, as an object of class \code{ms_outlook_attachment}. Note that multiple attachments can share the same name; in this case, you must specify the ID of the attachment.

\code{download_attachment(attachment_name, attachment_id, dest, overwrite)}: Downloads a file attachment. The default destination filename is the name of the attachment.

\code{remove_attachment(attachment_name, attachment_id)} removes (deletes) an attachment.
}

\section{Sending, replying and forwarding}{

Microsoft365R's default behaviour when creating, replying or forwarding emails is to create a draft message object, to allow for further edits. The draft is saved in the Drafts folder by default, and can be sent later by calling its \code{send()} method.

The methods for replying and forwarding are \code{create_reply()}, \code{create_reply_all()} and \code{create_forward()}. The first argument to these is the reply text, which will appear above the current message text in the body of the reply. For \code{create_forward()}, the other arguments are \code{to}, \code{cc} and \code{bcc} to specify the recipients of the forwarded email.
}

\section{Other methods}{

The \code{copy()} and \code{move()} methods copy and move an email to a different folder. The destination should be an object of class \code{ms_outlook_folder}.

The \code{get_message_headers()} method retrieves the Internet message headers for the email, as a named character vector.
}

\section{List methods}{

All \verb{list_*} methods have \code{filter} and \code{n} arguments to limit the number of results. The former should be an \href{https://docs.microsoft.com/en-us/graph/query-parameters#filter-parameter}{OData expression} as a string to filter the result set on. The latter should be a number setting the maximum number of (filtered) results to return. The default values are \code{filter=NULL} and \code{n=Inf}. If \code{n=NULL}, the \code{ms_graph_pager} iterator object is returned instead to allow manual iteration over the results.

Support in the underlying Graph API for OData queries is patchy. Not all endpoints that return lists of objects support filtering, and if they do, they may not allow all of the defined operators. If your filtering expression results in an error, you can carry out the operation without filtering and then filter the results on the client side.
}

\examples{
\dontrun{

outl <- get_personal_outlook()

##
## creating a new email
##

# a blank text email
em <- outl$create_email()

# add a body
em$set_body("Hello from R", content_type="html")

# add recipients
em$set_recipients(to="user@example.com")

# add subject line
em$set_subject("example email")

# add an attachment
em$add_attachment("mydocument.docx")

# add a shareable link to a file in OneDrive
mysheet <- get_personal_onedrive()$get_item("documents/mysheet.xlsx")
em$add_attachment(mysheet)

# add an inline image
em$add_image("myggplot.jpg")

# oops, wrong recipient, it should be someone else
# this removes user@example.com from the to: field
em$set_recipients(to="user2@example.com")

# and we should also cc a third user
em$add_recipients(cc="user3@example.com")

# send it
em$send()

# you can also compose an email as a pipeline
outl$create_email()$
    set_body("Hello from R")$
    set_recipients(to="user2@example.com", cc="user3@example.com")$
    set_subject("example email")$
    add_attachment("mydocument.docx")$
    send()

# using blastula to create a HTML email with Markdown
bl_msg <- blastula::compose_email(md(
"
## Hello!

This is an email message that was generated by the blastula package.

We can use **Markdown** formatting with the `md()` function.

Cheers,

The blastula team
"),
    footer=md("Sent via Microsoft365R"))
outl$create_email()
    set_body(bl_msg)$
    set_subject("example blastula email")


##
## replying and forwarding
##

# get the most recent email in the Inbox
em <- outl$list_emails()[[1]]

# reply to the message sender, cc'ing Carol
em$create_reply("I agree")$
    add_recipients(cc="carol@example.com")$
    send()

# reply to everyone, setting the reply-to address
em$create_reply_all("Please do not reply")$
    set_reply_to("do_not_reply@example.com")$
    send()

# forward to Dave
em$create_forward("FYI", to="dave@example.com")$
    send()


##
## attachments
##

# download an attachment by name (assumes there is only one 'myfile.docx')
em$download_attachment("myfile.docx")

# a more reliable way: get the list of attachments, and download via the object
atts <- em$list_attachments()
atts[[1]]$download()

# add and remove an attachment
em$add_attachment("anotherfile.pptx")
em$remove_attachment("anotherfile.pptx")


##
## moving and copying
##

# copy an email to a nested folder: /folder1/folder2
dest <- outl$get_folder("folder1")$get_folder("folder2")
em$copy(dest)

# move it instead
em$move(dest)

}
}
\seealso{
\code{\link{ms_outlook}}, \code{\link{ms_outlook_folder}}, \code{\link{ms_outlook_attachment}}

\href{https://docs.microsoft.com/en-us/graph/overview}{Microsoft Graph overview},
\href{https://docs.microsoft.com/en-us/graph/api/resources/mail-api-overview?view=graph-rest-1.0}{Outlook API reference}
}
