% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MR.mean.r
\name{MR.mean}
\alias{MR.mean}
\title{Multiply Robust Estimation of the Marginal Mean}
\usage{
MR.mean(reg.model = NULL, mis.model = NULL, data, bootstrap = FALSE,
  bootstrap.size = 500, alpha = 0.05)
}
\arguments{
\item{reg.model}{A list of outcome regression models defined by \code{\link{def.glm}}.}

\item{mis.model}{A list of missingness probability models defined by \code{\link{def.glm}}. The dependent variable is always specified as \code{R}.}

\item{data}{A data frame with missing data encoded as \code{NA}.}

\item{bootstrap}{Logical. Should a bootstrap method be applied to calculate the standard error of the estimator and construct a Wald confidence interval for the marginal mean. Default is \code{FALSE}.}

\item{bootstrap.size}{A numeric value. Number of bootstrap resamples generated if \code{bootstrap} is \code{TRUE}. Default is 500.}

\item{alpha}{Significance level used to construct the 100(1 - \code{alpha})\% Wald confidence interval.}
}
\value{
\item{\code{Estimate}}{The estimated value of the marginal mean. If \code{bootstrap} is \code{TRUE}, bootstrap standard error \code{SE} of the estimate and a Wald confidence interval are provided.}
}
\description{
\code{MR.mean()} is used to estimate the marginal mean of a variable which is subject to missingness. Multiple missingness probability models and outcome regression models can be included.
}
\examples{
# Simulated data set
set.seed(123)
n <- 400
gamma0 <- c(1, 2, 3)
alpha0 <- c(-0.8, -0.5, 0.3)
X <- runif(n, min = -2.5, max = 2.5)
X.2 <- X ^ 2
exp.X <- exp(X)
p.miss <- 1 / (1 + exp(alpha0[1] + alpha0[2] * X + alpha0[3] * X.2))
R <- rbinom(n, size = 1, prob = 1 - p.miss)
a.x <- gamma0[1] + gamma0[2] * X + gamma0[3] * exp.X
Y <- rnorm(n, a.x, sd = sqrt(4 * X.2 + 2))
dat <- data.frame(X, X.2, exp.X, Y)
dat[R == 0, 4] <- NA

# Define the outcome regression models and missingness probability models
reg1 <- def.glm(formula = Y ~ X + exp.X, family = gaussian)
reg2 <- def.glm(formula = Y ~ X + X.2, family = gaussian)
mis1 <- def.glm(formula = R ~ X + X.2, family = binomial(link = logit))
mis2 <- def.glm(formula = R ~ X + exp.X, family = binomial(link = cloglog))
est <- MR.mean(reg.model = list(reg1, reg2), mis.model = list(mis1, mis2), data = dat)
}
\references{
Han, P. and Wang, L. (2013). Estimation with missing data: beyond double robustness. \emph{Biometrika}, \strong{100}(2), 417--430.
}
