% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/function.R
\name{test.multiple.robust}
\alias{test.multiple.robust}
\title{robust test of multiple change point for matrix-valued online time series}
\usage{
test.multiple.robust(
  Y,
  k = 1,
  m = 20,
  epsilon1 = 0.25,
  epsilon2 = 0.05,
  r = 8,
  kmax = 4,
  type = "proj",
  method = "ps",
  eta = 0.25,
  cv = 2.386,
  S = 100,
  pr = 0.75
)
}
\arguments{
\item{Y}{data, a \eqn{T\times p1\times p2} array.}

\item{k}{a non-negative integer indicating the initial number of factors.}

\item{m}{a positive integer (\eqn{>1}) indicating the bandwidth of the rolling
windom.}

\item{epsilon1}{the rescaling parameter taking value in \eqn{(0,1)}, for the
test of new factors or the change of loading space; see He et al. (2021).}

\item{epsilon2}{the rescaling parameter taking value in \eqn{(0,1)}, for the
test of vanishing factors; see He et al. (2021).}

\item{r}{a positive integer indicating the order of the transformation
function \eqn{g(x)=|x|^r}; see also  \code{\link{gen.psi.tau.proj}}.}

\item{kmax}{a positive number determining the column number of the projection
matrix, should be larger than 0 but smaller than \eqn{p_2}, required when
\emph{type} not being "flat".}

\item{type}{indicates how to calculate the sample covariance. "flat" for the
flat version, while others for the projected version. See more details in He
et al. (2021).}

\item{method}{indicating the test statistic, ``ps'' for the partial-sum
method; others for the worst-case method.}

\item{eta}{a number between \eqn{[0,1)}, indicating the parameter \eqn{\eta}
used in the partial-sum statistic.}

\item{cv}{critical value; see also \code{\link{test.once.psi}}.}

\item{S}{an integer indicating the number of replications.}

\item{pr}{an number in \eqn{(0,1])}. The procedure reports a change point only
when the proportion of positive votes is over \emph{pr} in the
\emph{S} replications.}
}
\value{
a matrix with two columns. The first column reports the locations
of change points. The second column reports the number of row factors
after each change point.
}
\description{
This function tests multiple change points for matrix-valued online time
series, under a two-way factor structure. A change point will be reported
only when it's the majority vote in multiple replications. The function \code{\link{KSTP}}
is used to determine the initial number of factors in each regime. This function only outputs
the change points for row factors. For column factors, transpose the data.
}
\details{
#' See empirical study in He et al. (2021).
}
\examples{
\donttest{
k1=3
k2=3
Sample_T=100
p1=40
p2=40
kmax=8
r=8
m=p2

# generate data
Y1=gen.data(Sample_T,p1,p2,k1,k2,tau=0.5,change=1,pp=0.5)
Y2=gen.data(Sample_T,p1,p2,k1,k2,tau=0.5,change=0)
Y=array(rbind(matrix(Y1,Sample_T,p1*p2),matrix(Y2,Sample_T,p1*p2)),c(Sample_T*2,p1,p2))

# calculate cv for "ps" with eta=0.45 and "wc"
cv1=getcv(0.05,method="ps",eta=0.45)
cv2=getcv(0.05,method="wc")

# test with Y
test.multiple.robust(Y,k1,m,epsilon1=0.25,epsilon2=0.05,r,type="proj",kmax,method="ps")

test.multiple.robust(Y,k1,m,epsilon1=0.25,epsilon2=0.05,r,type="proj",kmax,method="wc",cv=cv2)

test.multiple.robust(Y,k1,m,epsilon1=0.25,epsilon2=0.05,r,type="flat",method="wc",cv=cv2)

test.multiple.robust(Y,k1,m,epsilon1=0.25,epsilon2=0.05,r,type="flat",method="ps",eta=0.45,cv=cv1)
}
}
\references{
He Y, Kong X, Trapani L, & Yu L(2021). Online change-point detection for
matrix-valued time series with latent two-way factor structure. \emph{arXiv
preprint}, arXiv:2112.13479.
}
\author{
Yong He, Xinbing Kong, Lorenzo Trapani, Long Yu
}
