#' Plot and match singular enrichment results
#'
#' @description This function can be used to plot results of singular enrichment analysis for one set of protein.
#' It can also be used to integrate and match the results of two separate singular enrichment analysis and
#' plot the common PTMs. For more details please see examples.
#'
#' @param x A data frame that contains singular enrichment results generated by \code{\link{runEnrichment}}
#' @param y Default value is NULL. If provided by a singular enrichment results, the matching results
#' of x and y are plotted.
#' @param sig.level The significance level to select post-translational modification (based on their
#' corrected p-value). Note that `sig.level` applies to both x and y simultaneously.
#' @param number.rep Only plot PTM terms that occurred more than a specific number of times in UniProt database.
#' This number is set by number.rep parameter. The default value is NULL.
#'
#' @return Plot.
#'
#' @import ggplot2
#' @import dplyr
#'
#' @export
#'
#' @examples
#' ## Enrichment analysis for the first protein list
#' enrich1 <- runEnrichment(protein = exmplData1$pl1, os.name = 'Homo sapiens (Human)')
#' ## Plot results for first protein list
#' plotEnrichment(x = enrich1)
#'
#' ## Enrichment analysis for the second protein list
#' enrich2 <- runEnrichment(protein = exmplData1$pl2, os.name = 'Homo sapiens (Human)')
#' ## Plot results for second protein list
#' plotEnrichment(x = enrich2)
#'
#' ## Integrate and match the results of two separate singular enrichment analysis
#' plotEnrichment(x = enrich1, y = enrich2)
#' plotEnrichment(x = enrich1, y = enrich2, number.rep = 100)
plotEnrichment = function(x, y = NULL, sig.level = 0.05, number.rep = NULL){

  # Plot one enriched list
  if( is.null(y) ){

    # Undo scientific number presentation in x
    # formatC from base package

    x$`pvalue`           <- as.numeric( formatC(x$`pvalue`,           digits = 0, format = 'g') )
    x$`corrected pvalue` <- as.numeric( formatC(x$`corrected pvalue`, digits = 0, format = 'g') )


    # Filter based on corrected pvalue
    # Add a new column for relative frequency and log of corrected pvalues.
    # Select necessary columns for output

    data.for.plot <- x %>%
                     filter(`corrected pvalue` < sig.level) %>%
                     mutate(relFreq = round(`FreqinList`/Sample, digits = 3),
                            logCorrectPvalue = round(-log(`corrected pvalue`+1e-22), digits = 3) )



    if( !is.null(number.rep) ){
      data.for.plot <- data.for.plot %>% filter(`FreqinUniprot` >= number.rep)
    }

    data.for.plot <- data.for.plot %>% select(PTM, logCorrectPvalue, relFreq, `FreqinList`, Sample, Population)


    #  Change the order of PTM levels according to relFreq
    data.for.plot$PTM <- fct_reorder(data.for.plot$PTM, data.for.plot$relFreq)


    g1 <- ggplot(data = data.for.plot, aes(x = PTM, y = relFreq, fill = logCorrectPvalue, size = `FreqinList`)) +
          geom_segment( aes(x = PTM, y = 0, xend = PTM, yend = relFreq), size = 1, col = 'grey40' ) +
          geom_point(shape = 21) +
          theme(axis.text.x  = element_text(size = 12, face = 'bold'),
                axis.text.y  = element_text(size = 12, face = 'bold'),
                legend.title = element_text(color = 'blue', size = 10),
                legend.text  = element_text(size = 12),
                axis.title   = element_text(size = 16)) +
                scale_color_gradient(name = 'Adjusted p-value (-log)', high = 'mediumslateblue', low = 'maroon3')+
                xlab('PTM keywords') +
                ylab('Relative frequency') +
                coord_flip()


    plot(g1)

  }

  # Plot two enriched lists
  if( !is.null(y) ){

    # Undo scientific number presentation in x and y
    # formatC from base package
    x$`pvalue`           <- as.numeric( formatC(x$`pvalue`,           digits = 0, format = 'g') )
    x$`corrected pvalue` <- as.numeric( formatC(x$`corrected pvalue`, digits = 0, format = 'g') )
    y$`pvalue`           <- as.numeric( formatC(y$`pvalue`,           digits = 0, format = 'g') )
    y$`corrected pvalue` <- as.numeric( formatC(y$`corrected pvalue`, digits = 0, format = 'g') )


    # Filter x and y based on corrected pvalue
    x <- x %>% filter(`corrected pvalue` < sig.level)
    y <- y %>% filter(`corrected pvalue` < sig.level)

    # Calculate relative frequency
    x <- x %>% mutate(relFreq = round(`FreqinList`/Sample, digits = 4))
    y <- y %>% mutate(relFreq = round(`FreqinList`/Sample, digits = 4))

    # Calculate log10 corrected pvalue
    x <- x %>% mutate(logCorrectPvalue = round(-log(`corrected pvalue`+1e-22), digits = 4))
    y <- y %>% mutate(logCorrectPvalue = round(-log(`corrected pvalue`+1e-22), digits = 4))

    # Convert PTM from factor to character
    x <- x %>% mutate(PTM = as.character(PTM))
    y <- y %>% mutate(PTM = as.character(PTM))

    if( !is.null(number.rep) ){
      x <- x %>% filter(`FreqinUniprot` >= number.rep)
      y <- y %>% filter(`FreqinUniprot` >= number.rep)
    }

    # Select required columns
    x <- x %>% select(PTM, relFreq, logCorrectPvalue)
    y <- y %>% select(PTM, relFreq, logCorrectPvalue)


    # Unite dataframes
    temp <- data.frame( rbind(x,y), Group = c( rep('List1',nrow(x)), rep('List2',nrow(y)) ), stringsAsFactors = FALSE)

    # Check if there are common PTM
    ptm.names <- names( which( table(temp$PTM) == 2 ) )
    if( length(ptm.names) == 0 ){
      stop('No PTM intersection in x and y.')
    }


    # Filter on common PTMs
    data.for.plot <- temp[temp$PTM %in% ptm.names,]


    # Change the order of PTM levels according to relFreq
    data.for.plot     <- data.for.plot %>% arrange(relFreq)
    data.for.plot$PTM <- fct_reorder(data.for.plot$PTM, data.for.plot$relFreq)


    # Create ggplot
    p <- ggplot(data.for.plot, aes(x = relFreq, y = PTM)) +
         geom_line(aes(group = PTM), color = 'grey40', size = 1) +
         geom_point(aes(shape = Group, color = logCorrectPvalue, size = relFreq)) +
         xlab('Relative frequency') +
         ylab('PTM keywords') +
         theme(axis.text.x  = element_text(size = 12, face = 'bold'),
               axis.text.y  = element_text(size = 12, face = 'bold'),
               legend.title = element_text(color = 'blue', size = 10),
               legend.text  = element_text(size = 12),
               axis.title   = element_text(size = 16), legend.position = 'right')
    plot(p)


  }

}
