% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_pin.R
\name{pin_bayes}
\alias{pin_bayes}
\title{PIN estimation - Bayesian approach}
\usage{
pin_bayes(data, xtraclusters = 4, sweeps = 1000, burnin = 500,
                 prior.a = 1, prior.b = 2, verbose = TRUE)
}
\arguments{
\item{data}{A dataframe with 2 variables: the first
corresponds to buyer-initiated trades (buys), and the second corresponds
to seller-initiated trades (sells).}

\item{xtraclusters}{An integer used to divide trading days into
\code{#(2 + xtraclusters)} clusters, thereby resulting in
\code{#comb(1 + xtraclusters, 1)} initial parameter sets in line with
\insertCite{ErsanAlici2016;textual}{PINstimation}. The default value is \code{4}.}

\item{sweeps}{An integer referring to the number of iterations for the Gibbs
Sampler. This has to be large enough to ensure convergence of the Markov chain.
The default value is \code{1000}.}

\item{burnin}{An integer referring to the number of initial iterations for
which the parameter draws should be discarded. This is to ensure that we keep
the draws at the point where the MCMC has converged to the parameter space in
which the parameter estimate is likely to fall. This figure must always be
less than the sweeps. The default value is \code{500}.}

\item{prior.a}{An integer controlling the mean number of informed trades,
such as the prior of informed buys and sells is the Gamma density function
with \eqn{\mu} ~ \verb{Ga(prior.a,} \eqn{\eta}\verb{)}. The default value is \code{1}.
For more details, please refer to
\insertCite{griffin2021;textual}{PINstimation}.}

\item{prior.b}{An integer controlling the mean number of uninformed trades,
such as the prior of uninformed buys and sells is the Gamma density function
with \eb ~ \verb{Ga(prior.b,} \eqn{\eta}\verb{)}, and \es ~ \verb{Ga(prior.b,} \eqn{\eta}\verb{)}.
The default value is \code{2}. For more details, please refer to
\insertCite{griffin2021;textual}{PINstimation}.}

\item{verbose}{A binary variable that determines whether detailed
information about the steps of the estimation of the PIN model is displayed.
No output is produced when \code{verbose} is set to \code{FALSE}. The default
value is \code{TRUE}.}
}
\value{
Returns an object of class \code{estimate.pin}
}
\description{
Estimates the Probability of Informed Trading (\code{PIN}) using
Bayesian Gibbs sampling as in
\insertCite{griffin2021;textual}{PINstimation} and the initial sets
from the algorithm in \insertCite{ErsanAlici2016;textual}{PINstimation}.
}
\details{
The argument 'data' should be a numeric dataframe, and contain
at least two variables. Only the first two variables will be considered:
The first variable is assumed to correspond to the total number of
buyer-initiated trades, while the second variable is assumed to
correspond to the total number of seller-initiated trades. Each row or
observation correspond to a trading day. \code{NA} values will be ignored.

The function \code{pin_bayes()} implements the algorithm detailed in
\insertCite{ErsanAlici2016;textual}{PINstimation}.
The higher the number of the additional clusters (\code{xtraclusters}), the
better is the estimation. \insertCite{ErsanAlici2016;textual}{PINstimation},
however, have shown the benefit of increasing this number beyond 5 is
marginal, and statistically insignificant.\cr \cr
The function \code{initials_pin_ea()} provides the initial parameter sets
obtained through the implementation of the
\insertCite{ErsanAlici2016;textual}{PINstimation} algorithm.
For further information on the initial parameter set determination, see
\code{initials_pin_ea()}.
}
\examples{
# Use the function generatedata_mpin() to generate a dataset of
# 60 days according to the assumptions of the original PIN model.

sdata <- generatedata_mpin(layers = 1)
xdata <- sdata@data

# Estimate the PIN model using the Bayesian approach developed in
# Griffin et al. (2021), and initial parameter sets generated using the
# algorithm of Ersan and Alici (2016). The argument xtraclusters is
# set to 1. We also leave the arguments 'sweeps' and 'burnin' at their
# default values.

estimate <- pin_bayes(xdata, xtraclusters = 1, verbose = FALSE)


# Display the empirical PIN value at the data, and the PIN value
# estimated using the bayesian approach

setNames(c(sdata@emp.pin, estimate@pin), c("data", "estimate"))

# Display the empirial and the estimated parameters

show(unlist(sdata@empiricals))
show(estimate@parameters)

# Find the initial set that leads to the optimal estimate
optimal <- which.max(estimate@details$likelihood)

# Store the matrix of Monte Carlo simulation for the optimal
# estimate, and display its last five rows

mcmatrix <- estimate@details$markovmatrix[[optimal]]
show(tail(mcmatrix, 5))

# Display the summary of Geweke test for the Monte Carlo matrix above.
show(estimate@details$summary[[optimal]])

}
\references{
\insertAllCited
}
