% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pladmm.R
\name{pladmm}
\alias{pladmm}
\title{Fit a Plackett-Luce Model with Linear Predictor for Log-worth}
\usage{
pladmm(rankings, formula, data, rho = 1, n_iter = 500, rtol = 1e-04)
}
\arguments{
\item{rankings}{a \code{"\link{rankings}"} object, or an object that can be
coerced by \code{as.rankings}.}

\item{formula}{a \link{formula} specifying the linear model for log-worth.}

\item{data}{a data frame containing the variables in the model.}

\item{rho}{the penalty parameter in the penalized likelihood, see details.}

\item{n_iter}{the maximum number of iterations (also for inner loops).}

\item{rtol}{the convergence tolerance (also for inner loops)}
}
\description{
Fit a Plackett-Luce model where the log-worth is predicted by a linear
function of covariates. The rankings may be partial
(each ranking completely ranks a subset of the items), but ties are not
supported.
}
\details{
The log-worth is modelled as a linear function of item covariates:
\deqn{\log \alpha_i = \beta_0 + \beta_1 x_{i1} + \ldots + \beta_p x_{ip}}{
log \alpha_i = \beta_0 + \beta_1 x_{i1} + ... + \beta_p x_{ip}
}
where \eqn{\beta_0} is fixed by the constraint that
\eqn{\sum_i \alpha_i = 1}.

The parameters are estimated using an Alternating Directions Method of
Multipliers (ADMM) algorithm proposed by Yildiz (2020). ADMM alternates
between estimating the worths \eqn{\alpha_i} and the linear
coefficients \eqn{\beta_k}, encapsulating them in a quadratic penalty on the
likelihood:
\deqn{L(\boldsymbol{\beta}, \boldsymbol{\alpha}, \boldsymbol{u}) =
\mathcal{L}(\mathcal{D}|\boldsymbol{\alpha}) +
\frac{\rho}{2}||\boldsymbol{X}\boldsymbol{\beta} -
\log \boldsymbol{\alpha} + \boldsymbol{u}||^2_2 -
\frac{\rho}{2}||\boldsymbol{u}||^2_2}{
L(\beta, \alpha, u) = L(D|\alpha) + \rho/2 ||X\beta - log \alpha + u||^2_2 -
\rho/2 ||u||^2_2
}
where \eqn{\boldsymbol{u}}{u} is a dual variable that imposes the equality
constraints (so that \eqn{\log \boldsymbol{\alpha}}{log \alpha} converges to
\eqn{\boldsymbol{X}\boldsymbol{\beta}}{X \beta}).
}
\note{
This is a prototype function and the user interface is planned to
change in upcoming versions of PlackettLuce.
}
\examples{

if (require(prefmod)){
  data(salad)
  # data.frame of rankings for salad dressings A B C D
  # 1 = most tart, 4 = least tart
  salad[1:3,]

  # create data frame of corresponding features
  # (acetic and gluconic acid concentrations in salad dressings)
  features <- data.frame(salad = LETTERS[1:4],
                         acetic = c(0.5, 0.5, 1, 0),
                         gluconic = c(0, 10, 0, 10))

  # fit Plackett-Luce model based on covariates
  res_PLADMM <- pladmm(salad, ~ acetic + gluconic, data = features, rho = 8)
  ## coefficients
  coef(res_PLADMM)
  ## worth
  res_PLADMM$pi
  ## worth as predicted by linear function
  res_PLADMM$tilde_pi
  ## equivalent to
  drop(exp(res_PLADMM$x \%*\% coef(res_PLADMM)))

}

}
\references{
Yildiz, I., Dy, J., Erdogmus, D., Kalpathy-Cramer, J., Ostmo, S.,
Campbell, J. P., Chiang, M. F. and Ioannidis, S. (2020) Fast and Accurate
Ranking Regression In Proceedings of the Twenty Third International
Conference on Artificial Intelligence and Statistics, \bold{108}, 77–-88.
}
