% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotConfidence.R
\name{plotConfidence}
\alias{plotConfidence}
\title{Plot confidence intervals}
\usage{
plotConfidence(x, y.at, lower, upper, pch = 16, cex = 1, lwd = 1,
  col = 4, xlim, xlab, labels, title.labels, values, title.values,
  section.pos, section.sep, section.title = NULL, section.title.x,
  section.title.offset, order, leftmargin = 0.025, rightmargin = 0.025,
  stripes, factor.reference.pos, factor.reference.label = "Reference",
  factor.reference.pch = 16, refline = 1, title.line = TRUE, xratio,
  y.offset = 0, y.title.offset, digits = 2, format,
  extremearrows.length = 0.05, extremearrows.angle = 30, add = FALSE,
  layout = TRUE, xaxis = TRUE, ...)
}
\arguments{
\item{x}{Either a vector containing the point estimates or a list
whose first element contains the point estimates. Further list
elements can contain the confidence intervals and labels. In this
case the list needs to have names 'lower' and 'upper' to indicate
the values of the lower and the upper limits of the confidence
intervals, respectively, and may have an element 'labels' which is
a vector or matrix or list with labels.}

\item{y.at}{Optional vector of y-position for the confidence intervals and corresponding values and labels.}

\item{lower}{Lower confidence limits. Used if object \code{x} is a
vector and if \code{x} is a list \code{lower} overwrites element
\code{x$lower}.}

\item{upper}{Upper confidence limits. Used if object \code{x} is a
vector and if \code{x} is a list \code{upper} overwrites element
\code{x$upper}.}

\item{pch}{Symbol for points.}

\item{cex}{Defaults size of all figures and plotting symbol.
Single elements are controlled separately. See \code{...}.}

\item{lwd}{Default width of all lines Single elements are
controlled separately. See \code{...}.}

\item{col}{Default colour of confidence intervals.}

\item{xlim}{Plotting limits for the confidence intervals. See also
\code{xratio} on how to control the layout.}

\item{xlab}{Label for the x-axis.}

\item{labels}{Vector or matrix or list with \code{labels}. Used if
object \code{x} is a vector and if \code{x} is a list it
overwrites element \code{x$labels}. To avoid drawing of labels, set \code{labels=FALSE}.}

\item{title.labels}{Main title for the column which shows the \code{labels}. If \code{labels}
is a matrix or list \code{title.labels} should be a vector with as
many elements as labels has columns or elements.}

\item{values}{Either logical or vector, matrix or list with
values. If \code{values=TRUE} values are constructed according to
\code{format} from \code{lower} and \code{upper} overwrites
constructed values. If \code{values=FALSE} do not draw values.}

\item{title.values}{Main title for the column \code{values}. If \code{values}
is a matrix or list \code{title.labels} should be a vector with as
many elements as values has columns or elements.}

\item{section.pos}{Vector with y-axis posititions for section.titles.}

\item{section.sep}{Amount of space between paragraphs (applies only if \code{labels} is a named  list)}

\item{section.title}{Intermediate section headings.}

\item{section.title.x}{x-position for section.titles}

\item{section.title.offset}{Y-offset for section.titles}

\item{order}{Order of the three columns: labels, confidence limits,
values. See examples.}

\item{leftmargin}{Percentage of plotting region used for
leftmargin. Default is 0.025. See also Details.}

\item{rightmargin}{Percentage of plotting region used for
rightmargin. Default is 0.025. See also Details.}

\item{stripes}{Vector of up to three Logicals. If \code{TRUE} draw
stripes into the background. The first applies to the labels, the
second to the graphical presentation of the confidence intervals
and the third to the values. Thus, stripes}

\item{factor.reference.pos}{Position at which factors attain
reference values.}

\item{factor.reference.label}{Label to use at
\code{factor.reference.pos} instead of values.}

\item{factor.reference.pch}{Plotting symbol to use at
\code{factor.reference.pos}}

\item{refline}{Position of a vertical line to indicate the null
hypothesis. Default is 1 which would work for odds ratios and
hazard ratios.}

\item{title.line}{Position of a horizontal line to separate the title line from the plot}

\item{xratio}{One or two values between 0 and 1 which determine
how to split the plot window in horizontal x-direction. If there
are two columns (labels, CI) or (CI, values) only one value is used
and the default is 0.618 (goldener schnitt) which gives the
graphical presentation of the confidence intervals 38.2 % of the
graph. The remaining 61.8 % are used for the labels (or values).
If there are three columns (labels, CI, values), xratio has two
values which default to fractions of 0.7 according to the relative
widths of labels and values, thus by default only 0.3 are used for
the graphical presentation of the confidence intervals. The
remaining 30 % are used for the graphical presentation of the
confidence intervals. See examles.}

\item{y.offset}{Either a single value or a vector determining the
vertical offset of all rows. If it is a single value all rows are
shifted up (or down if negative) by this value. This can be used
to add a second set of confidence intervals to an existing graph
or to achieve a visual grouping of rows that belong
together. See examples.}

\item{y.title.offset}{Numeric value by which to vertically shift
the titles of the labels and values.}

\item{digits}{Number of digits, passed to \code{pubformat} and
\code{formatCI}.}

\item{format}{Format for constructing values of confidence
intervals. Defaults to '(u;l)' if there are negative lower or
upper values and to '(u-l)' otherwise.}

\item{extremearrows.length}{Length of the arrows in case of
confidence intervals that stretch beyond xlim.}

\item{extremearrows.angle}{Angle of the arrows in case of
confidence intervals that stretch beyond xlim.}

\item{add}{Logical. If \code{TRUE} do not draw labels or values
and add confidence intervals to existing plot.}

\item{layout}{Logical. If \code{FALSE} do not call layout. This is useful when
several plotConfidence results should be combined in one graph and hence layout is called
externally.}

\item{xaxis}{Logical. If \code{FALSE} do not draw x-axis.}

\item{...}{Used to control arguments of the following subroutines:
\code{plot}: Applies to plotting frame of the graphical
presentation of confidence intervals. Use arguments of
\code{plot}, e.g., \code{plot.main="Odds ratio"}.  \code{points},
\code{arrows}: Use arguments of \code{points} and \code{arrows},
respectively. E.g., \code{points.pch=8} and \code{arrows.lwd=2}.
\code{refline}: Use arguments of \code{segments}, e.g.,
\code{refline.lwd=2}. See \link{segments}.  \code{labels},
\code{values}, \code{title.labels}, \code{title.values}: Use
arguments of \code{text}, e.g., \code{labels.col="red"} or
\code{title.values.cex=1.8}.  \code{xaxis}: Use arguments of
\code{axis}, e.g., \code{xaxis.at=c(-0.3,0,0.3)} \code{xlab}: Use
arguments of \code{mtext}, e.g., \code{xlab.line=2}.
\code{stripes}: Use arguments of \code{stripes}. See examples.
See examples for usage.}
}
\value{
List of dimensions and coordinates
}
\description{
Function to plot confidence intervals with their values and additional labels.
One anticipated use of this function involves first the generation of a regression object,
then arrangement of a result table with "regressionTable", further arrangment of table with
with e.g. "fixRegressionTable" and various user defined changes - and then finally table
along with forest plot using the current function.
}
\details{
Function to plot means and other point estimates with confidence intervals,
their values and additional labels .
Horizonal margins as determined by par()$mar are ignored.
Instead layout is used to divide the plotting region horizontally
into two or three parts plus leftmargin and rightmargin.

When values is FALSE there are only two parts. The default order is
labels on the left confidence intervals on the right.
When no labels are given or labels is FALSE there are only two parts. The default order is
confidence intervals on the left values on the right.

The default order of three parts from left to right is
labels, confidence intervals, values. The order can be changed as shown
by the examples below. The relative widths of the two or three parts
need to be adapted to the actual size of the text of the labels. This
depends on the plotting device and the size of the font and figures and
thus has to be adjusted manually.

Oma can be used to further control horizontal margins, e.g., par(oma=c(0,4,0,4)).

If confidence limits extend beyond the range determined by xlim, then
arrows are drawn at the x-lim borders to indicate that the confidence
limits continue.
}
\examples{

library(Publish)
data(CiTable) 

## A first draft version of the plot is obtained as follows
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper","p")],
          labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")])

## if argument labels is a named list the table is subdivided:
labellist <- split(CiTable[,c("Dose","Time","Mean","SD","n")],CiTable[,"Drug"])
labellist
## the data need to be ordered accordingly
CC= data.table::rbindlist(split(CiTable[,c("HazardRatio","lower","upper")],CiTable[,"Drug"]))
plotConfidence(x=CC, labels=labellist)


## The graph consist of at most three columns:
##
## column 1: labels
## column 2: printed values of the confidence intervals
## column 3: graphical presentation of the confidence intervals
##
## NOTE: column 3 appears always, the user decides if also
##       column 1, 2 should appear
##
## The columns are arranged with the function layout
## and the default order is 1,3,2 such that the graphical
## display of the confidence intervals appears in the middle
##
## the order of appearance of the three columns can be changed as follows
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               order=c(1,3,2))
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               order=c(2,3,1))
## if there are only two columns the order is 1, 2
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               values=FALSE,
               order=c(2,1))
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               values=FALSE,
               order=c(1,2))



## The relative size of the columns needs to be controlled manually
## by using the argument xratio. If there are only two columns
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               xratio=c(0.4,0.15))

## The amount of space on the left and right margin can be controlled
## as follows:
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               xratio=c(0.4,0.15),
               leftmargin=0.1,rightmargin=0.00)

## The actual size of the current graphics device determines
## the size of the figures and the space between them.
## The sizes and line widths are increased as follows:
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               xlab="Hazard ratio",
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               points.cex=3,
               cex=2,
               lwd=3,
               xaxis.lwd=1.3,
               xaxis.cex=1.3)
## Note that 'cex' of axis ticks is controlled via 'par' but
## cex of the label via argument 'cex' of 'mtext'.
## The sizes and line widths are decreased as follows:
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               cex=0.8,
               lwd=0.8,
               xaxis.lwd=0.8,
               xaxis.cex=0.8)

## Another good news is that all figures can be controlled separately

## The size of the graphic device can be controlled in the usual way, e.g.:
\dontrun{
    pdf("~/tmp/testCI.pdf",width=8,height=8)
    plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
                   labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")])
    dev.off()
}

## More control of the x-axis and confidence intervals that
## stretch outside the x-range end in an arrow. 
## the argument xlab.line adjusts the distance of the x-axis
## label from the graph
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               xlab="Hazard ratio",
               xlab.line=1.8,
               xaxis.at=c(0.8,1,1.3),
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               xlim=c(0.8,1.3))

## log-scale
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               xlab="Hazard ratio",
               xlab.line=1.8,
               xaxis.at=c(0.8,1,1.3),
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               xlim=c(0.8,1.3),plot.log="x")
## More pronounced arrows
## Coloured xlab expression
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               xlab=expression(HR[1](s)),
               xlab.line=1.8,
               xlab.col="darkred",
               extremearrows.angle=50,
               extremearrows.length=0.1,
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               xlim=c(0.8,1.3))

## Controlling the labels and their titles
## and the values and their titles
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               xlab="Hazard ratio",
               title.values=expression(bold(HR (CI[95]))),
               title.labels=c("Drug/Time","Dose","Mean","St.dev.","N"),
               factor.reference.pos=c(1,10,19),
               factor.reference.pch=16,
               cex=1.3,
               xaxis.at=c(0.75,1,1.25,1.5,2))

## For factor reference groups, one may want to replace the
## confidence intervals by the word Reference, as in the previous example.
## To change the word 'Reference' we use the argument factor.reference.label:
## To change the plot symbol for the reference lines factor.reference.pch
## To remove the plot symbol in the reference lines use 'NA' as follows:
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               xlab="Hazard ratio",
               factor.reference.label="Ref",
               title.values=expression(bold(HR (CI[95]))),
               title.labels=c("Drug/Time","Dose","Mean","St.dev.","N"),
               factor.reference.pos=c(1,10,19),
               factor.reference.pch=NA,
               cex=1.3,
               xaxis.at=c(0.75,1,1.25,1.5,2))


## changing the style of the graphical confidence intervals
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               xlab="Hazard ratio",
               factor.reference.pos=c(1,10,19),
               points.pch=15,
               points.col=rainbow(27),
               points.cex=2,
               arrows.col="darkblue",
               cex=1.3,
               order=c(1,3,2),
               xaxis.at=c(0.75,1,1.25,1.5))

## the values column of the graph can have multiple columns as well
## to illustrate this we create the confidence intervals
## before calling the function and then cbind them
## to the pvalues
HR <- pubformat(CiTable[,6])
CI95 <- formatCI(lower=CiTable[,7],upper=CiTable[,8],format="(l-u)")
pval <- format.pval(CiTable[,9],digits=3,eps=10^{-3})
pval[pval=="NA"] <- ""
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               values=list("HR"=HR,"CI-95"=CI95,"P-value"=pval),
               cex=1.2,
               xratio=c(0.5,0.3))

## Finally, vertical columns can be delimited with background color
## NOTE: this may slow things down and potentially create
##       large figures (many bytes)
col1 <- rep(c(prodlim::dimColor("green",density=22),
              prodlim::dimColor("green")),length.out=9)
col2 <- rep(c(prodlim::dimColor("orange",density=22),
              prodlim::dimColor("orange")),length.out=9)
col3 <- rep(c(prodlim::dimColor("blue",density=22),
              prodlim::dimColor("blue")),length.out=9)
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               stripes=c(1,0,1),
               stripes.col=c(col1,col2,col3))
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               stripes=c(1,1,1),
               stripes.col=c(col1,col2,col3))

threegreens <- c(prodlim::dimColor("green",density=55),
                 prodlim::dimColor("green",density=33),
                 prodlim::dimColor("green",density=22))
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
               labels=CiTable[,c("Drug.Time","Dose","Mean","SD","n")],
               values=FALSE,
               xlim=c(0.75,1.5),
               stripes=c(1,1,1),
               xratio=c(0.5,0.15),
               stripes.horizontal=c(0,9,18,27)+0.5,
               stripes.col=threegreens)

# combining multiple plots into one
layout(t(matrix(1:5)))
plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
       labels=CiTable[,c("Mean","n")],
       layout=FALSE)
 plotConfidence(x=CiTable[,c("HazardRatio","lower","upper")],
       layout=FALSE)


}
\author{
Thomas A. Gerds <tag@biostat.ku.dk>
}
