\name{sensitivity.analysis}
\alias{sensitivity.analysis}
\title{Sensitivity analysis of basic reproduction ratio to begin/end dates}
\description{Sensitivity analysis of reproduction ratio using supported estimation methods.}
\usage{sensitivity.analysis(incid, GT = NULL, begin = NULL, end = NULL, 
    est.method = NULL, sa.type, res = NULL, GT.type = NULL, GT.mean.range = NULL, 
    GT.sd.range = NULL, t = NULL, date.first.obs = NULL, time.step = 1, 
    ...)}
\arguments{
  \item{incid}{incident cases}
  \item{GT}{generation time distribution}
  \item{begin}{Vector of begins date of the estimation of epidemic}
  \item{end}{Vector of end dates of estimation of the epidemic}
  \item{est.method}{Estimation method used for sensitivity analysis}
  \item{sa.type}{string argument to choose between "time" and "GT" sensitivity analysis.}
  \item{res}{If specified, will extract most of data from a R0.R-class result already generated by est.R0 and run sensitivity analysis on it.}
  \item{GT.type}{Type of distribution for GT (see GT.R for details)}
  \item{GT.mean.range}{mean used for all GT distributions throughout the simulation}
  \item{GT.sd.range}{Range of standard deviation used for GT distributions. Must be provided as a vector.}
  \item{t}{Dates vector to be passed to estimation function}
  \item{date.first.obs}{Optional date of first observation, if t not specified}
  \item{time.step}{Optional. If date of first observation is specified, number of day between each incidence observation}
  \item{\dots}{parameters passed to inner functions}
}
\details{This is a generic call function to use either sa.time or sa.GT. Argument must be chosent accordingly to sa.type.
Please refer to \code{\link{sa.time}} and \code{\link{sa.GT}} for further details about arguments.

'begin' and 'end' vector must have the same length for the sensitivity analysis to run.
They can be provided either as "dates" or "numeric" values, depending on the other parameters (see \code{\link{check.incid}}).
If some begin/end dates overlap, they are ignored, and corresponding uncomputed data are set to NA.
Also, note that unreliable Rsquared values are achieved for very small time period (begin ~ end).
These values are not representative of the epidemic outbreak behaviour.}
\value{An sensitivity analysis object of class "R0.S" with components depending on sensitivity analysis type.}

\author{Pierre-Yves Boelle, Thomas Obadia}




\examples{#Loading package
library(R0)

## Data is taken from the paper by Nishiura for key transmission parameters of an institutional
## outbreak during 1918 influenza pandemic in Germany)
data(Germany.1918)

## For this exemple, we use the exact same call as for the internal sensitivity analysis function

## sa.type = "GT"

## Here we will test GT with means of 1 to 5, each time with SD constant (1)
## GT and SD can be either fixed value or vectors of values
## Actual value in simulations may differ, as they are adapted according to the distribution type
tmp<-sensitivity.analysis(sa.type="GT", incid=Germany.1918, GT.type="gamma", GT.mean=seq(1,5,1), 
                          GT.sd.range=1, begin=1, end=27, est.method="EG")

## Results are stored in a matrix, each line dedicated to a (mean,sd) couple
plot(x=tmp[,"GT.Mean"], xlab="mean GT (days)", y=tmp[,"R"], ylim=c(1.2, 2.1), ylab="R0 (95\% CI)", 
     type="p", pch=19, col="black", main="Sensitivity of R0 to mean GT")
arrows(x0=as.numeric(tmp[,"GT.Mean"]), y0=as.numeric(tmp[,"CI[lower]"]), 
       y1=as.numeric(tmp[,"CI[upper]"]), angle=90, code=3, col="black", length=0.05)
## Note : One could tweak this example to change sorting of values (per mean, or per standard deviation)
## eg: 'x=tmp[,c('GT.Mean')]' could become 'x=tmp[,c('GT.SD')]'


## sa.type="time"

mGT<-generation.time("gamma", c(2.6,1))
sen=sensitivity.analysis(sa.type="time", incid=Germany.1918, GT=mGT, begin=1:15, end=16:30, 
                         est.method="EG")
# ...
# Warning message:
# If 'begin' and 'end' overlap, cases where begin >= end are skipped.
# These cases often return Rsquared = 1 and are thus ignored.
## A list with different estimates of reproduction ratio, exponential growth rate and 95\%CI 
## wtih different pairs of begin and end dates in form of data frame is returned.
## If method is "EG", results will include growth rate and deviance R-squared measure
## Else, if "ML" method is used, growth rate and R-squared will be set as NA

## Interesting results include the variation of R0 given specific begin/end dates.
## Such results can be plot as a colored matrix and display Rsquared=f(time period)
plot(sen, what=c("criterion","heatmap"))
## Returns complete data.frame of best R0 value for each time period 
## (allows for quick visualization)
## The "best.fit" is the time period over which the estimate is the more robust

# $best.fit
#    Time.period Begin.dates  End.dates       R Growth.rate  Rsquared CI.lower. CI.upper.
# 92          15  1970-01-08 1970-01-23 1.64098   0.1478316 0.9752564  1.574953  1.710209}
