% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/base-NullObject.R
\name{NullObject}
\alias{NullObject}
\title{Null Object Pattern}
\usage{
NullObject()
}
\description{
Model a domain concept using natural lingo of the domain
experts, such as "Passenger", "Address", and "Money".
}
\details{
<div class="alert alert-danger">
**Caution:** NullObject is designed for demonstration purposes. Instead of directly using the design pattern as it appears in the package, you'd have to adjust the source code to the problem you are trying to solve.
</div>
<!-- One line about what the function does -->


\emph{\strong{Null Object}} provides special behaviour for particular cases.<div class="alert alert-warning">
**Note:** The Null Object is not the same as the reserved word in R `NULL` (all
 caps).
</div>

\subsection{How It Works}{

When a function fails in R, some functions produce a run-time error while others
return \code{NULL} (and potentially prompt a warning). What the function evokes in
case of a failure is subjected to its programmer discretion. Usually, the
programmer follows either a punitive or forgiving policy regarding how
run-time errors should be handled.

In other occasions, \code{NULL} is often the result of unavailable data. This could
happened when querying a data source matches no entries, or when the system is
waiting for user input (mainly in Shiny).

If it is possible for a function to return \code{NULL} rather than an error,
then it is important to surround it with null test code, e.g.
\code{if(is.null(...)) do_the_right_thing()}. This way the software would do the
right thing if a null is present.

Often the right thing is the same in many contexts, so you end up writing
similar code in lots of places—committing the sin of code duplication.

Instead of returning \code{NULL}, or some odd value such as \code{NaN} or \code{logical(0)}, return a \strong{Null Object} that has the same interface as what the caller expects. In R, this often means returning a \code{data.frame} structure, i.e. column names and variables types, with no rows.
}

\subsection{When to Use It}{
\itemize{
\item In situations when a subroutine is likely to fail, such as loss of Internet
or database connectivity. Instead of prompting a run-time error, you could
return the \strong{Null Object} as part of a \href{https://en.wikipedia.org/wiki/Graceful_exit}{gracefully failing} strategy. A common
strategy employs \code{tryCatch} that returns the \strong{Null Object} in the case of
an error:
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Simulate a database that is 5\% likely to fail 
read_mtcars <- function() if(runif(1) < 0.05) stop() else return(mtcars)

# mtcars null object constructor
NullCar <- function() mtcars[0,]

# How does the null car object look like? 
NullCar()
#>  [1] mpg  cyl  disp hp   drat wt   qsec vs   am   gear carb
#> <0 rows> (or 0-length row.names)

# Subroutine with gracefully failing strategy
set.seed(1814)
cars <- tryCatch(
  # Try reading the mtcars dataset
  read_mtcars(), 
  # If there is an error, return the Null Car object
  error = function(e) return(NullCar()) 
)

# Notice: Whether the subroutine fails or succeeds, it returns a tibble with 
# the same structure.
colnames(cars)
#>  [1] "mpg"  "cyl"  "disp" "hp"   "drat" "wt"   "qsec" "vs"   "am"   "gear"
#> [11] "carb"
}\if{html}{\out{</div>}}
\itemize{
\item In Shiny dashboards
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{geom_null <- function(...)\{
  ggplot2::ggplot() + ggplot2::geom_blank() + ggplot2::theme_void()
\}

if(exists("user_input"))\{
  ggplot2::ggplot(user_input, ggplot::aes(x = mpg, y = hp)) + ggplot2::geom_point()
\} else \{
  geom_null() + geom_text(aes(0,0), label = "choose an entry from the list")
\}
}\if{html}{\out{</div>}}
\itemize{
\item In unit-tests
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{classes <- function(x) sapply(x, class)
    
test_that("mtcars follows a certain table structure", \{
    # Compare column names
    expect_identical(colnames(mtcars), colnames(NullCar()))
    # Compare variable types
    expect_identical(classes(mtcars), classes(NullCar()))
\})
}\if{html}{\out{</div>}}
}
}
\examples{
# See more examples at <https://tidylab.github.io/R6P/articles>

colnames(NullObject())
nrow(NullObject())
}
\seealso{
Other base design patterns: 
\code{\link{Singleton}},
\code{\link{ValueObject}()}
}
\concept{base design patterns}
