% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/medde.R
\name{medde}
\alias{medde}
\title{Maximum Entropy [De]Regularized Density Estimation}
\usage{
medde(x, v = 300, lambda = 0.5, alpha = 1, Dorder = 1, w = NULL,
  rtol = 1e-06, verb = 0, control = NULL)
}
\arguments{
\item{x}{Data: either univariate or bivariate (not yet implemented in
Rmosek)}

\item{v}{Undata: either univariate or bivariate, by default there is an
equally spaced grid of 300 values}

\item{lambda}{total variation penalty parameter, if lambda is in [-1,0], a
concavity constraint is imposed. If lambda is in \eqn{(-\infty, -1)} a convexity
constraint on \eqn{0.5 x^2 + \log f(x)} is imposed. See Koenker and Mizera (2013) for
further details on this last option, and Koenker and Mizera (2010) for
further details on the concavity constrained options.  The demo Brown
recreates the monotonized Bayes rule example in Figure 1 of the 2013 paper.}

\item{alpha}{Renyi entropy parameter characterizing fidelity criterion
by default 1 is log-concave and 0.5 is Hellinger.}

\item{Dorder}{Order of the derivative operator for the penalty}

\item{w}{weights corresponding to x observations}

\item{rtol}{Convergence tolerance for Mosek algorithm}

\item{verb}{Parameter controlling verbosity of solution, 0 for silent, 5
gives rather detailed iteration log.}

\item{control}{Mosek control list see KWDual documentation}
}
\value{
An object of class "medde" with components \item{x}{points of
evaluation on the domain of the density} \item{y}{estimated function values
at the evaluation points x}  \item{logLik}{log Likelihood provided \code{alpha = 1}
otherwise NULL} \item{status}{exit status from Mosek}
}
\description{
Density estimation based on maximum entropy methods
}
\details{
See the references for further details. And also Mosek "Manuals". The
acronym, according to the urban dictionary has a nice connection to
a term used in Bahamian dialect, mostly on the Family Islands like Eleuthera
and Cat Island meaning "mess with" "get involved," "get entangled," "fool
around," "bother:"
"I don't like to medder up with all kinda people"
"Don't medder with people (chirren)"
"Why you think she medderin up in their business."

This version implements a class of penalized density estimators solving:
\deqn{\min_x \phi(x_1) | A_1 x_1 - A_2 x_2 = b,  0 \le x_1, -\lambda \le x_2 \le \lambda }{
min_x \phi(x_1) | A_1 x_1 - A_2 x_2 = b,  0 \le x_1, -\lambda \le x_2 \le \lambda }
where \eqn{x} is a vector with two component subvectors: \eqn{x_1} is a
vector of function values of the density \eqn{x_2} is a vector of dual values,
\eqn{\lambda} is typically positive, and controls the fluctuation of the Dorder
derivative of some transform of the density. When alpha = 1 this transform is
simply the logarithm of the density, and Dorder = 1 yields a piecewise exponential
estimate; when Dorder = 2 we obtain a variant of Silverman's (1982) estimator
that shrinks the fitted density toward the Gaussian, i.e. with total variation
of the second derivative of \eqn{log f} equal to zero.  See demo(Silverman) for
an illustration of this case.  If \eqn{\lambda} is in \eqn{(-1,0]} then the
\eqn{x_2} constraint is replaced by \eqn{x_2 \geq 0}, which for \eqn{\alpha = 1},
constrains the fitted density to be log-concave; for \eqn{\alpha = 0.5},  \eqn{-1/\sqrt f}
is constrained to be concave; and for \eqn{\alpha \le 0}, \eqn{1/f^{\alpha -1}} is
constrained to be concave.  In these cases no further regularization of the smoothness
of density is required as the concavity constraint acts as  regularizer.
As explained further in Koenker and Mizera (2010) and
Han and Wellner (2016) decreasing \eqn{\alpha} constrains the fitted density to lie
in a larger class of quasi-concave
densities.  See \code{demo(velo)} for an illustration of these options, but be aware
that more extreme \eqn{\alpha} pose more challenges from an numerical optimization
perspective.  Fitting for \eqn{\alpha < 1} employs a fidelity criterion closely
related to Renyi entropy that is more suitable than likelihood for very peaked, or very heavy
tailed target densities.  For \eqn{\lambda < 0}  fitting for \code{Dorder != 1}
proceed at your own risk.  When \eqn{\lambda < -1} a convexity constraint is
imposed on \eqn{0.5 x^2 + log f(x)} that ensures that the resulting Bayes rule,
aka Tweedie formula, is monotone in \eqn{x}, as described further in Koenker and
Mizera (2013).
}
\examples{
#Maximum Likelihood Estimation of a Log-Concave Density
set.seed(1968)
x <- rgamma(50,10)
m <- medde(x, v = 50, lambda = -.5, verb = 5)
plot(m, type = "l")
lines(m$x,dgamma(m$x,10),col = 2)
points(x,m$g,cex = 0.5)
rug(x)
title(paste("log likelihood = ", round(m$logLik,2)))
legend(14,.12,c("ghat","true"),lty = 1, col = 1:2)

#Maximum Likelihood Estimation of a Gamma Density with TV constraint
set.seed(1968)
x <- rgamma(50,5)
f <- medde(x, v = 50, lambda = 0.005, verb = 5)
plot(f, type = "l")
lines(f$x,dgamma(f$x,5),col = 2)
legend(10,.15,c("ghat","true"),lty = 1, col = 1:2)
}
\author{
Roger Koenker and Ivan Mizera
}
\references{
Chen, Y. and R.J. Samworth, (2013) "Smoothed log-concave
maximum likelihood estimation with applications", \emph{Statistica Sinica},
23, 1373--1398.

Han, Qiyang and Jon Wellner (2016) ``Approximation and estimation of s-concave
densities via Renyi divergences, \emph{Annals of Statistics}, 44, 1332-1359.

Koenker, R and I. Mizera, (2007) ``Density Estimation by Total Variation
Regularization,'' \emph{Advances in Statistical Modeling and Inference:
Essays in Honor of Kjell Doksum}, V.N. Nair (ed.), 613-634.

Koenker, R and I. Mizera, (2006) ``The alter egos of the regularized maximum
likelihood density estimators: deregularized maximum-entropy, Shannon,
Renyi, Simpson, Gini, and stretched strings,'' \emph{ Proceedings of the 7th
Prague Symposium on Asymptotic Statistics}.

Koenker, R and I. Mizera, (2010) ``Quasi-Concave Density Estimation''
\emph{Annals of Statistics}, 38, 2998-3027.

Koenker, R and I. Mizera, (2013) ``Convex Optimization, Shape Constraints,
Compound Decisions, and Empirical Bayes Rules,'' JASA, 109, 674--685.

Koenker, R and I. Mizera, (2014) ``Convex Optimization in R.'',
\emph{Journal of Statistical Software}, 60, 1-23.
}
\seealso{
This function is based on an earlier function of the same name in
the deprecated package MeddeR that was based on an R-Matlab interface.
A plotting method is available, or medde estimates can be added to plots
with the usual \code{lines(meddefit, ...} invocation.  For log concave
estimates there is also a quantile function \code{qmedde} and a random
number generation function \code{rmedde}.
}
\keyword{nonparametric}

