#' @title
#' Extract data tibbles from a REDCapTidieR supertibble and bind them to an
#' environment
#'
#' @description
#' Take a supertibble generated with `read_redcap()`
#' and bind its data tibbles (i.e. the tibbles in the `redcap_data` column) to
#' an environment. The default is the global environment.
#'
#' @returns
#' This function returns nothing as it's used solely for its side effect of
#' modifying an environment.
#'
#' @param supertbl A supertibble generated by `read_redcap()`. Required.
#' @param environment The environment to bind the tibbles to. Default is
#' `rlang::global_env()`.
#' @param tbls A vector of the `redcap_form_name`s of the data tibbles to bind to
#' the environment. Default is `NULL` which binds all data tibbles.
#'
#' @importFrom dplyr filter pull %>%
#' @importFrom rlang env_poke current_env new_environment global_env .data
#' @importFrom purrr map2 pluck
#'
#' @examples
#' # Create an empty environment
#' my_env <- new.env()
#'
#' ls(my_env)
#'
#' # Mock up a supertibble
#' supertbl <- tibble::tribble(
#'   ~redcap_form_name,    ~redcap_data,   ~structure,
#'   "super_hero_powers",  list(),         "repeating",
#'   "heroes_information", list(),         "nonrepeating"
#' )
#'
#' bind_tibbles(supertbl, my_env)
#'
#' ls(my_env)
#'
#' @export

bind_tibbles <- function(supertbl,
                         environment = global_env(),
                         tbls = NULL) {
  # Name variables
  my_supertbl <- supertbl

  # Apply conditional loading for specific instruments
  if (!is.null(tbls)) {
    my_supertbl <- my_supertbl %>%
      filter(.data$redcap_form_name %in% tbls)
  }

  table_names <- my_supertbl$redcap_form_name

  # Map over table names and environment data to load into environment
  map2(
    .x = table_names,
    .y = my_supertbl$redcap_data,
    .f = ~ env_poke(
      env = environment,
      nm = .x,
      value = .y
    )
  )
  return(invisible(NULL))
}
