% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prediction_model.v16.R
\name{estimate.mle}
\alias{estimate.mle}
\title{Fit the Model}
\usage{
estimate.mle(
  hessian = FALSE,
  H.E = NULL,
  H.W = NULL,
  cases = NULL,
  cfg = NULL,
  ini_params = NULL,
  params_limits = NULL,
  restrictions = NULL,
  restriction.starts = NULL,
  ranges = NULL,
  rt_func = 1,
  silence.errors = FALSE,
  fit.t.pred = NULL,
  param_scale = NULL,
  num.iter = NULL,
  scenario = NULL,
  adj.period = NULL,
  population = NULL,
  rho = NULL,
  serial_mean = serial_mean,
  serial_var = serial_var,
  lt = NULL,
  window_size = NULL,
  verbose = FALSE
)
}
\arguments{
\item{hessian}{Logical. If TRUE, computes the variance-covariance matrix at the MLE.}

\item{H.E}{Mobility metrics for category Retail & Entertainment. Currently unsupported.}

\item{H.W}{Mobility metrics for category Workplaces. Currently unsupported.}

\item{cases}{Vector of case counts.}

\item{cfg}{The object that contains all variables from the configuration file.
This includes all function arguments except for \code{cases}, \code{hessian}, \code{H.E}, and \code{H.W}. All other
arguments are overridden if \code{cfg} is passed to the method.}

\item{ini_params}{Initial parameter values to be used in optimization. Includes the following sets of parameters in a vector, in this order:
\itemize{
  \item   (a1,a2,a3,a4) = parameters for curve c() specified by \code{rt_func};
  \item   nu = loss of immunity rate;
  \item   (v2,v3,v4,v5) = transmissibility of variants in waves 2+, as relative multiplication factors compared to transmissibility in wave 1;
  \item   (psi1,psi2,psi3,psi4) = psi parameters for severity levels 1,2,3 and 4.
  \item   (u,v) = variance parameters. Only u is currently in use.
  \item   (beta0,beta.R,beta.E,beta.W), when restrictions = NULL. Currently unsupported.
  }}

\item{params_limits}{Boundaries/limits of the ini_params.}

\item{restrictions}{A numeric integer vector giving the severity of restrictions.
Zero means no restriction, and higher numbers means greater severity/disruption.
The ordered unique values should be consecutive integers starting from zero.
Each number (other than 0) adds a new parameter to the fit. restrictions = NULL
causes the function to use mobility data instead of the psi values (currently unsupported).}

\item{restriction.starts}{A vector of same length as restrictions, of times when restrictions
came into effect. Note: the first index time should be 1.}

\item{ranges}{An vector of time ranges for the different waves.
The waves ranges should be contiguous, with at least one unit of time
between consecutive waves.}

\item{rt_func}{The parametric form of function c(). Options are listed under function c_helper.}

\item{silence.errors}{Logical. If TRUE, ignores certain errors to allow optimization to proceed. Not all errors can be ignored.}

\item{fit.t.pred}{Time from which prediction is done. If use.actual.not.predicted is TRUE, values of \eqn{S_t} before this time will be computed using actual counts.}

\item{param_scale}{Parameter scale. Passed as argument parscale to optim.}

\item{num.iter}{Maximum number of iterations. Passed as argument maxit to optim.}

\item{scenario}{A character string describing options to deal with restrictions. Currently unsupported.}

\item{adj.period}{Delays in society adjusting.}

\item{population}{total population size.}

\item{rho}{A vector of under-reporting rates of the same length as cases. If a scalar is supplied, the vector will be constant with this value.}

\item{serial_mean}{Mean of the serial interval on the log scale.}

\item{serial_var}{Variance of the serial interval on the log scale.}

\item{lt}{The length of cases.}

\item{window_size}{The maximum value for the serial interval.}

\item{verbose}{Logical. If TRUE, provides additional details while running the function.}
}
\value{
A list of maximum likelihood estimates of the parameters. Includes:
 \itemize{
   \item a1
   \item a2
   \item a3
   \item a4
   \item nu
   \item vt_params_est
   \item Psi
   \item betas
   \item negative_log_lik
   \item mle
   \item hessian
   \item SE
 }
}
\description{
Estimate the parameters of the model by maximizing the likelihood function (or, rather, by minimizing the negative log likelihood).
}
\examples{
library(REffectivePred)
## Read in the data
path_to_data <- system.file("extdata/NY_OCT_4_2022.csv", package = "REffectivePred")
data <- read.csv(path_to_data)
head(data)
cases <- diff(c(0, data$cases)) # Convert cumulative cases into daily cases
lt <- length(cases)             # Length of cases
Time <- as.Date(data$date, tryFormats = c("\%d-\%m-\%Y", "\%d/\%m/\%Y"))

navigate_to_config() # Open the config file, make any necessary changes here.
path_to_config <- system.file("config.yml", package = "REffectivePred")  # Read config file
cfg <- load_config()    # Build the cfg object

##### Option 1: populate the global environment with args to pass to function.
population <- cfg$population # Population size
window_size <- cfg$window.size
adj.period <- cfg$adj.period
fit.t.pred <- cfg$fit.t.pred # Time of prediction
not.predict <- cfg$not.predict
rt.func.num <- cfg$rt.func.num # choose which Rt function you want to use
num.iter <- cfg$num.iter
silence.errors <- cfg$silence.errors
predict.beyond <- cfg$predict.beyond
curve_params <- as.double(unlist(cfg$curve_params))
vt_params <- as.double(unlist(cfg$vt_params)) # The vt initial values, starting at wave 2
restriction_levels <- as.double(unlist(cfg$restriction_levels)) # Psi, u, and v parameters
betas <- as.double(unlist(cfg$betas)) #   betas
ini_params <- c(curve_params, vt_params, restriction_levels, betas)
restrictions_params <- cfg$restrictions_params
restriction_st_params <- cfg$restriction_st_params
param_scale <- abs(ini_params) / 10
waves_list <- ranges_to_waves(cfg$waves_list)
params_limits <- cfg$params_limits
num_waves <- cfg$num_waves
waves <- waves_1d_list(num_waves, waves_list)
rho <- eval(parse(text = cfg$rho))
serial_mean <- cfg$serial_mean
serial_var <- cfg$serial_var

est <- estimate.mle(
  ini_params = ini_params,
  params_limits = params_limits,
  restrictions = restrictions_params,
  restriction.starts = restriction_st_params,
  ranges = waves,
  rt_func = rt.func.num,
  silence.errors = silence.errors,

  fit.t.pred = fit.t.pred,
  param_scale = param_scale,
  num.iter = num.iter,
  cases = cases,
  scenario = NULL,
  H.E = NULL,
  H.W = NULL,
  adj.period = adj.period,
  population = population,
  rho = rho,
  serial_mean = serial_mean,
  serial_var = serial_var,
  lt = lt,
  window_size = window_size,
  hessian = FALSE
)
print(est)

##### Option 2: pass the cfg object instead.
est <- estimate.mle(
    cases = cases,
    cfg = cfg,
    hessian = FALSE
    )
print(est)
}
