% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/do.optimGraphviz.R
\name{do.optimGraphviz}
\alias{do.optimGraphviz}
\title{Optimize the Dimensional Anchors Position using the Graphviz algorithm}
\usage{
do.optimGraphviz(
  x,
  graph,
  attractG = 1,
  repelG = 1,
  law = 0,
  steps = 10,
  springs = NULL
)
}
\arguments{
\item{x}{a data.frame or matrix to be projected, with column names matching row names in springs}

\item{graph}{\code{igraph} object}

\item{attractG}{Number specifying the weight of the attractive forces}

\item{repelG}{Number specifying the weight of the repulsive forces}

\item{law}{Integer, specifying how forces change with distance: 0 = (inverse) linear, 1 = (inverse) square}

\item{steps}{Number of iterations of the algorithm before re-considering convergence criterion}

\item{springs}{Numeric matrix with initial anchor coordinates. When \code{NULL} (=default), springs are initialized by \code{\link{make.S}}}
}
\value{
A matrix with 2 columns (x and y coordinates of dimensional anchors) and 1 line
         per dimensional anchor (so called springs).
}
\description{
Allows to compute the best arrangement of Dimensional Anchors so that
visualization efficiency (i.e. maintaining graph structure) is optimized.
The Graphviz algorithm is implemented in C++ for optimal computational efficiency.
}
\details{
Graphviz is a variant of Freeviz (\code{\link{do.optimFreeviz}}, applicable to a dataset for which a graph structure (i.e. \code{igraph} object) is available.
Attractive forces are defined between connected nodes in the graph, and repulsive forces between all non-connected nodes.
To better maintain the original graph structure after projection, spring constants between connected nodes are proportional to their edge weights.
Graphviz can be used as an alternative to Freeviz when class labels are not available.
}
\examples{
data(iris)
das <- c('Sepal.Length','Sepal.Width','Petal.Length','Petal.Width')
S <- make.S(das)
rv <- do.radviz(iris,S)

plot(rv,anchors.only=FALSE)

## compute distance matrix
d.iris <- dist(iris[,das])

## define a kNN matrix
n.iris <- as.matrix(d.iris)
n.iris <- apply(n.iris,1,function(x,k=12) {
  x[order(x)>(k+1)] <- 0
  return(x)
})
diag(n.iris) <- 0

## compute weights for kNN matrix
w.iris <- n.iris
w.iris <- exp(-w.iris^2/(2*median(w.iris[w.iris!=0])^2))
w.iris[n.iris==0] <- 0

## create graph
library(igraph)
g.iris <- graph.adjacency(w.iris,mode='undirected',weight=TRUE,diag=FALSE)

V(g.iris)$Species <- as.character(iris[,'Species'])
V(g.iris)$color <- as.numeric(iris[,'Species'])

plot(g.iris,
     vertex.label=NA)

## project using Radviz
new.S <- do.optimGraphviz(iris[,das],
                          g.iris)

grv <- do.radviz(iris[,das],
                new.S,
                graph=g.iris)

library(ggplot2)
plot(grv)+
  geom_point(aes(color=iris[,'Species']))
}
\author{
Nicolas Sauwen
}
