/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.docs;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.Vector;

import org.lysis.reporters.html4r.elements.Iframe;
import org.lysis.reporters.html4r.elements.RAPHAELGraphics;
import org.lysis.reporters.html4r.elements.TOC;
import org.lysis.reporters.html4r.elements.Title;
import org.lysis.reporters.html4r.elements.menu.BootstrapMenu;
import org.lysis.reporters.img.Image;
import org.lysis.reporters.lists.NumberingDefinition;
import org.lysis.reporters.tables.FlexTable;
import org.lysis.reporters.text.ParagraphSet;
import org.lysis.reporters.text.RScript;
import org.lysis.reporters.tools.HTML4R;


public class BootstrapPage {
	
	public static int error = 0;
	public static int noproblem = 1;
	public static int fileproblem = 2;
	
	private TOC titles;
	private String title;
	private LinkedHashMap<Integer, HTML4R> content;
	private int ncontent;
	private String charset;
	private String description;
	private String keywords;
	private Vector<String> cssfiles;
	private Vector<String> jsfiles;
	private Vector<String> jscodes;
	private Vector<String> csscodes;
	
	private NumberingDefinition numbDefinition;
	private long list_id;
	private Vector<String> counters;
	
	private BootstrapMenu headerMenu;
	private LinkedList<ParagraphSet> footer;

	public BootstrapPage ( String title, String charset, NumberingDefinition numberingDefinition, String desc, String kwds ) throws IOException {
		this.title = title;
		titles = new TOC(this.title);
		content = new LinkedHashMap<Integer, HTML4R>();
		ncontent = 0;
		setCharset(charset);
		cssfiles = new Vector<String>();
		jsfiles = new Vector<String>();
		jscodes = new Vector<String>();
		csscodes = new Vector<String>();
		list_id = 1;
		counters = new Vector<String>();
		numbDefinition = numberingDefinition;
		jscodes.add("$(document).ready(function() {$('.tip').tooltip();});");
		csscodes.add(".tip::before{counter-increment:counter-tip;content:\"*\";vertical-align:super;}");
		footer = new LinkedList<ParagraphSet>();
		headerMenu = null;
		description = desc;
		keywords = kwds;
	}
	
	private int addHTML4R( HTML4R elt )  {
		
		if( elt instanceof org.lysis.reporters.html4r.elements.Title ){
			titles.addTitle((org.lysis.reporters.html4r.elements.Title)elt);
		}
		content.put(ncontent, elt);
		if( elt.hasJS() ) jscodes.add(elt.getJS());
		if( elt.hasCSS() ) csscodes.add(elt.getCSS());
		ncontent++;
		return noproblem;
	}
	
	public void addBootstrapMenu( BootstrapMenu hm ){
		headerMenu = hm;
	}
	public int add( Title elt )  {
		return addHTML4R(elt);
	}
	
	public int add( RScript elt )  {
		return addHTML4R(elt);
	}
	
	public int add( Iframe elt )  {
		return addHTML4R(elt);
	}

	public int add( FlexTable elt )  {
		elt.setHTMLReference(list_id, counters, numbDefinition);
		return addHTML4R(elt);
	}
	
	public void restartNumbering(){
		list_id++;
	}

	public int add( ParagraphSet elt )  {
		elt.setHTMLReference(list_id, counters, numbDefinition);
		return addHTML4R(elt);
	}

	public void addToFooter( ParagraphSet elt )  {
		elt.setHTMLReference(list_id, counters, numbDefinition);
		footer.add( elt );
		if( elt.hasJS() ) jscodes.add(elt.getJS());
		if( elt.hasCSS() ) csscodes.add(elt.getCSS());
	}
	
	public int add( Image elt )  {
		return addHTML4R(elt);
	}
	
	public int add( RAPHAELGraphics elt )  {
		return addHTML4R(elt);
	}
	
	public void setCharset(String charset ){
		this.charset = charset;
	}

	public void addStylesheet( String file ){
		if( !cssfiles.contains(file) ) cssfiles.add(file);
	}
	
	public void addJavascript( String file ){
		if( !jsfiles.contains(file) ) jsfiles.add(file);
	}
	
	public void addJavascriptCode( String text ){
		jscodes.add(text);
	}
	
	private String getStylesheets(){
		String out = "";
		for( int i = 0 ; i < cssfiles.size() ; i++ ){
			out += "<link rel=\"stylesheet\" href=\"" + cssfiles.get(i) + "\" type=\"text/css\" media=\"all\" />";
		}
		return out;
	}
	
	private String getScripts(){
		String out = "";
		for( int i = 0 ; i < jsfiles.size() ; i++ ){
			out += "<script type=\"text/javascript\" src=\"" + jsfiles.get(i) + "\"></script>";
		}
		return out;
	}
	
	public String getHead() {
		String out = "";
		out += "<head>";

		out += "<meta charset=\"" + charset + "\">";
		out += "<meta http-equiv=\"X-UA-Compatible\" content=\"IE=edge\">";
		out += "<meta name=\"viewport\" content=\"width=device-width, initial-scale=1.0\" />";
		out += "<title>"; 
		out += org.apache.commons.lang.StringEscapeUtils.escapeHtml(title);
		out += "</title>";
		out += "<meta name=\"description\" content=\"" + description + "\">";
		out += "<meta name=\"keywords\" content=\"" + keywords + "\">";
		out += getStylesheets();
		out += getScripts();
		out += "<!--[if lt IE 9]><script src=\"js/ie8-responsive-file-warning.js\"></script><![endif]-->";
		out += "<!-- IE10 viewport hack for Surface/desktop Windows 8 bug -->";
		out += "<script src=\"js/ie10-viewport-bug-workaround.js\"></script>";
		out += "<script src=\"js/ie-emulation-modes-warning.js\"></script>";
		out += "<!-- HTML5 shim and Respond.js IE8 support of HTML5 elements and media queries -->";
		out += "<!--[if lt IE 9]>";
		out += "  <script src=\"js/html5shiv.min.js\"></script>";
		out += "  <script src=\"js/respond.min.js\"></script>";
		out += "<![endif]-->";
		out += getCSSCode();

		if( counters.size() > 0 ){
			out += "<style type=\"text/css\">";
			out += numbDefinition.getCSS();
			out += "</style>";
		}
		out += "</head>";
		return out;

	}
	
	public String getJSCode() {
		String out = "";
		out += "<script type=\"text/javascript\">";

			for( int i = 0 ; i < jscodes.size() ; i++ ){
				out += jscodes.get(i);
			}
		out += "</script>";
		return out;

	}
	public String getCSSCode() {
		String out = "";
		out += "<style type=\"text/css\">";

			for( int i = 0 ; i < csscodes.size() ; i++ ){
				out += csscodes.get(i);
			}
		out += "</style>";
		return out;

	}
	public String getBody( int index ) {
		return content.get(index).getHTML();
	}
	
	public int writeHtml(String target) throws IOException {		
		File f = new File(target);
		BufferedWriter output;
        try {
			output = new BufferedWriter(new FileWriter(f));
		} catch (IOException e) {
			return fileproblem;
		}
		if( titles.hasJS() ) jscodes.add(titles.getJS());
		if( headerMenu != null ) csscodes.add(headerMenu.getCSS());

		String mainContent = "";
		for( int i = 0 ; i < ncontent ; i++ ){
			HTML4R elt_output = content.get(i);

			if( elt_output.hasHTML() ) {
				mainContent += elt_output.getHTML();
			}
		}
        output.write( "<!DOCTYPE html>");
        output.write( "<html>" );
        output.write( getHead() );
		output.write("<body>");
		output.write("<a class=\"sr-only sr-only-focusable\" href=\"#content\">Skip to main content</a>");
		
		if( headerMenu != null ) output.write(headerMenu.getHTML());

		
		output.write("<div class=\"container bs-docs-container\">");//container
		    output.write("<div class=\"row\">");//row
		        output.write("<div class=\"col-md-9\" role=\"main\">");;

		        output.write(mainContent);
		        
		        output.write("</div>");//col-md-9
		    output.write("<div class=\"col-md-3\">");
		        output.write(titles.getHTML());
	        output.write("</div>");

	        output.write("</div>");//row
        output.write("</div>");//container

        if( footer.size() > 0 ){
        	output.write("<footer class=\"bs-docs-footer\" role=\"contentinfo\">"
				  + "<div class=\"container\">");

			for( int i = 0 ; i < footer.size() ; i++ ){
				output.write(footer.get(i).getHTML());
			}
			output.write("</div>");
			output.write("</footer>");
        }

		
		output.write(getJSCode());
		
		output.write("</body>");
		
        output.write( "</html>" );

		output.close();
        return noproblem;
        
	}
}

