/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.tables;

import java.io.IOException;
import java.math.BigInteger;
import java.util.LinkedHashMap;
import java.util.List;

import javax.xml.bind.JAXBElement;

import org.docx4j.dml.CTTable;
import org.docx4j.dml.CTTableCol;
import org.docx4j.dml.CTTableGrid;
import org.docx4j.dml.CTTableProperties;
import org.docx4j.dml.CTTableRow;
import org.docx4j.dml.Graphic;
import org.docx4j.dml.GraphicData;
import org.docx4j.jaxb.Context;
import org.docx4j.wml.CTTblLayoutType;
import org.docx4j.wml.Jc;
import org.docx4j.wml.JcEnumeration;
import org.docx4j.wml.STTblLayoutType;
import org.docx4j.wml.Tbl;
import org.docx4j.wml.TblGrid;
import org.docx4j.wml.TblGridCol;
import org.docx4j.wml.TblPr;
import org.docx4j.wml.Tr;
import org.docx4j.wml.TrPr;
import org.lysis.reporters.formats.BorderProperties;
import org.lysis.reporters.formats.CellProperties;
import org.lysis.reporters.formats.ParProperties;
import org.lysis.reporters.formats.TextProperties;
import org.lysis.reporters.html4r.tools.Format;
import org.lysis.reporters.text.Paragraph;
import org.lysis.reporters.text.ParagraphSet;
import org.lysis.reporters.tools.DOCX4R;
import org.lysis.reporters.tools.Debug;
import org.lysis.reporters.tools.DocComponent;
import org.lysis.reporters.tools.HTML4R;
import org.lysis.reporters.tools.PPTX4R;
import org.pptx4j.pml.CTGraphicalObjectFrame;

public class FlexTable extends DocComponent implements HTML4R, PPTX4R, DOCX4R {

	private MetaRows headerFlexRowList;
	private MetaRows footerFlexRowList;

	private int nrow, ncol;
	private ParagraphSet[][] cellTextValue;
	private CellProperties[][] cellPropertiesIndex;
	protected LinkedHashMap<Integer, int[]> rowSpanInstructions ;
	protected LinkedHashMap<Integer, int[]> colSpanInstructions ;
	protected double[] widths ;

	protected long table_width ;
	
	public FlexTable(int n, int c
			, TextProperties tp, ParProperties pp, CellProperties cp ) {
		cellTextValue = new ParagraphSet[n][c];
		widths = new double[c];
		cp.setTextDirection("lrtb");
		for( int j = 0 ; j < c ; j++ ) widths[j] = -1;
		
		cellPropertiesIndex = new CellProperties[n][c];
		for(int i = 0 ; i < n ; i++ ){
			for( int j = 0 ; j < c ; j++ ){
				Paragraph par = new Paragraph();
				try {
					par.addText("", tp);
				} catch (IOException e) {
				}
				cellTextValue[i][j] = new ParagraphSet(pp);
				cellTextValue[i][j].addParagraph(par);
				cellPropertiesIndex[i][j] = cp.getClone();
			}
		}
		
		headerFlexRowList = new MetaRows();
		footerFlexRowList = new MetaRows();
		nrow = n;
		ncol = c;
		rowSpanInstructions = new LinkedHashMap<Integer, int[]>();
		colSpanInstructions = new LinkedHashMap<Integer, int[]>();
		table_width = -1;
	}
	
	public MetaRows getHeader(){
		return headerFlexRowList;
	}
	public MetaRows getFooter(){
		return footerFlexRowList;
	}
	public void setWidths( double[] w ) {
		widths = w;
	}
	public void setTableWidth( long w ) {
		table_width = w;
	}
	
	public void setRowSpanInstructions( int colindex, int[] x ) {
		rowSpanInstructions.put(colindex, x);
	}
	public void setColSpanInstructions( int rowindex, int[] x ) {
		colSpanInstructions.put(rowindex, x);
	}
			
	public void addBodyText(int[] i, int[] j, String par[], TextProperties tp, boolean newPar) throws IOException{
//		if( Debug.debug ) System.err.println("addBodyText String["+par.length+"]");

		int li = i.length;
		int lj = j.length;
		for(int row = 0 ; row < li ; row++ ){
			for( int col = 0 ; col < lj ; col++ ){
				ParagraphSet value = cellTextValue[i[row]][j[col]];

				if( newPar ) {
					Paragraph p = new Paragraph();
					p.addText(par[row * lj + col], tp);

					value.addParagraph(p);
				} else {
					Paragraph p = value.getLast();
					p.addText(par[row * lj + col], tp);

				}

			}
		}
	}
	
	

	
	
	public void addBodyText(int[] i, int[] j, Paragraph par, boolean newPar) throws IOException{
		if( Debug.debug ) System.err.println("addBodyText Paragraph");

		int li = i.length;
		int lj = j.length;
		for(int row = 0 ; row < li ; row++ ){
			for( int col = 0 ; col < lj ; col++ ){
				ParagraphSet value = cellTextValue[i[row]][j[col]];
				if( newPar ) {
					value.addParagraph(par);
				} else {
					Paragraph p = value.getLast();
					p.addParagraph(par);
				}
			}
		}
	}
	
	public void setRowsColors(int[] i, String[] colors){
		int li = i.length;
		for(int row = 0 ; row < li ; row++ ){
			for(int col = 0 ; col < ncol ; col++ ){
				cellPropertiesIndex[i[row]][col].setBackgroundColor(colors[row]);
			}
		}
	}
	
	public void setColumnsColors(int[] j, String[] colors){
		int lj = j.length;
		for(int col = 0 ; col < lj ; col++ ){
			for(int row = 0 ; row < nrow ; row++ ){
				cellPropertiesIndex[row][j[col]].setBackgroundColor(colors[col]);
			}
		}
	}
	
	public void setOddEvenColor(String odd_color, String even_color){
		for(int row = 0 ; row < nrow ; row = row +2 ){
			for(int col = 0 ; col < ncol ; col++ ){
				cellPropertiesIndex[row][col].setBackgroundColor(odd_color);
			}
		}
		for(int row = 1 ; row < nrow ; row = row +2 ){
			for(int col = 0 ; col < ncol ; col++ ){
				cellPropertiesIndex[row][col].setBackgroundColor(even_color);
			}
		}
	}

	public void setBodyBorders(BorderProperties inner_v, BorderProperties inner_h, BorderProperties outer_v, BorderProperties outer_h){
		if( Debug.debug ) System.err.println("setBodyBorders");
//		inner_v.print();

		for(int row = 0 ; row < nrow ; row++ ){
			for(int col = 0 ; col < ncol ; col++ ){
				if( Debug.debug ) {
					System.out.println("Inner v");inner_v.print();
					System.out.println("Inner h");inner_h.print();
				}

				cellPropertiesIndex[row][col].setBorderLeft(inner_v);
				cellPropertiesIndex[row][col].setBorderRight(inner_v);
				cellPropertiesIndex[row][col].setBorderTop(inner_h);
				cellPropertiesIndex[row][col].setBorderBottom(inner_h);
			}
		}
		if( Debug.debug ) {
			System.out.println("Outer v");inner_v.print();
			System.out.println("Outer h");inner_h.print();
		}
		for(int col = 0 ; col < ncol ; col++ ){
			cellPropertiesIndex[0][col].setBorderTop(outer_h);
			cellPropertiesIndex[nrow-1][col].setBorderBottom(outer_h);
		}

		for(int row = 0 ; row < nrow ; row++ ){
			cellPropertiesIndex[row][0].setBorderLeft(outer_v);
			cellPropertiesIndex[row][ncol-1].setBorderRight(outer_v);
		}
		
		for(int col = 0 ; col < ncol ; col++ ){
			if( rowSpanInstructions.containsKey( col ) ){
				if( rowSpanInstructions.get( col )[nrow-1] < 1 ){
					int line = -1;
					for( int row = nrow-1; row >= 0 ; row-- ){
						line = row;
						if( rowSpanInstructions.get( col )[row] > 0 ){
							break;
						}
					}
					cellPropertiesIndex[line][col].setBorderBottom(outer_h);
				}
			}
		}

		
	}
	

	public void setBodyBorderProperties(int[] i, int[] j, BorderProperties properties, String side ){
		
		int li = i.length;
		int lj = j.length;

		for(int row = 0 ; row < li ; row++ ){
			for( int col = 0 ; col < lj ; col++ ){

				if( side.equals("top")){
					if( Debug.debug ) System.err.println("setBodyBorderProperties side "+ side + " at :" + i[row] + " ; " + j[col] );

					cellPropertiesIndex[i[row]][j[col]].setBorderTop(properties);
					if( i[row] > 0 ) cellPropertiesIndex[i[row] - 1 ][j[col]].setBorderBottom(properties);
					else {
						if( headerSize() > 0 ){
							headerFlexRowList.set(headerSize() - 1, j[col], 
									properties, "bottom");;
						}
					}
					
				}
				else if( side.equals("right")){
					cellPropertiesIndex[i[row]][j[col]].setBorderRight(properties);
					if( j[col] < (ncol-2) ) cellPropertiesIndex[i[row]][j[col]+1].setBorderLeft(properties);
				}
				else if( side.equals("bottom")){
					cellPropertiesIndex[i[row]][j[col]].setBorderBottom(properties);
					if( i[row] < (nrow-1) ) {
						cellPropertiesIndex[i[row] + 1 ][j[col]].setBorderTop(properties);
					} else {
						if( footerSize() > 0 ){
							footerFlexRowList.set(0, j[col], 
									properties, "top");
						}
					}
				} else if( side.equals("left")){
					cellPropertiesIndex[i[row]][j[col]].setBorderLeft(properties);
					if( j[col] > 0 ) cellPropertiesIndex[i[row]][j[col]-1].setBorderRight(properties);
				}
			}
		}

	}

	public void setBackgroundColors(int[] i, int[] j, String[] colors ){

		int li = i.length;
		int lj = j.length;

		for(int row = 0 ; row < li ; row++ ){
			for( int col = 0 ; col < lj ; col++ ){
				CellProperties newone = cellPropertiesIndex[i[row]][j[col]].getClone();
				newone.setBackgroundColor(colors[row * lj + col]);
				cellPropertiesIndex[i[row]][j[col]] = newone;
			}
		}
	}
	

	public void setCellProperties(int[] i, int[] j, CellProperties cp ){

		int li = i.length;
		int lj = j.length;
		cp.setTextDirection("lrtb");

		for(int row = 0 ; row < li ; row++ ){
			for( int col = 0 ; col < lj ; col++ ){
				cellPropertiesIndex[i[row]][j[col]] = cp.getClone();
			}
		}
	}
	

	public void setParProperties(int[] i, int[] j, ParProperties pp ){
		int li = i.length;
		int lj = j.length;

		for(int row = 0 ; row < li ; row++ ){
			for( int col = 0 ; col < lj ; col++ ){
				cellTextValue[i[row]][j[col]].setParProperties(pp.getClone());
			}
		}
	}
	

	public void setTextProperties(int[] i, int[] j, TextProperties tp ){
		int li = i.length;
		int lj = j.length;
		for(int row = 0 ; row < li ; row++ ){
			for( int col = 0 ; col < lj ; col++ ){
				cellTextValue[i[row]][j[col]].setTextProperties(tp.getClone());
			}
		}
	}
	
	private int headerSize(){
		return headerFlexRowList.size();
	}
	private int footerSize(){
		return footerFlexRowList.size();
	}
	
	public String toString(){
		String out = "";
		for(int i = 0 ; i < nrow ; i++ ){
			out += "\n";
			for( int j = 0 ; j < ncol ; j++ ){
				out += "\t" + cellTextValue[i][j].toString();
			}
		}

		return out;
	}

	
	private String HeaderHTML( ) {
		//Grouped Headers
		String out = "";
		out += "<thead>";
		for( int i = 0 ; i < headerSize() ; i++ ){
			out += headerFlexRowList.get(i).getHTML();
		} 
		out += "</thead>";
		return out;
	}
	
	private String FooterHTML( ) {
		//Grouped Headers
		String out = "";
		out += "<tfoot>";
		for( int i = 0 ; i < footerSize() ; i++ ){
			out += footerFlexRowList.get(i).getHTML();
		} 
		out += "</tfoot>";
		return out;
	}
	
	private void sendMetaDocxReference(){
		if( headerSize() > 0 )
			for( int i = 0 ; i < headerSize() ; i++ ){
				headerFlexRowList.get(i).setDOCXMLPackage(docx_mlp);;
				headerFlexRowList.get(i).setOrderedNumID(ordered_num_id);
				headerFlexRowList.get(i).setUnorderedNumID(unordered_num_id);
			}	
		if( footerSize() > 0 )
			for( int i = 0 ; i < footerSize() ; i++ ){
				footerFlexRowList.get(i).setDOCXMLPackage(docx_mlp);;
				footerFlexRowList.get(i).setOrderedNumID(ordered_num_id);
				footerFlexRowList.get(i).setUnorderedNumID(unordered_num_id);
			}
	}
	private void sendMetaHTMLReference(){
		if( headerSize() > 0 )
			for( int i = 0 ; i < headerSize() ; i++ ){
				headerFlexRowList.get(i).setHTMLReference(list_id, counters, numbDefinition);
			}	
		if( footerSize() > 0 )
			for( int i = 0 ; i < footerSize() ; i++ ){
				footerFlexRowList.get(i).setHTMLReference(list_id, counters, numbDefinition);
			}
	}
	private void HeaderDOCX( Tbl reviewtable ) throws Exception{
		if( headerSize() > 0 )
		for( int i = 0 ; i < headerSize() ; i++ ){
			
			Tr workingRow = headerFlexRowList.get(i).getTr();
			TrPr trpr;
			if( workingRow.getTrPr() == null ){
				trpr = new TrPr();
				workingRow.setTrPr(trpr);
			} else trpr = workingRow.getTrPr();
			
			List<JAXBElement<?>> cnfStyleOrDivIdOrGridBefore = trpr.getCnfStyleOrDivIdOrGridBefore();
			cnfStyleOrDivIdOrGridBefore.add(Context.getWmlObjectFactory().createCTTrPrBaseTblHeader(Context.getWmlObjectFactory().createBooleanDefaultTrue()));

			reviewtable.getContent().add(workingRow);
		}
	}
	
	private void FooterDOCX( Tbl reviewtable ) throws Exception{
		if( footerSize() > 0 )
		for( int i = 0 ; i < footerSize() ; i++ ){
			Tr workingRow = footerFlexRowList.get(i).getTr();
			reviewtable.getContent().add(workingRow);
		}
	}
	
	private void HeaderPPTX( CTTable reviewtable ) throws Exception{
		if( headerSize() < 1 ) return;
		for( int i = 0 ; i < headerSize() ; i++ ){
			CTTableRow workingRow = headerFlexRowList.get(i).getCTTableRow(slidePart);
			reviewtable.getTr().add(workingRow);
		}
		
	}

	private void FooterPPTX( CTTable reviewtable ) throws Exception{
		if( footerSize() < 1 ) return;

		for( int i = 0 ; i < footerSize() ; i++ ){
			CTTableRow workingRow = footerFlexRowList.get(i).getCTTableRow(slidePart);
			reviewtable.getTr().add(workingRow);
		}
		
	}
	private void BodyDOCX( Tbl reviewtable ) throws Exception{
		for( int i = 0 ; i < nrow ; i++ ){
			Tr workingRow = new Tr();
			for( int j = 0 ; j < ncol ; j++ ){
				FlexCell fc = new FlexCell(cellTextValue[i][j], cellPropertiesIndex[i][j] );
				fc.setDOCXReference(ndp, ordered_num_id, unordered_num_id);
				fc.setDOCXMLPackage(getDOCXMLPackage());
				fc.getParagraphsSection().getParProperties().setOrderedNumid(ordered_num_id);
				fc.getParagraphsSection().getParProperties().setUnorderedNumid(unordered_num_id);
				
				if( rowSpanInstructions.containsKey( j ) ){
					fc.setRowspan(rowSpanInstructions.get( j )[i]);
				}
				if( colSpanInstructions.containsKey( i ) ){
					fc.setColspan(colSpanInstructions.get( i )[j]);
				}
				if( !colSpanInstructions.containsKey( i ) )
					workingRow.getContent().add(fc.getTc());
				else if( colSpanInstructions.containsKey( i ) && colSpanInstructions.get( i )[j] > 0)
					workingRow.getContent().add(fc.getTc()); 
			}
			reviewtable.getContent().add(workingRow);
		} 
	}
	
	private void BodyPPTX( CTTable reviewtable ) throws Exception{
		for( int i = 0 ; i < nrow ; i++ ){
			CTTableRow workingRow = new CTTableRow();
			for( int j = 0 ; j < ncol ; j++ ){
				FlexCell fc = new FlexCell(cellTextValue[i][j], cellPropertiesIndex[i][j] );
				fc.setPPTXSlidePart(slidePart);
				if( rowSpanInstructions.containsKey( j ) ){
					fc.setRowspan(rowSpanInstructions.get( j )[i]);
				}
				if( colSpanInstructions.containsKey( i ) ){
					fc.setColspan(colSpanInstructions.get( i )[j]);
				}

				workingRow.getTc().add(fc.getCTTableCell());
			}
			reviewtable.getTr().add(workingRow);
		} 
	}
	
	private String BodyHTML( ) {
		String out = "";
		out += "<tbody>";
		for( int i = 0 ; i < nrow ; i++ ){
			out += "<tr>";
			for( int j = 0 ; j < ncol ; j++ ){
				FlexCell fc = new FlexCell(cellTextValue[i][j], cellPropertiesIndex[i][j] );
				fc.setHTMLReference(list_id, counters, numbDefinition);
				if( rowSpanInstructions.containsKey( j ) ){
					fc.setRowspan(rowSpanInstructions.get( j )[i]);
				}
				if( colSpanInstructions.containsKey( i ) ){
					fc.setColspan(colSpanInstructions.get( i )[j]);
				}
				out +=  fc.getHTML();
			}
			out += "</tr>";
		} 
		out += "</tbody>";
		return out;
	}
	
	private CTTableGrid get_pptx_table_grid( long width){
		CTTableGrid tg = new CTTableGrid();
		for(int i = 0 ; i < ncol ; i++ ){
			List<CTTableCol> gc = tg.getGridCol();
			CTTableCol tc = new CTTableCol();
			if( widths[i] < 0 ) tc.setW(width);
			else tc.setW((long) (widths[i] * 914400) );
			gc.add( tc );
		}
		return tg;
	}
	
	private CTTblLayoutType getCTTblLayoutType() {
		CTTblLayoutType layoutType = new CTTblLayoutType();
		if( widths[0] > 0 ){
			layoutType.setType(STTblLayoutType.FIXED);
		} else {
			layoutType.setType(STTblLayoutType.AUTOFIT);
		}
		return layoutType;
	}
	private TblGrid get_docx_table_frid() {
		TblGrid tblGrid = new TblGrid();
		if( widths[0] > 0 ){			
			for (int i = 0 ; i < ncol; i++) {
				TblGridCol gridCol = Context.getWmlObjectFactory().createTblGridCol();
				gridCol.setW(BigInteger.valueOf( (long)(widths[i] * 1440 ) ));
				tblGrid.getGridCol().add(gridCol);
			}
		}
		return tblGrid;
	}
	
	private CTTable build_and_get_pptx_table(long width) throws Exception {
		CTTable newTable = new CTTable();
		CTTableProperties tablpro = new CTTableProperties();
		newTable.setTblPr(tablpro);
		newTable.setTblGrid(get_pptx_table_grid(width));
		
		if( nrow > 0 ){
			HeaderPPTX(newTable);
			BodyPPTX(newTable);
			FooterPPTX(newTable);
		}
		return newTable;
	} 
	
	public CTGraphicalObjectFrame get_pptx_elt( ) throws Exception{
		if( !validatePPTXComponent() ) throw new Exception("class FlexTable: elements need to be set...");

		org.docx4j.dml.ObjectFactory dmlFactory = new org.docx4j.dml.ObjectFactory();
		org.pptx4j.pml.ObjectFactory pmlFactory = new org.pptx4j.pml.ObjectFactory();
		CTGraphicalObjectFrame graphicFrame = pmlFactory.createCTGraphicalObjectFrame();
		
		org.pptx4j.pml.CTGraphicalObjectFrameNonVisual nvGraphicFramePr = pmlFactory.createCTGraphicalObjectFrameNonVisual();

		org.docx4j.dml.CTNonVisualDrawingProps cNvPr = dmlFactory.createCTNonVisualDrawingProps();

		org.docx4j.dml.CTNonVisualGraphicFrameProperties cNvGraphicFramePr = dmlFactory.createCTNonVisualGraphicFrameProperties();
		org.docx4j.dml.CTGraphicalObjectFrameLocking graphicFrameLocks = new org.docx4j.dml.CTGraphicalObjectFrameLocking();
		
//		org.docx4j.dml.CTTransform2D xfrm = dmlFactory.createCTTransform2D();
		Graphic graphic = dmlFactory.createGraphic();
		GraphicData graphicData = dmlFactory.createGraphicData();
		
		
		graphicFrame.setNvGraphicFramePr(nvGraphicFramePr);
		nvGraphicFramePr.setCNvPr(cNvPr);
		cNvPr.setName("nvGraphicFrame " + shape_id);
		cNvPr.setId(shape_id);

		nvGraphicFramePr.setCNvGraphicFramePr(cNvGraphicFramePr);
		cNvGraphicFramePr.setGraphicFrameLocks(graphicFrameLocks);
		graphicFrameLocks.setNoGrp(true);
		nvGraphicFramePr.setNvPr(pmlFactory.createNvPr());
		
		graphicFrame.setXfrm(getXfrm());
		
		graphicFrame.setGraphic(graphic);
		
		graphic.setGraphicData(graphicData);
		graphicData.setUri("http://schemas.openxmlformats.org/drawingml/2006/table");
		
		CTTable ctTable = build_and_get_pptx_table( new Double(table_width/ncol).longValue());
		JAXBElement<CTTable> tbl = dmlFactory.createTbl(ctTable);
		graphicData.getAny().add(tbl);

		return graphicFrame;
		}
	
	public boolean validateDOCXComponent(){
		if( unique_id < 0 ) {
			if( Debug.debug) System.err.println("unique_id is missing");
			return false;
		}
		if( docx_mlp == null ) {
			if( Debug.debug) System.err.println("docx_mlp is missing");
			return false;
		}
		if( ordered_num_id < 0 ) {
			if( Debug.debug) System.err.println("ordered_num_id uninitialized");
			return false;
		}
		return true;
	}
	public Tbl get_docx_elt() throws Exception{
		
		if( !validateDOCXComponent() ) throw new Exception("class FlexTable: elements need to be set...");
		sendMetaDocxReference();
		
		Tbl newTable = new Tbl();
		HeaderDOCX(newTable);
		BodyDOCX(newTable);
		FooterDOCX(newTable);
		TblPr tblpr = new TblPr();
		Jc alignment = new Jc();
		
		String textalign = getParProperties().getTextalign();
		
		if( textalign.equals("left")) alignment.setVal(JcEnumeration.LEFT);
		else if( textalign.equals("center")) alignment.setVal(JcEnumeration.CENTER);
		else if( textalign.equals("right")) alignment.setVal(JcEnumeration.RIGHT);
		else if( textalign.equals("justify")) alignment.setVal(JcEnumeration.BOTH);

		tblpr.setJc(alignment);
		tblpr.setTblLayout(getCTTblLayoutType());
		newTable.setTblGrid(get_docx_table_frid());
		newTable.setTblPr(tblpr);
		return newTable;
	}
	
	@Override
	public String getHTML() {
		sendMetaHTMLReference();
		String pStart = "";
		String align = "left";
		String pStop = "";	
		String tableCSS = " style=\"margin-left:0px;margin-right:auto;\"";
		try{
			LinkedHashMap<String, String> ppr = getParProperties().get_html_pp();
			pStart = "<p style=\"" + Format.getJSString(ppr) + "\">";
			align = getParProperties().getTextalign();
			if( align.equals("justify" )) align = "left";
			if( align.equals("left" ) ){
				tableCSS = " style=\"margin-left:0px;margin-right:auto;border-collapse:collapse;\"";
			} else if( align.equals("center" ) ){
				tableCSS = " style=\"margin-left:auto;margin-right:auto;border-collapse:collapse;\"";
			} else {
				tableCSS = " style=\"margin-left:auto;margin-right:0px;border-collapse:collapse;\"";
			}
			pStop = "</p>";
		} catch( Exception e){}
		
		String out = pStart + "<table" + tableCSS + ">";
		if( widths[0] > 0 ){
			for( int j = 0 ; j < ncol ; j++ ){
				out += "<col width=\"" + (int)(72.2*widths[j]) + "\">";
			}
		}
		out += HeaderHTML( );
		out += BodyHTML( );
		out += FooterHTML( );
		out += "</table>" + pStop;
		return out;
	}

	
	@Override
	public String getCSS() {
		return "";
	}

	@Override
	public String getJS() {
		return "";
	}

	@Override
	public boolean hasJS() {
		return false;
	}

	@Override
	public boolean hasHTML() {
		return true;
	}

	@Override
	public boolean hasCSS() {
		return false;
	}
	
}
