## SimInf, a framework for stochastic disease spread simulations
## Copyright (C) 2015 - 2017  Stefan Engblom
## Copyright (C) 2015 - 2017  Stefan Widgren
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.

##' Class \code{"SimInf_mparse"}
##'
##' Class to handle the SimInf mparse data
##' @section Slots:
##' \describe{
##'   \item{latex}{
##'     Character vector with the model LaTeX code.
##'   }
##'   \item{C_code}{
##'     Character vector with the model C code.
##'   }
##'   \item{G}{
##'     Dependency graph that indicates the transition rates that need
##'     to be updated after a given state transition has occured.  A
##'     non-zero entry in element \code{G[i, i]} indicates that
##'     transition rate \code{i} needs to be recalculated if the state
##'     transition \code{j} occurs. Sparse matrix (\eqn{Nt \times Nt})
##'     of object class \code{"\linkS4class{dgCMatrix}"}, where
##'     \eqn{Nt} is the number of transitions.
##'   }
##'   \item{S}{
##'     Each column corresponds to a state transition, and execution
##'     of state transition \code{j} amounts to adding the \code{S[,
##'     j]} column to the state vector \code{u[, i]} of node \emph{i}
##'     where the transition occurred. Sparse matrix (\eqn{Nc \times
##'     Nt}) of object class \code{"\linkS4class{dgCMatrix}"}, where
##'     \eqn{Nc} is the number of transitions and \eqn{Nt} is the
##'     number of transitions.
##'   }
##' }
##' @keywords methods
##' @export
setClass("SimInf_mparse",
         slots = c(latex  = "character",
                   C_code = "character",
                   G      = "dgCMatrix",
                   S      = "dgCMatrix"),
         validity = function(object) {
             errors <- character()

             ## Check C code
             if (nchar(paste0(object@C_code, collapse = "\n")) == 0L) {
                 errors <- c(errors, "'C_code' is empty.")
             }

             ## Check S.
             if (identical(dim(object@S), c(0L, 0L))) {
                 errors <- c(errors, "'S' is empty.")
             } else if (!all(is_wholenumber(object@S@x))) {
                 errors <- c(errors,
                             "'S' matrix must be an integer matrix.")
             }

             ## Check G.
             Nt <- dim(object@S)[2]
             if (!identical(dim(object@G), c(Nt, Nt))) {
                 errors <- c(errors,
                             "Wrong size of dependency graph 'G'.")
             }

             if (length(errors) == 0) TRUE else errors
         }
)

## C code: heading
C_heading <- function() {
    c(sprintf("/* Generated by SimInf (v%s) %s */",
            utils::packageVersion("SimInf"),
            format(Sys.time(), "%Y-%m-%d %H:%M")),
      "")
}

## C code: include directives
C_include <- function() {
    c("#include <R_ext/Rdynload.h>",
      "#include \"SimInf.h\"",
      "")
}

## C code: compartments
C_compartments <- function(compartments) {
    lines <- character(0)
    if (length(compartments)) {
        lines <- c("/* Compartments */",
                   paste0("enum{",
                          paste0(compartments, collapse = ", "),
                          "};"),
                   "")
    }
    lines
}

## C code: rate constants
C_rate_constants <- function(rates) {
    lines <- character(0)
    if (length(rates)) {
        lines <- "/* Rate constants */"
        for (i in seq_len(length(rates))) {
            lines <- c(lines, sprintf("const double %s = %g;",
                                      names(rates)[i],
                                      rates[[i]]))
        }
        lines <- c(lines, "")
    }
    lines
}

## C code: transition rate functions.
C_trFun <- function(transitions) {
    lines <- character(0)
    for (i in seq_len(length(transitions))) {
        lines <- c(lines,
                   sprintf("double trFun%i(", i),
                   "    const int *u,",
                   "    const double *v,",
                   "    const double *ldata,",
                   "    const double *gdata,",
                   "    double t)",
                   "{",
                   sprintf("    return %s;", transitions[[i]]$propensity),
                   "}",
                   "")
    }
    lines
}

## C code: post time step function.
C_ptsFun <- function() {
    c("int ptsFun(",
      "    double *v_new,",
      "    const int *u,",
      "    const double *v,",
      "    const double *ldata,",
      "    const double *gdata,",
      "    int node,",
      "    double t)",
      "{",
      "    return 0;",
      "}",
      "")
}

## C code: SimInf model run function.
C_run <- function(transitions) {
    c("SEXP SimInf_model_run(SEXP model, SEXP threads, SEXP seed)",
      "{",
      sprintf("    TRFun tr_fun[] = {%s};",
              paste0("&trFun", seq_len(length(transitions)), collapse = ", ")),
      "    DL_FUNC SimInf_run = R_GetCCallable(\"SimInf\", \"SimInf_run\");",
      "    return SimInf_run(model, threads, seed, tr_fun, &ptsFun);",
      "}",
      "")
}

## C code: Generate C code for mparse.
C_code_mparse <- function(transitions, rates, compartments) {
    c(C_heading(),
      C_include(),
      C_compartments(compartments),
      C_rate_constants(rates),
      C_trFun(transitions),
      C_ptsFun(),
      C_run(transitions))
}

## Split the propensity in order to separate preprocessor and
## punctuator tokens from identifiers, for example:
##
## > tokens(" bR * R ")
## [1] "bR" "*"  "R"
tokens <- function(propensity) {
    ## List of valid preprocessor operator or punctuator tokens.
    operators <- c("...", "<<=", ">>=", "!=", "%=", "##", "&&", "&=", "*=",
                   "++", "+=", "--", "-=", "->", "/=", "<<", "<=", "==",
                   ">=", ">>", "^=", "|=", "||", "!", "~", "%", "&", "(",
                   ")", "*", "+", ",", "-", ".", "/", ":", ";", "<", "=",
                   ">", "?", "[", "]", "^", "{", "|", "}", "#")

    ## Create a matrix (1 x 2) of the propensity, where the first
    ## column is the token and the second column indicates if the
    ## token is one of the operators (indicated with 'op').
    propensity <- cbind(token = propensity, type = "")

    ## Iterate over each operator and try to split each row in the
    ## propensity in smaller pieces.
    for (op in operators) {
        propensity <- lapply(seq_len(nrow(propensity)), function(i) {
            x <- propensity[i, , drop = FALSE]

            ## Is it a non-operator token that we could split?
            if (nchar(x[1, 2]) == 0) {
                m <- gregexpr(op, x[1, 1], fixed = TRUE)[[1]]
                if (m[1] != -1) {
                    ## The operator exists in the token. Split the
                    ## token in smaller pieces. The cut-points are
                    ## deterimined by the position and length of op
                    ## e.g. "A op B" -> "A", "op", "B".
                    x <- as.character(x[1, 1])
                    j <- 1
                    xx <- NULL
                    for (i in seq_len(length(m))) {
                        if (m[i] > j)
                            xx <- c(xx, substr(x, j, m[i] - 1))
                        j <- m[i] + attr(m, "match.length")[i]
                        xx <- c(xx, substr(x, m[i], j - 1))
                    }

                    ## Make sure last sub-string is copied.
                    if (j <= nchar(x))
                        xx <- c(xx, substr(x, j, nchar(x)))

                    ## Remove leading and trailing whitespace and drop
                    ## empty strings
                    xx <- gsub("(^\\s+)|(\\s+$)", "", xx)
                    xx <- xx[nchar(xx) > 0]

                    ## Create a 2-column matrix from all sub-strings
                    x <- cbind(token = xx, type = ifelse(xx == op, "op", ""))
                }
            }

            x
        })

        propensity <- do.call("rbind", propensity)
    }

    propensity[, 1]
}

## Rewrite propensity
##
## Rewrite the propensity by replacing all compartments by
## \code{u[compartments[j]]} where \code{j} is the numbering in
## compartments. On return, 'depends' contains all compartments upon
## which the propensity depends.
rewriteprop <- function(propensity, compartments) {
    orig_prop <- propensity
    propensity <- tokens(propensity)
    depends <- integer(length(compartments))
    i <- match(propensity, compartments)
    propensity <- ifelse(is.na(i),
                         propensity,
                         sprintf("u[%s]", compartments[i]))
    propensity <- paste0(propensity, collapse = "")
    i <- i[!is.na(i)]
    if (length(i))
        depends[i] <- 1

    list(orig_prop  = orig_prop,
         propensity = propensity,
         depends = depends)
}

## Generate labels from the parsed transitions
as_labels <- function(transitions) {
    sapply(transitions, function(x) {
        if (length(x$from)) {
            from <- paste0(x$from, collapse = " + ")
        } else {
            from <- "@"
        }

        if (length(x$dest)) {
            dest <- paste0(x$dest, collapse = " + ")
        } else {
            dest <- "@"
        }

        paste(from, "->", dest)
    })
}

## Generate LaTeX code from the transitions.
LaTeX <- function(transitions)
{
    lbl <- as_labels(transitions)
    lbl <- gsub("@", "\\\\emptyset", lbl)
    propensity <- sapply(transitions, function(x) x$orig_prop)
    lines <- c("\\begin{align}",
               "  \\left",
               "    \\begin{array}{rcl}")
    for (i in seq_len(length(lbl))) {
        tmp <- unlist(strsplit(lbl[i], "->"))
        lines <- c(lines, paste0("      ",
                                 tmp[1],
                                 "& \\xrightarrow{",
                                 propensity[i],
                                 "} &",
                                 tmp[2],
                                 " \\\\"))
    }

    c(lines,
      "    \\end{array}",
      "  \\right\\}",
      "\\end{align}")
}

##' Model parser
##'
##' Describe your model in a logical way in R. \code{mparse} creates a
##' \code{\linkS4class{SimInf_mparse}} object with your model
##' definition that is ready to be initialised with data and then
##' \code{\link{run}}.
##' @param transitions character vector containing transitions on the
##'     form \code{"X -> ... -> Y"}. The left (right) side is the
##'     initial (final) state and the propensity is written in between
##'     the \code{->}-signs. The special symbol \code{@} is reserved
##'     for the empty set. For example, \code{transitions =
##'     c("S -> k1*S*I -> I", "I -> k2*I -> R")} expresses a SIR
##'     model.
##' @param compartments contains the names of the involved
##'     compartments, for example, \code{compartments = c("S", "I",
##'     "R")}.
##' @param ... rate-constants for the model.
##' @return \linkS4class{SimInf_mparse}
##' @export
##' @template mparse-example
mparse <- function(transitions = NULL, compartments = NULL, ...)
{
    rates <- list(...)
    stopifnot(all(sapply(rates, class) == "numeric"))

    if (is.null(transitions))
        stop("'transitions' must be specified.")
    if(!is.character(transitions))
        stop("'transitions' must be specified in a character vector.")
    if (is.null(compartments))
        stop("'compartments' must be specified.")
    if(!is.character(compartments))
        stop("'compartments' must be specified in a character vector.")
    if (!all(identical(length(compartments), length(unique(compartments))),
             identical(length(names(rates)), length(unique(names(rates))))))
        stop("'compartments' and 'rates' must consist of unique names.")

    reserved = c("v_new", "u", "v", "ldata", "gdata", "node", "t", "rng")
    if (length(intersect(compartments, reserved)))
        stop(paste("Invalid compartment names:",
                   paste0(intersect(compartments, reserved), collapse = ", ")))
    if (length(intersect(names(rates), reserved)))
        stop(paste("Invalid rate names:",
                   paste0(intersect(names(rates), reserved), collapse = ", ")))

    transitions <- lapply(strsplit(transitions, "->"), function(x) {
        if (!identical(length(x), 3L))
            stop(paste0("Invalid transition: '", paste0(x, collapse = "->"), "'"))

        ## Remove spaces and the empty set
        from <- gsub(" ", "", gsub("@", "", x[1]))
        propensity <- gsub(" ", "", x[2])
        dest <- gsub(" ", "", gsub("@", "", x[3]))

        ## Split from and dest into 'compartment1 + compartment2 + ..'
        from <- unlist(strsplit(from, "+", fixed = TRUE))
        dest <- unlist(strsplit(dest, "+", fixed = TRUE))

        ## Assign each compartment into its number according to the
        ## ordering in compartments
        ifrom <- match(from, compartments)
        if (any(is.na(ifrom)))
            stop(sprintf("Unknown compartment: '%s'.", from[is.na(ifrom)]))
        idest <- match(dest, compartments)
        if (any(is.na(idest)))
            stop(sprintf("Unknown compartment: '%s'.", dest[is.na(idest)]))

        ## The corresponding column in the state change vector S is
        ## now known.
        S <- integer(length(compartments))
        S[ifrom] <- -1
        S[idest] <- 1

        propensity <- rewriteprop(propensity, compartments)

        list(from       = from,
             dest       = dest,
             orig_prop  = propensity$orig_prop,
             propensity = propensity$propensity,
             depends    = propensity$depends,
             S          = S)
    })

    S <- methods::as(do.call("cbind", lapply(transitions, function(x) x$S)), "dgCMatrix")
    depends <- do.call("rbind", lapply(transitions, function(x) x$depends))
    G <- methods::as(((depends %*% abs(S)) > 0) * 1, "dgCMatrix")

    colnames(G) <- as.character(seq_len(dim(G)[2]))
    rownames(G) <- as_labels(transitions)
    colnames(S) <- as.character(seq_len(dim(S)[2]))
    rownames(S) <- compartments

    methods::new("SimInf_mparse", latex = LaTeX(transitions),
                 C_code = C_code_mparse(transitions, rates, compartments),
                 G = G, S = S)
}

##' @rdname init-methods
##' @export
setMethod("init",
          signature(model = "SimInf_mparse"),
          function(model, u0, tspan, events, E, N)
          {
              compartments <- rownames(model@S)

              ## Check u0
              if (!is.data.frame(u0))
                  u0 <- as.data.frame(u0)
              if (!all(compartments %in% names(u0)))
                  stop("Missing columns in u0")
              u0 <- u0[, compartments, drop = FALSE]

              if (is.null(E))
                  E <- methods::as(matrix(integer(0),
                                          nrow = 0,
                                          ncol = 0),
                                   "dgCMatrix")

              if (is.null(N))
                  N <- matrix(integer(0), nrow = 0, ncol = 0)

              v0 <- matrix(numeric(0), nrow  = 0, ncol = nrow(u0))
              storage.mode(v0) <- "double"

              ldata <- matrix(numeric(0), nrow = 0, ncol = nrow(u0))
              storage.mode(ldata) <- "double"

              SimInf_model(G      = model@G,
                           S      = model@S,
                           E      = E,
                           N      = N,
                           tspan  = tspan,
                           events = events,
                           ldata  = ldata,
                           gdata  = numeric(),
                           u0     = u0,
                           v0     = v0,
                           C_code = model@C_code)
          }
)

##' @rdname C_code-methods
##' @export
setMethod("C_code",
          signature(model = "SimInf_mparse", pkg = "missing"),
          function(model)
          {
              model@C_code
          }
)

##' @rdname C_code-methods
##' @export
setMethod("C_code",
          signature(model = "SimInf_mparse", pkg = "character"),
          function(model, pkg)
          {
              stopifnot(identical(length(pkg), 1L), nchar(pkg[1]) > 0)

              lines <- model@C_code

              lines <- c(
                  lines[1:2],
                  "#include <R_ext/Visibility.h>",
                  lines[-(1:2)],
                  "static const R_CallMethodDef callMethods[] =",
                  "{",
                  "    {\"SimInf_model_run\", (DL_FUNC)&SimInf_model_run, 3},",
                  "    {NULL, NULL, 0}",
                  "};",
                  "",
                  paste0("void attribute_visible R_init_", pkg, "(DllInfo *info)"),
                  "{",
                  "    R_registerRoutines(info, NULL, callMethods, NULL, NULL);",
                  "    R_useDynamicSymbols(info, FALSE);",
                  "    R_forceSymbols(info, TRUE);",
                  "}",
                  "")

              lines
          }
)
