% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/idm.R
\name{idm}
\alias{idm}
\title{Fit an illness-death model}
\usage{
idm(formula01, formula02, formula12, data, maxiter = 200, eps = c(5, 5, 3),
  n.knots = c(7, 7, 7), knots = "equidistant", CV = FALSE,
  kappa = c(1000000, 500000, 20000), method = "Weib", conf.int = 0.95,
  print.iter = FALSE, subset = NULL, na.action = na.fail)
}
\arguments{
\item{formula01}{A formula specifying a regression model for the
\code{0 --> 1} transition from the initial state to the transient
state of the illness-death model.  The right hand side of the
formula specifies the covariate terms, and the left hand side must
be an event history object as returned by the function \code{Hist}.}

\item{formula02}{A formula specifying a regression model for the
\code{0 --> 2} transition from the initial state to the absorbing
state. The left hand side must be equal to the left hand side of
\code{formula01}. If missing it is set to \code{formula01}.}

\item{formula12}{A formula specifying a regression model for the
\code{1 --> 2} transition from the transient state to the absorbing
state.  operator is not required. If missing it is set to
\code{formula01}.}

\item{data}{A data frame in which to interpret the variables of
\code{formula01}, \code{formula02} and \code{formula12}.}

\item{maxiter}{Maximum number of iterations. The default is 200.}

\item{eps}{A vector of 3 integers >0 used to define the power of
three convergence criteria: 1. for the regression parameters,
2. for the likelihood, 3. for the second derivatives. The default
is \code{c(5,5,3)} which is translated into convergence if the
respective values change less then \eqn{10^{-5}} (for regression
parameters and likelihood) and \eqn{10^{-3}} for the second
derivatives between two iterations.}

\item{n.knots}{For \code{method="Splines"} only, a vector of length
3 specifing the number of knots, one for each transition, for the
M-splines estimate of the baseline intensities in the order \code{0
--> 1}, \code{0 --> 2}, \code{1 --> 2}. The default is c(7,7,7). When \code{knots}
are specified as a list this argument is ignored.
The algorithm needs least 5 knots and at most 20 knots.}

\item{knots}{Argument only active for the penalized likelihood approach \code{method="Splines"}.
There are three ways to control the placement of the knots between the smallest and the largest
of all time points:
\itemize{
 \item{\code{knots="equidistant"}}{Knots are placed with same distance on the time scale.}
 \item{\code{knots="quantiles"}}{Knots are placed such that the number of observations is roughly the same between knots.}
\item{knots=list()}{List of 1 or 2 or three vectors. The list elements are the actual placements
(timepoints) of the knots for the M-spline. The list may contain 
one vector of placements for each transition in the order \code{0 --> 1}, \code{0 --> 2}, \code{1 --> 2}.
If only vector is specifified the knots are used for all transitions. If only 2 vectors are specifified, the
knots for the \code{0 --> 1} transition are also used for the \code{1 --> 2} transition.}
}
The algorithm needs at least 5 knots and allows no more than 20 knots.}

\item{CV}{Binary variable equals to 1 when search (by approximated
cross validation) of the smoothing parameters \code{kappa} and 0
otherwise. Argument for the penalized likelihood approach. The
default is 0.}

\item{kappa}{Argument only active for the penalized likelihood approach \code{method="Splines"}.
A vector with 3 positive values (smoothing parameters), one for each transition, in the order 
0 --> 1, 0 --> 2 and 1 --> 2..
If CV=1 these are used as starting values for a cross validation search to optimize kappa.}

\item{method}{type of estimation method: "Splines" for a
penalized likelihood approach with approximation of the transition
intensities by M-splines, "Weib" for a parametric approach with a
Weibull distribution on the transition intensities. Default is
"Weib".}

\item{conf.int}{Level of confidence pointwise confidence intervals of the transition intensities, i.e.,
a value between 0 and 1, the default is \code{0.95}.
The default is also used when \code{conf.int=TRUE}.
To avoid computation of confidence intervals, set \code{conf.int} to FALSE or NULL.}

\item{print.iter}{boolean parameter. Equals to \code{TRUE} to print
the likelihood during the iteration process, \code{FALSE}
otherwise. Default is \code{FALSE}. This option is not running on
Windows.}

\item{subset}{expression indicating the subset of the rows of data
to be used in the fit. All observations are included by default.}

\item{na.action}{how NAs are treated. The default is first, any
na.action attribute of data, second a na.action setting of options,
and third 'na.fail' if that is unset. The 'factory-fresh' default
is na.omit. Another possible value is NULL.}
}
\value{
\item{call}{the call that produced the result.} \item{coef}{regression
parameters.} \item{loglik}{vector containing the log-likelihood without and
with covariate.} \item{cv}{vector containing the convergence criteria.}
\item{niter}{number of iterations.} \item{converged}{integer equal to 1 when
the model converged, 2, 3 or 4 otherwise.} \item{modelPar}{Weibull
parameters.} \item{N}{number of subjects.} \item{events1}{number of events 0
--> 1.} \item{events2}{number of events 0 --> 2 or 0 --> 1 --> 2.}
\item{NC}{vector containing the number of covariates on transitions 0 --> 1,
0 --> 2, 1 --> 2.} \item{responseTrans}{model response for the 0 --> 1
transition. \code{Hist} or \code{Surv} object.} \item{responseAbs}{model
response for the 0 --> 2 transition. \code{Hist} or \code{Surv} object.}
\item{time}{times for which transition intensities have been evaluated for
plotting. Vector in the Weibull approach. Matrix in the penalized likelihhod
approach for which the colums corresponds to the transitions 0 --> 1, 1 -->
2, 0 --> 2.} \item{intensity01}{matched values of the intensities for
transition 0 --> 1.} \item{lowerIntensity01}{lower confidence intervals for
the values of the intensities for transition 0 --> 1.}
\item{upperIntensity01}{upper confidence intervals for the values of the
intensities for transition 0 --> 1.} \item{intensity02}{matched values of
the intensities for transition 0 --> 2.} \item{lowerIntensity02}{lower
confidence intervals for the values of the intensities for transition 0 -->
2.} \item{upperIntensity02}{upper confidence intervals for the values of the
intensities for transition 0 --> 2.} \item{intensity12}{matched values of
the intensities for transition 1 --> 2.} \item{lowerIntensity12}{lower
confidence intervals for the values of the intensities for transition 1 -->
2.} \item{upperIntensity12}{upper confidence intervals for the values of the
intensities for transition 1 --> 2.} \item{RR}{vector of relative risks.}
\item{V}{variance-covariance matrix derived from the Hessian of the log-likelihood 
if using method="Weib" or, from the Hessian of the penalized log-likelihood 
if using method="Splines".} 
\item{se}{standart errors of the
regression parameters.} \item{Xnames01}{names of covariates on 0 --> 1.}
\item{Xnames02}{names of covariates on 0 --> 2.} \item{Xnames12}{names of
covariates on 1 --> 2.} \item{knots01}{knots to approximate by M-splines the
intensity of the 0 --> 1 transition.} \item{knots02}{knots to approximate by
M-splines the intensity of the 0 --> 2 transition.} \item{knots12}{knots to
approximate by M-splines the intensity of the 1 --> 2 transition.}
\item{nknots01}{number of knots on transition 0 --> 1.}
\item{nknots02}{number of knots on transition 0 --> 2.}
\item{nknots12}{number of knots on transition 1 --> 2.}
\item{theta01}{square root of splines coefficients for transition 0 --> 1.}
\item{theta02}{square root of splines coefficients for transition 0 --> 2.}
\item{theta12}{square root of splines coefficients for transition 1 --> 2.}
\item{CV}{a binary variable equals to 1 when search of the smoothing
parameters \link{kappa} by approximated cross-validation, 1 otherwise. The
default is 0.} \item{kappa}{vector containing the smoothing parameters for
transition 0 --> 1, 0 --> 2, 1 --> 2 used to estimate the model by the
penalized likelihood approach.} \item{CVcrit}{cross validation criteria.}
\item{DoF}{degrees of freedom of the model.} \item{na.action}{observations
deleted if missing values.}
}
\description{
Fit an illness-death model using either a semi-parametric approach
(penalized likelihood with an approximation of the transition intensity
functions by linear combination of M-splines) or a parametric approach
(specifying Weibull distributions on the transition intensities).
Left-truncated, right-censored, and interval-censored data are allowed.
State 0 corresponds to the initial state, state 1 to the transient one,
state 2 to the absorbant one. The allowed transitions are: 0 --> 1, 0 --> 2
and 1 --> 2.
}
\details{
The estimated parameters are obtained using the robust Marquardt algorithm
(Marquardt, 1963) which is a combination between a Newton-Raphson algorithm
and a steepest descent algorithm.
}
\examples{
library(lava)
library(prodlim)
set.seed(17)
d <- simulateIDM(100)
# right censored data
fitRC <- idm(formula01=Hist(time=observed.illtime,event=seen.ill)~X1+X2,
             formula02=Hist(time=observed.lifetime,event=seen.exit)~X1+X2,
             formula12=Hist(time=observed.lifetime,event=seen.exit)~X1+X2,data=d,
             conf.int=FALSE)
fitRC

\dontrun{
set.seed(17)
d <- simulateIDM(300)
fitRC.splines <- idm(formula01=Hist(time=observed.illtime,event=seen.ill)~X1+X2,
             formula02=Hist(time=observed.lifetime,event=seen.exit)~X1+X2,
             formula12=Hist(time=observed.lifetime,event=seen.exit)~1,data=d,
             conf.int=FALSE,method="splines")
}
# interval censored data
fitIC <- idm(formula01=Hist(time=list(L,R),event=seen.ill)~X1+X2,
             formula02=Hist(time=observed.lifetime,event=seen.exit)~X1+X2,
             formula12=Hist(time=observed.lifetime,event=seen.exit)~X1+X2,data=d,
             conf.int=FALSE)
fitIC

\dontrun{

    data(Paq1000)

    # Illness-death model with certif on the 3 transitions
    # Weibull parametrization and likelihood maximization

    fit.weib <- idm(formula02=Hist(time=t,event=death,entry=e)~certif,
                    formula01=Hist(time=list(l,r),event=dementia)~certif,
                    data=Paq1000)

    # Illness-death model with certif on transitions 01 and 02
    # Splines parametrization and penalized likelihood maximization
    fit.splines <-  idm(formula02=Hist(time=t,event=death,entry=e)~certif,
                        formula01=Hist(time=list(l,r),event=dementia)~certif,
                        formula12=~1,
                        method="Splines",
                        data=Paq1000)
    fit.weib
    summary(fit.splines)
}

}
\references{
D. Marquardt (1963). An algorithm for least-squares estimation
of nonlinear parameters.  \emph{SIAM Journal of Applied Mathematics},
431-441.
}
\seealso{
\code{\link{print.idm}}
\code{\link{summary.idm}}
\code{\link{predict.idm}}
}
\author{
R: Celia Touraine <Celia.Touraine@isped.u-bordeaux2.fr> Fortran:
Pierre Joly <Pierre.Joly@isped.u-bordeaux2.fr>
}
\keyword{illness-death}
