% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spread-process.R
\name{directionFromEachPoint}
\alias{.pointDirection}
\alias{directionFromEachPoint}
\title{Calculate distances and directions between many points and many grid cells}
\usage{
directionFromEachPoint(from, to = NULL, landscape)

.pointDirection(from, to)
}
\arguments{
\item{from}{matrix with 2 or 3 columns, x and y, representing x and y coordinates of "from" cell,
and optional "id" which will be returned, and if "id" column is in \code{to},
it will be matched with that.}

\item{to}{matrix with 2  or 3 columns (or optionally more, all of which will be returned),
x and y, representing x and y coordinates of "to" cells, and
optional "id" which will be matched with "id" from \code{from}. It makes no sense to
have "id" column here with no "id" column in \code{from}}

\item{landscape}{RasterLayer. optional. This is only used if \code{to} is NULL, in which case
all cells are considered \code{to}}
}
\value{
A sorted matrix on \code{id} with same number of rows as \code{to},
        but with one extra column, \code{angles}
        indicating the angle in radians between from and to. For speed, this
        angle will be between -pi/2 and 3*pi/2. If the user wants this between
        say, 0 and 2*pi, then \code{angles \%\% (2*pi)} will do the trick. See example.
}
\description{
This is a modification of \code{\link[raster]{distanceFromPoints}} for the case of many points.
This version can often be faster for a single point because it does not return a RasterLayer. This is
different than \code{\link[raster]{distanceFromPoints}} because it does not take the minimum
distance from the set of points to all cells. Rather this returns the every pair-wise point distance.
As a result, this can be used for doing inverse distance weightings, seed rain, cumulative effects
of distance-based processes etc. If memory limitation is an issue, maxDistance will keep memory
use down, but with the consequences that there will be a maximum distance returned. This function
has the potential to use a lot of memory if there are a lot of \code{from} and \code{to} points.

This is meant to be used internally.
}
\details{
\code{directionFromEachPoint} calls \code{.pointDirection}, which is not intended to be called
directly by the user.

If knowing the which from cell matches with which to cell is important,
put a column "id" (e.g., starting cell) in the \code{from} matrix.
}
\examples{
library(raster)
N <- 2
dirRas <- raster(extent(0,40,0,40), res = 1)
coords <- cbind(x = round(runif(N, xmin(dirRas), xmax(dirRas)))+0.5,
                y = round(runif(N, xmin(dirRas), xmax(dirRas)))+0.5,
                id = 1:N)

dirs1 <- directionFromEachPoint(from = coords, landscape = dirRas)
require(CircStats)
dirs1[, "angles"] <- deg(dirs1[,"angles"] \%\% (2*pi))
indices <- cellFromXY(dirRas,dirs1[, c("x", "y")])
minDir <- tapply(dirs1[,"angles"], indices, function(x) min(x)) # minimum angle
dirRas[] <- as.vector(minDir)
if (interactive()) {
  Plot(dirRas, new = TRUE)
  library(sp)
  start <- SpatialPoints(coords[, c("x", "y"), drop = FALSE])
  Plot(start, addTo = "dirRas")
}
}
\seealso{
\code{\link{distanceFromEachPoint}}, which will also return directions if \code{angles}
is TRUE.
}

