% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/Predict.bagging.R
\name{Predict.bagging}
\alias{Predict.bagging}
\title{Make predictions for new data from a 'bagging' object.}
\description{
This function makes predictions for new data from a bagging LASSO linear or logistic regression model, using the 
stored 'bagging' object, with or without the use of trimmed bagging strategy.
}
\usage{
Predict.bagging(object, newx, y = NULL, trimmed = FALSE, scale.trimmed = 0.75)
}
\arguments{
\item{object}{a fitted 'bagging' object.}

\item{newx}{matrix of new values for x at which predictions are to be made. Must be a 
matrix. See documentation for Bagging.lasso.}

\item{y}{response variable. Defaults to NULL. If the response variable for the newx 
matrix is known and input, the corresponding validation measures can be calculated 
for evaluating prediction performance.}

\item{trimmed}{logical. Should a trimmed bagging strategy be performed? Defaults to 
FALSE. This argument should correspond to the same setting in the Bagging.lasso function. 
See documentation for Bagging.lasso.}

\item{scale.trimmed}{the portion to trim of the "worst" based-level models, in the sense 
of having the largest error rates, and to average only over the most accurate base-level 
models. Defaults to 0.75.}
}
\details{
This function makes a prediction based on the object fitted by the Bagging.lasso model.
}
\value{
\item{y.new}{the predicted values of response vector y.}

\item{probabilities}{the predicted probabilities of response vector y.}

\item{predicted.matrix}{the matrix of predicted values of response vector y based on the 
base-level LASSO regression models.}

\item{bagging.prediction}{the performance of bagging prediction accordig to the model 
validation measures defined.}
}
\references{
[1] Breiman, L. (2001). Random Forests. Machine Learning, 45(1), 5-32.

[2] Croux, C., Joossens, K., & Lemmens, A. (2007). Trimmed bagging. Computational Statistics &
Data Analysis, 52(1), 362-368.
}
\examples{
library(mlbench)
set.seed(0123)
mydata <- mlbench.threenorm(100, d=10)
x <- mydata$x
y <- mydata$classes
mydata <- as.data.frame(cbind(x, y))
colnames(mydata) <- c(paste("A", 1:10, sep=""), "y")
mydata$y <- ifelse(mydata$y==1, 0, 1)
# Split into training and testing data.
S1 <- as.vector(which(mydata$y==0))
S2 <- as.vector(which(mydata$y==1))
S3 <- sample(S1, ceiling(length(S1)*0.8), replace=FALSE)
S4 <- sample(S2, ceiling(length(S2)*0.8), replace=FALSE)
TrainInd <- c(S3, S4)
TestInd <- setdiff(1:length(mydata$y), TrainInd)
TrainXY <- mydata[TrainInd, ]
TestXY <- mydata[TestInd, ]
# Fit a bagging LASSO linear regression model, where the parameters
# of M in the following example is set as small values to reduce the
# running time, however the default value is proposed.
Bagging.fit <- Bagging.lasso(x=TrainXY[, -10], y=TrainXY[, 10],
family=c("gaussian"), M=2, predictor.subset=round((9/10)*ncol(x)),
predictor.importance=TRUE, trimmed=FALSE, weighted=TRUE, seed=0123)
Bagging.fit
# Make predictions from a bagging LASSO linear regression model.
pred <- Predict.bagging(Bagging.fit, newx=TestXY[, -10], y=NULL)
pred
}
