#'  Dynamic Spillover Index
#'
#' 
#' @description 
#' Estimates the dynamic spillover index given a rolling window as described in Diebold and Yilmaz (2012).
#' 
#' @param data Object of class \sQuote{\code{zoo}}.
#' @param width An integer specifying the window width which is aligned to the original sample.
#' @param n.ahead An integer indicating the how many steps ahead the spillover should be forecasted.
#' @param index A character string indicating whether the orthogonalized or the generalized index is computed.
#' @param ortho.type Applicable only if \code{index="orthogonalized"}.  A character string indicating the type of orthogonalized index is required. \code{"single"} takes the original ordering of variables in VAR model and applies Cholesky decomposition for the fevd. Whereas \code{"partial"} takes a random sample out of all the possible combinations generated for the Cholesky decomposition, while \code{"total"} uses all the combinations, therefore it takes more time to finish.
#' Both, \code{"partial"} and \code{"total"} provide average results.
#' @param ... Further arguments to be passed to \code{var} function.
#' 
#' @return A \code{zoo} object holding all the indeces.
#' 
#' 
#' 
#' @references
#' Diebold, F. X. & Yilmaz, K.(2012). \emph{Better to Give than to Receive: Predictive Directional Measurement of Volatility Spillovers}. International Journal of Forecasting.
#' 
#' @author Jilber Urbina
#' @export
#' @import fastSOM
#'
#' @keywords htest ts
#' 
#' @examples 
#' data(dy2012)
#' O_index <- roll.spillover(as.zoo(dy2012[1:300,c(2,3,4)]), width = 200, p=4)
#'  \donttest{
#' # Orthogonalized rolling spillover index based on a VAR(4), single order
#'  O_index <- roll.spillover(as.zoo(dy2012[,-1]), width = 200, p=4)
#' # Generalized rolling spillover index based on a VAR(4)
#'  G_index<- roll.spillover(as.zoo(dy2012[,-1]), width = 200, index="generalized", p=4) 
#' # A comparison: (warning: It can take several minutes.)
#'  single <- roll.spillover(as.zoo(dy2012[1:1200,2:4]), width = 200, p=4)
#'  partial <- roll.spillover(as.zoo(dy2012[1:1200,2:4]), width = 200, p=4, ortho.type = "partial")
#'  total <- roll.spillover(as.zoo(dy2012[1:1200,2:4]), width = 200, p=4, ortho.type = "total")
#'  out <- cbind(single, partial, total)
#'  head(out)
#'  plot(out, col=1:3, main="Spillover index")
#' }

roll.spillover <- function (data, width,  n.ahead = 10, index=c("orthogonalized", "generalized"), 
                            ortho.type = c("single", "partial", "total"), ...) 
{
  if (!(class(data) == "zoo")) {
    stop("\nPlease provide an object of class 'zoo', generated by 'as.zoo'.\n")
  }
  
  K <- ncol(data)+1
  
  roll.index <- switch(match.arg(index), 
                       orthogonalized = {
                         switch(match.arg(ortho.type),
                                single={rollapply(data, width = width,
                                                  FUN = function(z) {
                                                    O.spillover(VAR(z,...), ortho.type = c("single"))[K,K]
                                                  },
                                                  by.column = FALSE, align = "right")},
                                partial={rollapply(data, width = width,
                                                   FUN = function(z) {
                                                     O.spillover(VAR(z,...), ortho.type = c("partial"))[K,K]
                                                   },
                                                   by.column = FALSE, align = "right")},
                                total={rollapply(data, width = width,
                                                 FUN = function(z) {
                                                   O.spillover(VAR(z,...), ortho.type = c("total"))[K,K]
                                                 },
                                                 by.column = FALSE, align = "right")})
                          
                         
                         
                       }, 
                       generalized ={
                        
                         rollapply(data, width = width,
                                                 FUN = function(z) {
                                                   G.spillover(VAR(z,...))[K,K]
                                                 },
                                                 by.column = FALSE, align = "right") 
                       }  )
  
  
  
  return(roll.index)
  
}




