% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/e2e_plot_ycurve.R
\name{e2e_plot_ycurve}
\alias{e2e_plot_ycurve}
\title{Plot fishery yield curve data for planktivorous or demersal fish.}
\usage{
e2e_plot_ycurve(
  model,
  selection = "",
  use.saved = FALSE,
  use.example = FALSE,
  results = NULL,
  title = ""
)
}
\arguments{
\item{model}{R-list object defining the baseline model configuration used to generate the data and compiled by the e2e_read() function.}

\item{selection}{Text string from a list identifying the fish guild for which a yield curve is to be generated. Select from: "PLANKTIV", "DEMERSAL". Remember to include the phrase within "" quotes.}

\item{use.saved}{Logical. If TRUE use data from a prior user-defined run held as csv files data in the current results folder as set by an e2e_read() function call (default=FALSE).}

\item{use.example}{Logical. If TRUE use pre-computed example data from the internal North Sea model rather than user-generated data (default=FALSE).}

\item{results}{Dataframe generated by the function e2e_run_ycurve(). Only needed if use.saved2=FALSE and use.example2=FALSE. (Default=NULL).}

\item{title}{Optional free text (enclosed in "") to be added as a header for the plot (default = "").}
}
\value{
Dataframe of results from which the plot is created, graphical display in a new graphics window
}
\description{
Plot planktivorous or demersal fish yield curve data generated by the function e2e_run_ycurve().
}
\details{
In the function e2e_run_ycurve(), the baseline for the sequence of runs (harvest ratio multiplier = 1.0) is a model name and variant as loaded by the e2e_read() function.
The function then generates a set of biomass, landings and discards data for multiples of the target fish (planktivorous or demersal) harvest ratios relative to this baseline. This is done for a given value of
the alternative (demersal or planktivorous) harvest ratio (also a multiple of the the baseline). 

This function plots two graphs - the annual average fish biomass in the whole model domain (mMN/m2) as a function of harvest ratio multiplier, and the yield curve, ie the annual catch (and discards) (mMN/m2/y) as functions of the multiplier.

The yield curve represents the catch that would be generated from the stationary state of the model attained with long-term repeating annual cycles of all driving data including fishing.

Arguments for this function permit the input data to be drawn from an existing data object generated by the function e2e_run_ycurve(), a previously generated csv file, or example data
provided with the package for versions of the internal North Sea models.
}
\examples{
\donttest{
# Load the 1970-1999 version of the North Sea model supplied with the package and
# generate a yield data object:
# WARNING - this example will take several minutes to run even though the model is
# only run for 3 years per step in harvest ratio. A realistic run would require
# at least 50 years per step and take much longer.
    model <- e2e_read("North_Sea", "1970-1999")
    pfhr=c(0,0.5,0.75,1.0,1.25,1.5,2.0,2.5,3.0) 
    pf_yield_data <- e2e_run_ycurve(model,selection="PLANKTIV",nyears=3, HRvector=pfhr,
                                    HRfixed=1,csv.output=FALSE)
    data <- e2e_plot_ycurve(model,selection="PLANKTIV", results=pf_yield_data,
                            title="Planktivorous yield with baseline demersal fishing")

# Users can then plot other biomass, landings and discards data in the results
# object by, for example:
    par(mfrow=c(2,1))
    par(mar=c(3.2,5,2,0.8))
    ym<-1.1*max(pf_yield_data$Cetaceanbiom)
    plot(pf_yield_data$PlankFishHRmult, pf_yield_data$Cetaceanbiom, ylim=c(0,ym), 
         type="l",lwd=3,yaxt="n",xaxt="n",ann=FALSE)
       abline(v=1,lty="dashed")
       axis(side=1,las=1,cex.axis=0.9)
       axis(side=2,las=1,cex.axis=0.9)
       mtext("Planktiv. fish harvest ratio multiplier",cex=1,side=1,line=2)
       mtext("Cetacean biomass",cex=1,side=2,line=3.5)
       mtext(bquote("mMN.m"^-2),cex=0.7,side=3,line=-0.05,adj=-0.18)
    ym<-1.1*max(pf_yield_data$Cetaceandisc)
    plot(pf_yield_data$PlankFishHRmult, pf_yield_data$Cetaceandisc, ylim=c(0,ym),
         type="l",lwd=3,yaxt="n",xaxt="n",ann=FALSE)
       abline(v=1,lty="dashed")
       axis(side=1,las=1,cex.axis=0.9)
       axis(side=2,las=1,cex.axis=0.9)
       mtext("Planktiv. fish harvest ratio multiplier",cex=1,side=1,line=2)
       mtext("Cetacean by-catch",cex=1,side=2,line=3.5)
       mtext(bquote("mMN.m"^-2 ~ ".y"^-1),cex=0.7,side=3,line=-0.05,adj=-0.18)
}

# Using example data generated with selection="PLANKTIV" ...
# Plot example data for one of the North Sea model versions internal to the package
# This example requires the Strathe2E2examples supplementary data package.
if(require(StrathE2E2examples)){
     model <- e2e_read("North_Sea", "1970-1999")
     pf_yield_data<-e2e_plot_ycurve(model, selection="PLANKTIV", use.example=TRUE)
}

# Using example data generated with selection="DEMERSAL"...
# This example requires the Strathe2E2examples supplementary data package.
if(require(StrathE2E2examples)){
    model <- e2e_read("North_Sea", "1970-1999")
    df_yield_data<-e2e_plot_ycurve(model, selection="DEMERSAL", use.example=TRUE)
}

}
\seealso{
\code{\link{e2e_read}}, \code{\link{e2e_run}}, \code{\link{e2e_run_ycurve}}
}
