/*  
** Protocol for Application -- Database Interface
**
** Small changes to avoid abort and writing to stdout, disallowed in 
** R-2.14.2. Copyright P.Gilbert 2012
**
** Copyright 1995, 1996  Bank of Canada.
**
** The user of this software has the right to use, reproduce and distribute it.
** Bank of Canada makes no warranties with respect to the software or its 
** fitness for any particular purpose. The software is distributed by the Bank
** of Canada solely on an "as is" basis. By using the software, user agrees to 
** accept the entire risk of using this software.
**
*/

#if defined(FAME_SVC)
#define PADI_MAIN
#define PADI_SVC
#elif defined(FS_SVC)
#define PADI_MAIN
#define PADI_SVC
#else
#define PADI_CLIENT
#endif

#ifdef PADI_SELFTEST
#define PADI_MAIN
#ifdef PADI_SVC
#undef PADI_SVC
#endif
#endif

#include <R.h>
 
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#include <pwd.h>
#include "padi.h"

#ifdef FAME_SVC
#include <hli.h>
#endif

export PadiString_t Padi_EMPTY_STR = "";

#ifndef PADI_CLIENT

import PadiStatus_t initialize_1 PROTO1(PadiInitArg_tp, object);
import PadiStatus_t terminate_1 PROTO1(PadiTermArg_tp, object);
import char * status_1 PROTO1(int, status);

#endif /* PADI_CLIENT */

/*
This is all converted to use strerror instead
#include <errno.h>
import int sys_nerr;

#ifndef linux
import char * sys_errlist[];
#endif  
In Linux above gives
/usr/include/stdio.h:554: previous declaration of `sys_errlist'
commenting out the line compiles (in Linux)
*/

private struct signal_tab_t {
  int signal;
  int severity;
  char * message;
} signal_tab[] = {
#ifndef NOTRAP
  {SIGINT   , Padi_FATAL, "interrupt"},
  {SIGILL   , Padi_FATAL, "illegal instruction"},
  {SIGFPE   , Padi_FATAL, "arithmetic exception"},
  {SIGSEGV  , Padi_FATAL, "segmentation violation"},
  {SIGTERM  , Padi_FATAL, "software termination signal"},
  {SIGABRT  , Padi_FATAL, "abort (generated by abort(3) routine)"},
#ifdef _MSC_VER 
#ifdef WIN32
  {SIGBREAK , Padi_FATAL, "ctrl-break"},
#endif
#else
#ifndef __linux__
  {SIGEMT   , Padi_FATAL, "emulator trap"},
  {SIGSYS   , Padi_FATAL, "bad argument to system call"},
#endif
#ifndef _HPUX_SOURCE
  {SIGXCPU  , Padi_FATAL, "cpu time limit exceeded (see getrlimit(2))"},
  {SIGXFSZ  , Padi_FATAL, "file size limit exceeded (see getrlimit(2))"},
#endif
  {SIGHUP   , Padi_FATAL, "hangup"},
  {SIGQUIT  , Padi_FATAL, "quit"},
  {SIGTRAP  , Padi_FATAL, "trace trap"},
  {SIGKILL  , Padi_FATAL, "kill (cannot be caught, blocked, or ignored)"},
  {SIGBUS   , Padi_FATAL, "bus error"},
  {SIGPIPE  , Padi_ERROR, "write on a pipe or other socket with no one to read it"},
  {SIGALRM  , Padi_FATAL, "alarm clock"},
  {SIGURG   , Padi_FATAL, "urgent condition present on socket"},
  {SIGSTOP  , Padi_FATAL, "stop (cannot be caught, blocked, or ignored)"},
  {SIGTSTP  , Padi_WARNING,  "stop signal generated from keyboard"},
  {SIGCONT  , Padi_WARNING,  "continue after stop"},
  {SIGCHLD  , Padi_FATAL, "child status has changed"},
  {SIGTTIN  , Padi_FATAL, "background read attempted from control terminal"},
  {SIGTTOU  , Padi_FATAL, "background write attempted to control terminal"},
  {SIGIO    , Padi_WARNING,  "I/O is possible on a descriptor (see fcntl(2V))"},
  {SIGVTALRM, Padi_FATAL, "virtual time alarm (see getitimer(2))"},
  {SIGPROF  , Padi_FATAL, "profiling timer alarm (see getitimer(2))"},
  {SIGWINCH , Padi_WARNING,  "window changed (see termio(4) and win(4S))"},
  {SIGUSR1  , Padi_FATAL, "user-defined signal 1"},
  {SIGUSR2  , Padi_FATAL, "user-defined signal 2"},
#endif
#endif
  {0        , Padi_FATAL, NULL}
};

private char * SignalMessage PARAM1(int, sig)
{
  struct signal_tab_t *p;

  for (p = signal_tab; p->message; p++)
    if (p->signal == sig)
      return p->message;

  return Padi_EMPTY_STR;
}

export char * PadiStatus PARAM1(int, status)
/*
** Convert a PADI status code to a character string message.
**
** status (input)
**   The status code.  
**
** RETURN
**   A string message for the PADI status code.
**
** SEE ALSO
**   PADI_SELFTEST
**
** FILES
**   padi.c, padi.h
*/
{
  if (status > 0 && status < Padi_STATUS) {
  
#ifdef PADI_CLIENT
    return "database server error";
#else
    return status_1(status);
#endif

  } else if (status >= Padi_STATUS && status < Padi_SVC_STATUS) {

    switch (status) {
    case Padi_FAIL:
      return "unspecified failure";

    case Padi_OUT_OF_MEMORY:
      return "insufficient memory";

    case Padi_TOO_MANY_OPEN:
      return "attempt to open too many databases";

    case Padi_OVERWRITE:
      return "attempt to overwrite data in append mode";

    case Padi_FILE_OPEN:
      return "unable to open file";

    case Padi_TCP_CREATE:
      return "unable to create tcp service";

    case Padi_TCP_REGISTER:
      return "unable register tcp service";

    case Padi_FREE_ARG:
      return "unable to free argument memory";

    case Padi_READ_ACCESS:
      return "attempt to update data in read mode";

    case Padi_MISSING_OBJECT:
      return "missing object name";

    case Padi_UNKNOWN_OBJECT:
      return "unknown object name";

    case Padi_MISSING_FORMAT:
      return "missing format specification";

    case Padi_UNSUPPORTED_FORMAT:
      return "unsupported format specification";

    case Padi_MISSING_TYPE:
      return "missing type specification";

    case Padi_TYPE_MISMATCH:
      return "type mismatch";

    case Padi_OBJECT_EXISTS:
      return "object already exists";

    case Padi_MISSING_DBNAME:
      return "missing database name";

    case Padi_UNSUPPORTED_CLASS:
      return "unsupported class specification";

    case Padi_UNSUPPORTED_FREQ:
      return "unsupported frequency specification";

    case Padi_UNSUPPORTED_TYPE:
      return "unsupported type specification";
 
    case Padi_UNATH_USER:
      return "unauthorized user";

    default:
      return Padi_EMPTY_STR; 
  
    }
  } else if (status >= Padi_SVC_STATUS && status < Padi_CLNT_STATUS) {
    /* server errors are sent to client by RPC mechanism */
  } else if (status >= Padi_CLNT_STATUS && status < Padi_SYSTEM_STATUS) {
    switch (status) {
    case Padi_NO_CONNECTION:
      return clnt_spcreateerror("");
    case Padi_CLNT_SET_TIMEOUT:
      return "error setting client timeout period";
    default:
      return clnt_sperrno((enum clnt_stat)(status - Padi_CLNT_STATUS));

    }
  } else if (status >= Padi_SYSTEM_STATUS && status < Padi_HOST_STATUS) {
      int i = (int)(status - Padi_SYSTEM_STATUS);

/* this does not work in 64 bit app
      if (i < sys_nerr) {
        return sys_errlist[i];
      } 
*/
      return strerror(i);

  } else if (status >= Padi_HOST_STATUS && status < Padi_SIGNAL_STATUS) {
    return "host not found";

  } else if (status >= Padi_SIGNAL_STATUS) {
    return SignalMessage(status - Padi_SIGNAL_STATUS);

  }

  return Padi_EMPTY_STR;
}

export PadiBoolean_t PadiVerbose = FALSE;

export PadiStatus_t PadiInitialize PARAM1(PadiInitArg_tp, object)
/*
** Initialize the Protocol for Application -- Database Interface (PADI)
**
** object 
**   The name of the database object with user and password.
** 
**   struct PadiInitArg_t {
**     PadiString_t user;
**     PadiString_t password;
**     PadiString_t object_name;
**   }
**
** RETURN
**   Zero <==> success.  
**   Non-zero <==> failure or qualified success.
**
** NOTES
**   Must be called before any other Padi function.
**   May only be called once. Optional for remote clients.
**
** SEE ALSO
**   PadiStatus, PADI_SELFTEST
**
** FILES
**   padi.c, padi.h
*/
{
  PadiInitArg_t dummy;

  /* eliminate null pointers so that xdr doesn't barf */

  if (!object) { 
    object = &dummy;
    bzero((char *)(object), sizeof(*object));
  }
  if (!(object->user))
    object->user = Padi_EMPTY_STR;
  if (!(object->password))
    object->password = Padi_EMPTY_STR;
  if (!(object->object_dbname))
    object->object_dbname = Padi_EMPTY_STR;
#ifdef PADI_CLIENT

  return Padi_SUCCEED;

#else

  return initialize_1(object);

#endif /* PADI_CLIENT */
}

export PadiStatus_t PadiTerminate PARAM1(PadiTermArg_tp, object)
/*
** Terminate the Protocol for Application -- Database Interface (PADI)
**
** object 
**   The name of the  user and password.
** 
**   struct PadiInitArg_t {
**     PadiString_t user;
**     PadiString_t password;
**   }
**
** RETURN
**   Zero <==> success.  
**   Non-zero <==> failure or qualified success.
**
** NOTES
**   Call to close all open databases and terminate Padi.
**   Do not call other Padi functions after calling PadiTerminate
**   May only be called once. Optional for remote clients.
**
** SEE ALSO
**   PadiStatus, PADI_SELFTEST
**
** FILES
**   padi.c, padi.h
*/
{
  PadiTermArg_t dummy;

  PadiChannelClose();

  /* eliminate null pointers so that xdr doesn't barf */

  if (!object) { 
    object = &dummy;
    bzero((char *)(object), sizeof(*object));
  }
  if (!(object->user))
    object->user = Padi_EMPTY_STR;
  if (!(object->password))
    object->password = Padi_EMPTY_STR;

#ifdef PADI_CLIENT

  return Padi_SUCCEED;

#else

  return terminate_1(object);

#endif /* PADI_CLIENT */
}

export PadiInfoResult_tp PadiGetInfo PARAM2(PadiServer_tp, server, PadiInfoArg_tp, object)
/*
** Get information on an object.
**
** server->name 
**   The database server name.
** 
** object 
**   The name of the database object with user and password.
** 
**   struct PadiInfoArg_t {
**     PadiString_t user;
**     PadiString_t password;
**     PadiString_t object_name;
**     PadiString_t db_name;
**   }
** 
** RETURN
**   A pointer to the structure:
**   
**   struct PadiInfoResult_t {
**     PadiData_tp xdr_proc;
**     PadiStatus_t status;
**     struct PadiInfo_t {
**       PadiString_t name;
**       PadiString_t desc;
**       PadiString_t class;
**       PadiString_t type;
**       PadiString_t access;
**       PadiString_t frequency;
**       PadiString_t start;
**       PadiString_t end;
**       PadiString_t format;
**     } info;
**   };
**
**   xdr_proc 
**     Used by PadiFreeResult().
**
**   status 
**     Zero <==> success.
**     Non-zero <==> failure or qualified success.
**
**   info.name 
**     The object name.
**
**   info.desc 
**     The object description.
**
**   info.class 
**     The object class.  Only "SERIES" is supported.
**    
**   info.type 
**     The data type of the returned time series.  
**     Only "PRECISION" is supported.
**    
**   info.access 
**     "READ", "UPDATE", "SHARED"
**    
**   info.frequency 
**     The frequency of the returned time series.  Supported frequencies are:
**    
**           DAILY
**           BUSINESS DAILY
**           WEEKLY
**           MONTHLY
**           BIMONTHLY
**           QUARTERLY
**           SEMI-ANNUAL
**           ANNUAL
**    
**   info.start 
**     A string indicating the starting date in the format:
**    
**           yyyymmdd, e.g.: 19950322
**    
**     Partial dates are accepted in which case the 1st month 
**     and/or day is inferred for the missing portion.  E.g.:
**    
**           1995   is accepted as 19950101
**           199503 is accepted as 19950301
**    
**   info.end 
**     A string indicating the ending date in the same
**     format as "start".
**
**   info.format
**     The available data formats for the object.
**     Only "SimpleSeries" is supported.
**
** NOTE
**   Free the returned result with PadiFreeResult(), e.g.:
**
**   result = PadiGetInfo(server, object);
**
**     . . .
**
**   PadiFreeResult((PadiResult_tp)result);
**
** SEE ALSO
**   PadiStatus, PadiFreeResult
**
** FILES
**   padi.c, padi.h
*/
{

  PadiInfoArg_t dummy;

  /* eliminate null pointers so that xdr doesn't barf */
  if (!object) { 
    object = &dummy;
    bzero((char *)(object), sizeof(*object));
  }
  if (!(object->user))
    object->user = Padi_EMPTY_STR;
  if (!(object->password))
    object->password = Padi_EMPTY_STR;
  if (!(object->object_name))
    object->object_name = Padi_EMPTY_STR;
  if (!(object->db_name))
    object->db_name = Padi_EMPTY_STR;


#ifdef PADI_CLIENT

  return (PadiInfoResult_tp)PadiClntCall(server, 
                                         GETINFO, 
                                         (xdrproc_t)xdr_PadiInfoArg_t, 
                                         (PadiData_tp)(object), 
                                         (xdrproc_t)xdr_PadiInfoResult_t, 
                                         sizeof(PadiInfoResult_t)); 

#else

  return getinfo_1(object, NULL);

#endif /* PADI_CLIENT */
}





export PadiResult_tp PadiDestroy PARAM2(PadiServer_tp, server, PadiDestroyArg_tp, dobject)
/*
** Destroy an object.
**
** server->name 
**   The database server name.
**  
** dobject 
**   The name of the database object with user and password.
** 
**   struct PadiDestroySeries_t {
**     PadiString_t user;
**     PadiString_t password;
**     PadiString_t object_name;
**     PadiString_t db_name;
**   }

** RETURN
**   A pointer to the result structure:
**
**   struct PadiResult_t {
**     PadiFunc_tp xdr_proc;
**     PadiStatus_t status;
**   };
**
**   xdr_proc 
**     Used by PadiFreeResult().
**
**   status 
**     Zero <==> success.
**     Non-zero <==> failure or qualified success.
**
** NOTE
**   Free the returned result with PadiFreeResult(), e.g.:
**
**   result = PadiGetInfo(server, object);
**
**     . . .
**
**   PadiFreeResult((PadiResult_tp)result);
**
** SEE ALSO
**   PadiStatus, PadiFreeResult
**
** FILES
**   padi.c, padi.h
*/
{
  PadiDestroyArg_t dummy;

  /* eliminate null pointers, etc. so that xdr doesn't barf */
  if (!dobject) { 
    dobject = &dummy;
    bzero((char *)(dobject), sizeof(*dobject));
  }
  if (!(dobject->user))
    dobject->user = Padi_EMPTY_STR;

  if (!(dobject->password))
    dobject->password = Padi_EMPTY_STR;

  if (!(dobject->object_name))
    dobject->object_name = Padi_EMPTY_STR;
  if (!(dobject->db_name))
    dobject->db_name = Padi_EMPTY_STR;


#ifdef PADI_CLIENT

  return (PadiResult_tp)PadiClntCall(server, DESTROY, 
                              (xdrproc_t)xdr_PadiDestroyArg_t, (PadiData_tp)(dobject),
			     	(xdrproc_t)xdr_PadiResult_t, sizeof(PadiResult_t)); 

#else

  return destroy_1(dobject, NULL);

#endif /* PADI_CLIENT */
}

export PadiSeriesResult_tp PadiGetSeries PARAM2(PadiServer_tp, server, PadiRangeArg_tp, local_arg)
/*
** Get a Series Object from a Local database.
**
** server->name 
**   The database server name.
**
** database_name 
**   The database name.
**
** range_arg
**   A pointer to the range argument structure:
**
**   struct PadiRangeArg_t {
**     PadiString_t user;
**     PadiString_t password;
**     PadiString_t object_name;
**     PadiString_t db_name;
**     struct PadiRange_t {
**       PadiString_t start;
**       PadiString_t end;
**       PadiSize_t nobs;
**       PadiBoolean_t do_missing;
**       PadiDouble_t missing_translation[3];
**       PadiString_t format;
**     } range;
**   };
**
**   user
**     A name provided by the client (for possible use by the server).
**
**   password
**     A string provided by the client (for possible use by the server).
**
**   object_name
**     The name of the series object.
**    
**   range.start 
**     A string indicating the starting date in the format:
**    
**           yyyymmdd, e.g.: 19950322
**    
**     Partial dates are accepted in which case the 1st month 
**     and/or day is inferred for the missing portion.  E.g.:
**    
**           1995   is accepted as 19950101
**           199503 is accepted as 19950301
**    
**   range.end 
**     A string indicating the ending date in the same
**     format as "start".
**
**   range.nobs
**     The number of observations to retrieve
**
**   NOTE:  Any 2 of start, end, and nobs may be specified.
**
**   range.do_missing
**     Use the supplied missing translation values iff TRUE.
**
**   range.missing_translation
**     3 values to use for missing values, respectively:
**
**           Not Computable (NC),
**           No Data (ND), and
**           Not Available (NA).
**  
**   range.format
**     Desired data format for returned series.
**
** RETURN
**   A pointer to the series result structure:
**
**   struct PadiSeriesResult_t {
**     PadiFunc_tp xdr_proc;
**     PadiStatus_t status;
**     struct PadiSeries_t {
**       struct PadiInfo_t {
**         PadiString_t name;
**         PadiString_t desc;
**         PadiString_t class;
**         PadiString_t type;
**         PadiString_t access;
**         PadiString_t frequency;
**         PadiString_t start;
**         PadiString_t end;
**         PadiString_t format;
**       } info;
**       struct PadiRange_t {
**         PadiString_t start;
**         PadiString_t end;
**         PadiSize_t nobs;
**         PadiBoolean_t do_missing;
**         PadiDouble_t missing_translation[3];
**         PadiString_t format;
**       } range;
**       struct {
**         u_int data_len;
**         char *data_val;
**       } data;
**     } series;
**   };
**
**   xdr_proc 
**     Used by PadiFreeResult().
**
**   status 
**      Zero <==> success.
**      Non-zero <==> failure or qualified success.
**    
**   series.*name 
**     The object name.
**
**   series.*desc 
**     The object description.
**
**   series.*class 
**     The object class.  Only "SERIES" is supported.
**    
**   series.*type 
**     The data type of the returned time series.  
**     Only "PRECISION" is supported.
**    
**   series.*access 
**     "READ", "UPDATE", "SHARE", "APPEND UPDATE", or "APPEND SHARE".
**    
**   series.*frequency 
**     The frequency of the returned time series.  Supported frequencies are:
**    
**           DAILY
**           BUSINESS DAILY
**           WEEKLY
**           MONTHLY
**           BIMONTHLY
**           QUARTERLY
**           SEMI-ANNUAL
**           ANNUAL
**    
**   series.*start 
**     A string indicating the starting date of the entire series in the format:
**    
**           yyyymmdd, e.g.: 19950322
**    
**     Partial dates are accepted in which case the 1st month 
**     and/or day is inferred for the missing portion.  E.g.:
**    
**           1995   is accepted as 19950101
**           199503 is accepted as 19950301
**    
**   series.*end 
**     A string indicating the ending date of the entire series in the same
**     format as "start".
**
**   series.*format
**     The available data formats for the object.
**     Only "SimpleSeries" is supported.
**
**    
**   series.range.start 
**     A string indicating the starting date of the retrieved data 
**     in the format:
**    
**           yyyymmdd, e.g.: 19950322
**    
**     Partial dates are accepted in which case the 1st month 
**     and/or day is inferred for the missing portion.  E.g.:
**    
**           1995   is accepted as 19950101
**           199503 is accepted as 19950301
**    
**   series.range.end 
**     A string indicating the ending date of the retrieved data 
**     in the same format as "start".
**
**   series.range.nobs
**     The number of observations retrieved
**
**   series.range.do_missing
**     Use the supplied missing translation values iff TRUE.
**
**   series.range.missing_translation
**     3 values to use for missing values, respectively:
**
**           Not Computable (NC),
**           No Data (ND), and
**           Not Available (NA).
**  
**   series.range.format
**     Data format of the returned series.
**
**   series.data.data_len
**     The number of bytes of returned series.data. 
**
**   series.data.data_value
**     The returned series.data.  The contents depend on the returned
**     data format (series.range.format).  Only "SimpleSeries" is supported.
**     
**     For SimpleSeries:  series.data.data_value should be cast to a pointer
**                  to a vector of the returned type (series.*type)
**                  containing series.range.nobs elements.
**
** NOTE
**   Free the returned result with PadiFreeResult(), e.g.:
**
**   result = PadiGetInfo(server, object);
**
**     . . .
**
**   PadiFreeResult((PadiResult_tp)result);
**
** SEE ALSO
**   PadiStatus, PadiFreeResult
**
** FILES
**   padi.c, padi.h
*/
{
  PadiRangeArg_t dummy;


  /* eliminate null pointers so that xdr doesn't barf */
  if (!local_arg) { 
    local_arg = &dummy;
    bzero((char *)(local_arg), sizeof(*local_arg));
  }
  if (!(local_arg->user))
    local_arg->user = Padi_EMPTY_STR;
  if (!(local_arg->password))
    local_arg->password = Padi_EMPTY_STR;
  if (!(local_arg->object_name))
    local_arg->object_name = Padi_EMPTY_STR;
  if (!(local_arg->db_name))
    local_arg->db_name = Padi_EMPTY_STR;
  if (!(local_arg->range.format))
    local_arg->range.format = Padi_EMPTY_STR;
  if (!(local_arg->range.start))
    local_arg->range.start = Padi_EMPTY_STR;
  if (!(local_arg->range.end))
    local_arg->range.end = Padi_EMPTY_STR;

#ifdef PADI_CLIENT


  return (PadiSeriesResult_tp)PadiClntCall(server, GETSERIES, 
                          (xdrproc_t)xdr_PadiRangeArg_t, (PadiData_tp)local_arg, 
                          (xdrproc_t)xdr_PadiSeriesResult_t, sizeof(PadiSeriesResult_t)); 

#else

#ifdef PUBLIC
  return getseries_1(local_arg, NULL);
#else
  return (PadiSeriesResult_t *)getlocal_1(local_arg, NULL);
#endif /* PUBLIC */

#endif /* PADI_CLIENT */
}


export PadiResult_tp PadiNewSeries PARAM2(PadiServer_tp, server, PadiNewSeries_tp, new_series)
/*
** Create a new Series Object.
**
** server->name 
**   The database server name.
**
** new_series
**   A pointer to the new series structure:
**
**   struct PadiLocalSeries_t {
**     PadiString_t user;
**     PadiString_t password;
**     PadiString_t dbname;
**     struct PadiSeries_t {
**       struct PadiInfo_t {
**         PadiString_t name;
**         PadiString_t desc;
**         PadiString_t class;
**         PadiString_t type;
**         PadiString_t access;
**         PadiString_t frequency;
**         PadiString_t start;
**         PadiString_t end;
**         PadiString_t format;
**       } info;
**       struct PadiRange_t {
**         PadiString_t start;
**         PadiString_t end;
**         PadiSize_t nobs;
**         PadiBoolean_t do_missing;
**         PadiDouble_t missing_translation[3];
**         PadiString_t format;
**       } range;
**       struct {
**         u_int data_len;
**         char *data_val;
**       } data;
**     } series;
**   };
**
**   user
**     A name provided by the client (for possible use by the server).
**
**   password
**     A string provided by the client (for possible use by the server).
**
**   dbname
**     The name of the database where the new series is to be created.
**
**   series.info.name 
**     The new series object name.
**
**   series.info.desc 
**     Ignored.
**
**   series.info.class 
**     The object class.  Only "SERIES" is supported.
**    
**   series.info.type 
**     The data type of the new time series.  
**     Only "PRECISION" is supported.
**    
**   series.info.access 
**     Ignored.
**    
**   series.info.frequency 
**     The frequency of the new time series.  Supported frequencies are:
**    
**           DAILY
**           BUSINESS DAILY
**           WEEKLY
**           MONTHLY
**           BIMONTHLY
**           QUARTERLY
**           SEMI-ANNUAL
**           ANNUAL
**    
**   series.info.start 
**     Ignored.
**    
**   series.info.end 
**     Ignored.
**
**   series.info.format
**     Ignored.
**    
**   series.range.start 
**     A string indicating the starting date of the data passed in 
**     in the format:
**    
**           yyyymmdd, e.g.: 19950322
**    
**     Partial dates are accepted in which case the 1st month 
**     and/or day is inferred for the missing portion.  E.g.:
**    
**           1995   is accepted as 19950101
**           199503 is accepted as 19950301
**    
**   series.range.end 
**     A string indicating the ending date of the data passed in 
**     in the same format as "start".
**
**   series.range.nobs
**     The number of observations passed in. 
**
**   series.range.do_missing
**     Use the supplied missing translation values iff TRUE.
**
**   series.range.missing_translation
**     3 values to use for missing values, respectively:
**
**           Not Computable (NC),
**           No Data (ND), and
**           Not Available (NA).
**  
**   series.range.format
**     Data format of the series passed in.
**
**   series.data.data_len
**     The number of bytes of passed in series.data. 
**
**   series.data.data_value
**     The passed in series.data.  The contents depend on the passed in 
**     data format (series.range.format).  Only "SimpleSeries" is supported.
**     
**     For SimpleSeries:  series.data.data_value will be cast to a pointer
**                  to a vector of the passed in type (series.info.type)
**                  containing series.range.nobs elements.
**  
** RETURN
**   A pointer to the result structure:
**
**   struct PadiResult_t {
**     PadiFunc_tp xdr_proc;
**     PadiStatus_t status;
**   };
**
**   xdr_proc 
**     Used by PadiFreeResult().
**
**   status 
**     Zero <==> success.
**     Non-zero <==> failure or qualified success.
**
** NOTE
**   Free the returned result with PadiFreeResult(), e.g.:
**
**   result = PadiGetInfo(server, object);
**
**     . . .
**
**   PadiFreeResult((PadiResult_tp)result);
**
** SEE ALSO
**   PadiStatus, PadiFreeResult
**
** FILES
**   padi.c, padi.h
*/
{
  PadiNewSeries_t dummy;
  PadiSeries_tp series;
  PadiString_t *strp; 
  PadiPrecision_t dbl_dummy;

  /* eliminate null pointers so that xdr doesn't barf */
  if (!new_series) { 
    new_series = &dummy;
    bzero((char *)(new_series), sizeof(*new_series));
  }
  if (!(new_series->user))
    new_series->user = Padi_EMPTY_STR;
  if (!(new_series->password))
    new_series->password = Padi_EMPTY_STR;
  if (!(new_series->dbname))
    new_series->dbname = Padi_EMPTY_STR;

  series = &(new_series->series);
  if (!(series->range.start))
    series->range.start = Padi_EMPTY_STR;
  if (!(series->range.end))
    series->range.end = Padi_EMPTY_STR;
  if (!(series->range.format))
    series->range.format = Padi_EMPTY_STR;
  if (!(series->data.data_val)) {
    series->data.data_val = &dbl_dummy;
    series->data.data_len = series->range.nobs = 0;
    series->range.start = series->range.end = Padi_EMPTY_STR;
  }
  for (strp = &(series->info.name); 
       strp < (&(series->info.name) + Padi_NUM_INFO_STR); strp++)
    if (!(*strp))
      *strp = Padi_EMPTY_STR;

#ifdef PADI_CLIENT

  return (PadiResult_tp)PadiClntCall(server, NEWSERIES, 
                                (xdrproc_t)xdr_PadiNewSeries_t, (PadiData_tp)new_series, 
                                (xdrproc_t)xdr_PadiResult_t, sizeof(PadiResult_t)); 

#else

#ifdef PUBLIC
  return newseries_1(new_series, NULL);
#else
  return (PadiResult_t *)putlocal_1(new_series, NULL);
#endif /* PUBLIC */

#endif /* PADI_CLIENT */
}

export void PadiFreeResult PARAM1(PadiResult_tp, result)
/*
** Release memory used by a result structure. 
**
** result
**   A pointer to the result structure.
**
** FILES
**   padi.c, padi.h
*/
{
  if (!result)
    return;

#if defined(PADI_CLIENT) || defined(PADI_SVC)
  if (result->xdr_proc) 
    xdr_free((xdrproc_t)(result->xdr_proc), (char *)result);
#endif

  free((char *)result);
}


#ifndef NOT_R_CLIENT
extern void error(const char *, ...);
#endif
  
/*export void PadiError PARAM4(FILE *, fp, PadiString_t, source, PadiStatus_t, status, int, severity) */

export void PadiError(FILE *fp, PadiString_t source, PadiStatus_t status, int severity)
/*
** PADI Error Handler 
**
** fp
**   File pointer for messages.
** 
** source
**   Identifying text for error source.
**
** status
**   Status code for error.
**
** severity
**   Error severity:
**
**      Padi_WARNING, PadiError ==> log message and return.
**      Padi_FATAL ==> log message and error() to R (NOT abort).
**
** FILES
**   padi.c, padi.h
*/
{              
  PadiTermArg_t termobject;
  string_t s;
  time_t ltime;
  int n;

 
  switch(severity) {
  case Padi_WARNING:
    s = "WARNING";
    break;

  case Padi_ERROR:
    s = "ERROR";
    break;

  case Padi_FATAL:
    s = "FATAL ERROR";
    break;

  default:
    s = "s uninitialized";
    break;
  }

  time( &ltime );

n = (PadiVerbose) ? 2 : 1;

#ifdef NOT_R_CLIENT
 for (n = (PadiVerbose) ? 2 : 1; n--; fp = stdout) { 
    if (status)
      fprintf(fp, "\t%s %s# %d: %s.\n", source, s, status, PadiStatus(status));
    else
     fprintf(fp, "\t%s.\n", source);
  
    fflush(fp);
  }

  if (severity < Padi_FATAL)
    return;

   termobject.user = Padi_EMPTY_STR;
   termobject.password = Padi_EMPTY_STR;
   PadiTerminate(&termobject);
   exit(status);
#else
error(PadiStatus(status)); /* in fact, this routine should never be called for R client */
#endif 
}


#ifndef NOT_R_CLIENT
export void PadiErrorR(PadiString_t source, PadiStatus_t status, int severity)
/*
** PADI R Error Handler 
** 
** source
**   Identifying text for error source.
**
** status
**   Status code for error.
**
** severity
**   Error severity:
**
**      Padi_WARNING, PadiError ==> log message and return.
**      Padi_FATAL ==> log message and error() to R (NOT abort).
*/
{              
  PadiTermArg_t termobject;
  string_t s;
  time_t ltime;
  /*int n;*/

 
  switch(severity) {
  case Padi_WARNING:
    s = "WARNING";
    break;

  case Padi_ERROR:
    s = "ERROR";
    break;

  case Padi_FATAL:
    s = "FATAL ERROR";
    break;

  default:
    s = "s uninitialized";
    break;
  }

  time( &ltime );

  if (severity < Padi_FATAL)
    return;

   termobject.user = Padi_EMPTY_STR;
   termobject.password = Padi_EMPTY_STR;
   PadiTerminate(&termobject);
   /* error(s); */
   /* error(PadiStatus(status));*/
   error("Padi fatal error");
}
#endif 

#ifdef PADI_MAIN

private string_t app_name = "padi";
private string_t app_vers = "v1.11";
 
private void Notice PARAM1(FILE *, stream)
{
 
  fprintf(stream, "\n============= %s %s Copyright (C) 1995 Bank of Canada ============\n\n",
                  app_name, app_vers);

  fprintf(stream, "  Any person using this software has the right to use, \n");
  fprintf(stream, "  reproduce and distribute it.\n\n");

  fprintf(stream, "  The Bank does not warrant the accuracy of the information contained in\n");
  fprintf(stream, "  the software.  User assumes all liability for any loss or damages,\n");
  fprintf(stream, "  either direct or indirect, arising from the use of the software.\n\n");
 
}
 
private void Usage PARAM2(FILE *, stream, char *, name)
{
  Notice(stream);
#ifdef PADI_CLIENT
  fprintf(stream, "Usage: %s [serverhost] [-t sec usec]\n", name);
  fprintf(stream, "          [-s start] [-e end] [-n nobs] [-l logfile] [-v]\n");
#else
  fprintf(stream, "Usage: %s objectdb [-l logfile] [-v]\n", name);
#endif
  fprintf(stream, "       %s -help\n\n", name);
}
 
private void Help PARAM2(FILE *, stream, char *, name)
{
  Usage(stream, name);
#ifdef PADI_CLIENT
  fprintf(stream, "  Protocol for Application -- Database Interface Test Client.\n");
  fprintf(stream, "  serverhost: the server host name (local host).\n");
  fprintf(stream, "  -t sec usec: time out period in seconds + micro seconds.\n");
  fprintf(stream, "  -s start: start date for test series [YYYYMMDD].\n");
  fprintf(stream, "  -e end: end date for test series [YYYYMMDD].\n");
  fprintf(stream, "  -n nobs: number of observations for test series.\n");
#else
  fprintf(stream, "  Protocol for Application -- Database Interface Server.\n");
  fprintf(stream, "  objectdb: the object data base path name.\n");
#endif
  fprintf(stream, "  -l logfile: the log file path name (%s.log).\n", name);
  fprintf(stream, "  -v: echo log file messages to stdout\n\n");
}

export FILE *logfp;

private int SignalSeverity PARAM1(int, sig)
{
  struct signal_tab_t *p;

  for (p = signal_tab; p->message; p++)
    if (p->signal == sig)
      return p->severity;

  return Padi_FATAL;
}

private void sighandler PARAM1(int, sig) 
{
  char buf[128];
  
  sprintf(buf, "%s: ", app_name);

  PadiError(logfp, buf, Padi_SIGNAL_STATUS + sig, SignalSeverity(sig));
}
 
typedef struct mainopt_t {
  string_t objectdb;
  string_t logfile;
  string_t datetest;
  string_t start;
  string_t end;
  size_t nobs;
  int second;
  int usecond;
  bool_t verbose;
  unsigned port;
} mainopt_t, *mainopt_tp;
 
private void GetArg PARAM3(int, argc, char **, argv, mainopt_tp, option)
/*
 * get command line arguments and set options
 *
*/
{
  int i;
  char *s;
 
  /* set defaults */
  option->objectdb = NULL;
  option->logfile = NULL;
  option->datetest = NULL;
  option->start = NULL;
  option->end = NULL;
  option->nobs = 0;
  option->second = 25;
  option->usecond = 0;
  option->verbose = FALSE;
 
  /* process arguments */
  for (i = 1; i < argc; i++) {
    if (argv[i][0] == '-') {
      for (s = argv[i] + 1; *s; ) {
        switch (*s++) {
        case 'l':
          option->logfile = argv[++i];
          break;
 
        case 'v':
          option->verbose = TRUE;
          break;
 
        case 'd':
          option->datetest = argv[++i];
          break;
 
        case 's':
          option->start = argv[++i];
          break;
 
        case 'e':
          option->end = argv[++i];
          break;
 
        case 'n':
          option->nobs = atoi(argv[++i]);
          break;
 
        case 't':
          option->second = atoi(argv[++i]);
          option->usecond = atoi(argv[++i]);
          break;
 
        case 'p':
          option->port = atoi(argv[++i]);
          break;
 
        case 'h':
        case 'H':
        case '?':
          Help(stdout, app_name);
          exit(ERROR_EXIT);
          break;
 
        default:
          Usage(stdout, app_name);
          exit(ERROR_EXIT);
          break;
 
        }
      }
    } else {
      if (option->objectdb == NULL)
        option->objectdb = argv[i];
      else {
        Usage(stdout, app_name);
        exit(ERROR_EXIT);
      }
    }
  }

#ifndef PADI_CLIENT 
  if (option->objectdb == NULL) {
    Usage(stdout, app_name);
    exit(ERROR_EXIT);
  }
#endif
 
  if (option->objectdb && (s = strrchr(option->objectdb, ':'))) {
    *s++ = '\0';
    option->port = atoi(s);
  }
}

#endif /* PADI_MAIN */

#ifdef PADI_SELFTEST

#define WRITE_LEN 100

#ifndef NSIG
#define NSIG = SIGUSR2 + 1;
#endif

export int main PARAM2(int, argc, char **, argv)
/*
** Selftest main() for the Data Base Application Programer's Interface.
**
** RETURN
**   Zero <==> success.  
**   Non-zero <==> failure or qualified success.
**
** NOTES
**
**   The selftest main in padi.c contains working examples for all the PADI calls.
**
**   Make selftest with:
**
**   cc -g -I$(FAME)/hli -DPADI_SELFTEST -DPADI_SVC padi.c hlifunc.o -L$(FAME)/hli -lchli  -lm -lc -o padi
**
** FILES
**   padi.c, padi.h
*/
{
  PadiStatus_t status;
  PadiSize_t i, j;
  PadiServer_t server;
  PadiServer_t too_long;
  PadiInfoResult_tp info_result;
  PadiRangeArg_t range_arg;
  PadiSeriesResult_tp series_result;
  PadiNewSeries_t new_series;
  PadiInfoArg_t infoobject;
  PadiTermArg_t termobject;
  PadiSeries_tp series = &(new_series.series);
  PadiInitArg_t initobject;
  PadiResult_tp result;
  mainopt_t option;
  char log_buf[128];
  size_t nwrite;
  PadiPrecision_t *vector;
  char too_long_name[Padi_MAX_NAME_LEN<<1 + 1];
  char too_long_str[Padi_MAX_STR_LEN<<1 + 1];
#ifdef MSDOS
  char fname[16];

  _splitpath(*argv, NULL, NULL, fname, NULL);
  app_name = fname;
#else
  app_name = *argv;
#endif

  GetArg(argc, argv, &option);

  if (!(option.logfile)) {
    strcpy(log_buf, app_name);
    option.logfile = strcat(log_buf, ".log");
  }

  if (!(logfp = fopen(option.logfile, "a"))) {
    sprintf(log_buf, "%s (%s)", app_name, option.logfile);
    PadiError(stderr, log_buf, Padi_FILE_OPEN, Padi_FATAL);

  }

  PadiVerbose = option.verbose;

   initobject.user = Padi_EMPTY_STR;
   initobject.password = Padi_EMPTY_STR;
   initobject.object_dbname = option.objectdb;

   infoobject.user = Padi_EMPTY_STR;
   infoobject.password = Padi_EMPTY_STR;

   termobject.user = Padi_EMPTY_STR;
   termobject.password = Padi_EMPTY_STR;


  if ((status = PadiInitialize(&initobject))) {
    sprintf(log_buf, "%s (%s)", app_name, option.objectdb);
    PadiError(logfp, log_buf, status, Padi_FATAL);
  
  }

  (void)PadiSetTimeOut(option.second, option.usecond);

  if (option.datetest) {

#ifdef FAME_SVC

    import void datetest_1 PROTO1(PadiString_t, datetest);
    datetest_1(option.datetest);

#endif

    return Padi_SUCCEED;

  }

  server.name = option.objectdb;
  server.port = option.port;

#ifndef NOTRAP
  for (i = 0; i < NSIG; i++) {
    switch (i) {
    case SIGABRT:
#ifndef MSDOS
    case SIGKILL:
    case SIGSTOP:
    case SIGTSTP:
    case SIGCONT:
    case SIGIO:
    case SIGURG:
    case SIGWINCH:
#endif
      break;

    default:
      signal(i, sighandler); 
    }
  }
#endif

  /* initialize some data for writing series */
  bzero((char *)(&new_series), sizeof(new_series));
  nwrite = (option.nobs) ? option.nobs : WRITE_LEN;
  series->range.nobs = nwrite;
  series->range.start = "19931231";
  series->range.end = Padi_EMPTY_STR;
  series->range.format = "SimpleSeries";
  series->range.do_missing = TRUE;
  series->range.missing_translation[0] = (PadiPrecision_t)(nwrite - 10);
  series->range.missing_translation[1] = (PadiPrecision_t)(nwrite - 9);
  series->range.missing_translation[2] = (PadiPrecision_t)(nwrite - 8);

  series->data.data_len = nwrite*sizeof(PadiPrecision_t);
  if (!(series->data.data_val = malloc(nwrite*sizeof(PadiPrecision_t)))) {
    sprintf(log_buf, "%s (%s)", app_name, server.name);
    PadiError(logfp, log_buf, Padi_OUT_OF_MEMORY, Padi_FATAL);
    
  }
  bzero(series->data.data_val, series->data.data_len);
  vector = (PadiPrecision_t *)(series->data.data_val);
  for (i = 0; i < nwrite; i++)
  {
    vector[i] = (PadiPrecision_t)i;
  }

  *((float *)vector) = 1.0; /* for scalars */

  /* initialize the range arg */ 
  bzero((char *)(&range_arg), sizeof(range_arg));
  range_arg.user = Padi_EMPTY_STR;
  range_arg.password = Padi_EMPTY_STR;
  range_arg.range.do_missing = TRUE;
  range_arg.range.missing_translation[0] = -9999999990.0;
  range_arg.range.missing_translation[1] = -9999999991.0;
  range_arg.range.missing_translation[2] = -9999999992.0;
  if (option.nobs) 
    range_arg.range.nobs = option.nobs;
      
  /*
  ** try to crash with bogus calls 
  */

  info_result = PadiGetInfo(NULL, NULL);
  PadiError(logfp, "GetInfo(NULL, NULL)", info_result->status, Padi_WARNING);
  PadiFreeResult((PadiResult_tp)info_result);
  memset(too_long_name, 'n', Padi_MAX_NAME_LEN<<1);
  too_long_name[Padi_MAX_NAME_LEN<<1] = '\0'; 
  too_long.name = too_long_name;
  too_long.port = option.port;
  info_result = PadiGetInfo(&too_long, &infoobject);
  PadiError(logfp, "GetInfo(&too_long, &infoobject)", info_result->status, Padi_WARNING);
  PadiFreeResult((PadiResult_tp)info_result);
  memset(too_long_name, 'n', Padi_MAX_NAME_LEN<<1);
  too_long_name[Padi_MAX_NAME_LEN<<1] = '\0'; 
  infoobject.object_name = too_long_name;
  info_result = PadiGetInfo(&too_long, &infoobject);
  PadiError(logfp, "GetInfo(&too_long, &infoobject)", info_result->status, Padi_WARNING);
  PadiFreeResult((PadiResult_tp)info_result);
  
  series_result = PadiGetSeries(NULL, NULL);
  PadiError(logfp, "GetSeries(NULL, NULL)", series_result->status, Padi_WARNING);
  PadiFreeResult((PadiResult_tp)series_result);
  memset(too_long_name, 'n', Padi_MAX_NAME_LEN<<1);
  too_long_name[Padi_MAX_NAME_LEN<<1] = '\0'; 
  memset(too_long_str, 's', Padi_MAX_STR_LEN<<1);
  too_long_str[Padi_MAX_STR_LEN<<1] = '\0'; 
  range_arg.object_name = NULL;
  series_result = PadiGetSeries(&too_long, &range_arg);
  PadiError(logfp, "GetSeries(&too_long, too_long_str)", series_result->status, Padi_WARNING);
  PadiFreeResult((PadiResult_tp)series_result);
  memset(too_long_name, 'n', Padi_MAX_NAME_LEN<<1);
  too_long_name[Padi_MAX_NAME_LEN<<1] = '\0'; 
  memset(too_long_str, 's', Padi_MAX_STR_LEN<<1);
  too_long_str[Padi_MAX_STR_LEN<<1] = '\0'; 
  range_arg.object_name = too_long_name;
  series_result = PadiGetSeries(&server, &range_arg);
  PadiError(logfp, "GetSeries(&server, too_long_name)", series_result->status, Padi_WARNING);
  PadiFreeResult((PadiResult_tp)series_result);

  
  /* test new */ 
  new_series.dbname = "bogus_database_name";
  result = PadiNewSeries(&server, &new_series);
  PadiError(logfp, "NewSeries(&server, bogus_database_name)", result->status, Padi_WARNING);
  PadiFreeResult((PadiResult_tp)result);
  new_series.dbname = "etsbcpi";
  result = PadiNewSeries(&server, &new_series);
  PadiError(logfp, "NewSeries(&server, etsbcpi)", result->status, Padi_WARNING);
  PadiFreeResult((PadiResult_tp)result);
  new_series.dbname = "compare";
  series->info.class = "SERIES";
  series->info.frequency = "ANNUAL";
  series->info.type = "PRECISION";
  result = PadiNewSeries(&server, &new_series);
  PadiError(logfp, "NewSeries(&server, new_series)", result->status, Padi_WARNING);
  PadiFreeResult((PadiResult_tp)result);
  result = PadiNewSeries(&server, &new_series);
  PadiError(logfp, "NewSeries(&server, new_series)", result->status, Padi_WARNING);
  PadiFreeResult((PadiResult_tp)result);



  /* test destroy */
/*   result = PadiDestroy(&server, "COMPARE.C3.MONTHLY"); */
/*   PadiError(logfp, "Destroy(&server, COMPARE.C3.MONTHLY)",  */
/*             result->status, Padi_WARNING); */
/*   result = PadiDestroy(&server, "ETSBCPI.C1.DAILY"); */
/*   PadiError(logfp, "Destroy(&server, ETSBCPI.C1.DAILY)",  */
/*             result->status, Padi_WARNING); */
/*   result = PadiDestroy(&server, "new_series"); */
/*   PadiError(logfp, "Destroy(&server, new_series)",  */
/*             result->status, Padi_WARNING); */
/*   PadiFreeResult((PadiResult_tp)result); */
/*   info_result = PadiGetInfo(&server, "new_series"); */
/*   PadiError(logfp, "GetInfo(&server, new_series)",  */
/*             info_result->status, Padi_WARNING); */
/*   PadiFreeResult((PadiResult_tp)info_result); */

  PadiTerminate(&termobject);

  return Padi_SUCCEED;      
}

#endif

#ifdef PADI_SVC

typedef PadiResult_tp (*PadiLocalFunc_tp)();

export void padiprog_1 PARAM2(struct svc_req *, rqstp, SVCXPRT *, transp)
{
  PadiResult_tp result;
  PadiLocalFunc_tp local;
  bool_t (*xdr_argument)(), (*xdr_result)();
  string_t proc_name, object_name;
  char log_buf[512];
  struct authunix_parms *unix_cred;
  union {
    PadiInfoArg_t getinfo_1_arg;
    PadiRangeArg_t getseries_1_arg;
    PadiNewSeries_t newseries_1_arg;
    PadiDestroyArg_t destroy_1_arg;
  } argument;

  unix_cred = (rqstp->rq_cred.oa_flavor == AUTH_UNIX) ? 
                (struct authunix_parms *)(rqstp->rq_clntcred) :
                NULL;

  switch (rqstp->rq_proc) {
  case NULLPROC:
    (void) svc_sendreply(transp, (xdrproc_t) xdr_void, (char *)NULL);
    return;

 
  case GETINFO:
    xdr_argument = xdr_PadiInfoArg_t;
    xdr_result = xdr_PadiInfoResult_t;
    local = (PadiLocalFunc_tp)getinfo_1;
    break;
 
  case GETSERIES:
    xdr_argument = xdr_PadiRangeArg_t;
    xdr_result = xdr_PadiSeriesResult_t;

#ifdef PUBLIC
    local = (PadiLocalFunc_tp)getseries_1;
#else
    local = (PadiLocalFunc_tp)getlocal_1;
#endif /* PUBLIC */
    break;
 
  case NEWSERIES:
    xdr_argument = xdr_PadiNewSeries_t;
    xdr_result = xdr_PadiResult_t;
#ifdef PUBLIC
    local = (PadiLocalFunc_tp)newseries_1;
#else
    local = (PadiLocalFunc_tp)putlocal_1;
#endif /* PUBLIC */
    break;
 
  case DESTROY:
    xdr_argument = xdr_PadiDestroyArg_t;
    xdr_result = xdr_PadiResult_t;
    local = (PadiLocalFunc_tp)destroy_1;
    break;
 
  default:
    svcerr_noproc(transp);
    return;

  }

  bzero((char *)&argument, sizeof(argument));
  if (!svc_getargs(transp, (xdrproc_t) xdr_argument, (caddr_t)(&argument))) {
    svcerr_decode(transp);
    return;

  }

  switch (rqstp->rq_proc) {
 
  case GETINFO:
    proc_name = "GetInfo";
    object_name = argument.getinfo_1_arg.object_name;
    break;
 
  case GETSERIES:
    proc_name = "GetSeries";
    object_name = argument.getseries_1_arg.object_name;
    break;
 
  case NEWSERIES:
    proc_name = "NewSeries";
    object_name = argument.newseries_1_arg.series.info.name;
    break;

  case DESTROY:
    proc_name = "Destroy";
    object_name = argument.destroy_1_arg.object_name;
    break;

  }

  if (unix_cred) 
    sprintf(log_buf, "%d.%d@%s: %s(%.256s)", unix_cred->aup_uid, 
                                         unix_cred->aup_gid, 
                                         unix_cred->aup_machname, 
                                         proc_name, object_name);
  else
    sprintf(log_buf, "%s(%s)", proc_name, object_name);

  PadiError(logfp, log_buf, Padi_SUCCEED, Padi_WARNING);

  if (!(result = (*local)(&argument, rqstp))) {
    PadiError(logfp, proc_name, Padi_OUT_OF_MEMORY, Padi_FATAL);

  }

  if (result->status) 
    PadiError(logfp, proc_name, result->status, Padi_ERROR);

  if (!svc_sendreply(transp, (xdrproc_t) xdr_result, (caddr_t)result))
    svcerr_systemerr(transp);

  if (!svc_freeargs(transp, (xdrproc_t) xdr_argument, (caddr_t)(&argument))) {
    PadiError(logfp, proc_name, Padi_FREE_ARG, Padi_FATAL);

  }

  PadiFreeResult(result);
}

export int main PARAM2(int, argc, char **, argv)
/*
** RPC Server main() for the Data Base Application Programer's Interface.
**
** RETURN
**   Zero <==> success.  
**   Non-zero <==> failure or qualified success.
**
** NOTES
**
**   Make RPC Server main with:
**
**   cc -g -I$(FAME)/hli -DPADI_SVC padi.c hlifunc.o -L$(FAME)/hli -lchli  -lm -lc -o padi
**
** FILES
**   padi.c, padi.h
*/
{
#ifndef L_cuserid
#define L_cuserid 9
#endif
  PadiStatus_t status;
  PadiInitArg_t initobject;
  PadiTermArg_t termobject;
  mainopt_t option;
  PadiServer_t server;
  /* SVCXPRT *transp; */
  char log_buf[128];
  int i;
  char    *svcuser;
  struct passwd *ps;
  app_name = *argv;

  if( strrchr( app_name, '/') != NULL )
  {
     app_name = strrchr( app_name, '/')+1;
  }


  GetArg(argc, argv, &option);

  if (!(option.logfile)) {
    strcpy(log_buf, app_name);
    option.logfile = strcat(log_buf, ".log");
  }

  if (!(logfp = fopen(option.logfile, "a"))) {
    sprintf(log_buf, "%s (%s)", app_name, option.logfile);
    PadiError(stderr, log_buf, Padi_FILE_OPEN, Padi_FATAL);

  }

  PadiVerbose = option.verbose;

  ps = getpwuid(geteuid());
  svcuser = ps->pw_name;
  sprintf(log_buf,"PADI Server started by %s", svcuser);
  PadiError(logfp, log_buf, Padi_SUCCEED, Padi_WARNING);

   initobject.user = Padi_EMPTY_STR;
   initobject.password = Padi_EMPTY_STR;
  initobject.object_dbname = option.objectdb;

  if ((status = PadiInitialize(&initobject))) {
    sprintf(log_buf, "%s (%s)", app_name, option.objectdb);
    PadiError(logfp, log_buf, status, Padi_FATAL);
  
  }

#ifndef NOTRAP
  for (i = 0; i <= SIGUSR2; i++) {
    switch (i) {
    case SIGKILL:
    case SIGSTOP:
    case SIGABRT:
    case SIGTSTP:
    case SIGCONT:
    case SIGIO:
    case SIGURG:
    case SIGWINCH:
      break;

    default:
      signal(i, sighandler);
    }
  }
#endif
 
  server.name = NULL;
  server.port = option.port;
  if ((status = PadiServe(&server, padiprog_1))) {
    PadiError(logfp, app_name, status, Padi_FATAL);
 
  }

   termobject.user = Padi_EMPTY_STR;
   termobject.password = Padi_EMPTY_STR;
   PadiTerminate(&termobject);

  PadiError(logfp, "PADI Server end", Padi_SUCCEED, Padi_WARNING);

  return Padi_SUCCEED;      
}

#endif /* PADI_MAIN ==  */

