% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RVineSeqEst.R
\name{RVineSeqEst}
\alias{RVineSeqEst}
\title{Sequential Estimation of an R-Vine Copula Model}
\usage{
RVineSeqEst(
  data,
  RVM,
  method = "mle",
  se = FALSE,
  max.df = 30,
  max.BB = list(BB1 = c(5, 6), BB6 = c(6, 6), BB7 = c(5, 6), BB8 = c(6, 1)),
  progress = FALSE,
  weights = NA,
  cores = 1
)
}
\arguments{
\item{data}{An N x d data matrix (with uniform margins).}

\item{RVM}{An \code{\link[=RVineMatrix]{RVineMatrix()}} object including the structure, the
pair-copula families and the pair-copula parameters (if they are known).}

\item{method}{indicates the estimation method: either maximum
likelihood estimation (\code{method = "mle"}; default) or inversion of
Kendall's tau (\code{method = "itau"}). For \code{method = "itau"} only
one parameter families and the Student t copula can be used (\verb{family = 1,2,3,4,5,6,13,14,16,23,24,26,33,34} or \code{36}). For the t-copula,
\code{par2} is found by a crude profile likelihood optimization over the
interval (2, 10].}

\item{se}{Logical; whether standard errors are estimated (default: \code{se = FALSE}).}

\item{max.df}{Numeric; upper bound for the estimation of the degrees of
freedom parameter of the t-copula (default: \code{max.df = 30}; for more
details see \code{\link[=BiCopEst]{BiCopEst()}}).}

\item{max.BB}{List; upper bounds for the estimation of the two parameters
(in absolute values) of the BB1, BB6, BB7 and BB8 copulas \cr (default:
\code{max.BB = list(BB1=c(5,6),BB6=c(6,6),BB7=c(5,6),BB8=c(6,1))}).}

\item{progress}{Logical; whether the pairwise estimation progress is printed
(default: \code{progress = FALSE}).}

\item{weights}{Numerical; weights for each observation (optional).}

\item{cores}{integer; if \code{cores > 1}, estimation will be parallelized
within each tree (using \code{\link[foreach:foreach]{foreach::foreach()}}). However, the
overhead caused by parallelization is likely to make the function run slower
unless sample size is really large and \code{method = "itau"}.}
}
\value{
An \code{\link[=RVineMatrix]{RVineMatrix()}} object with the sequentially
estimated parameters stored in \code{RVM$par} and \code{RVM$par2}. The object
is augmented by the following information about the fit:
\item{se, se2}{standard errors for the parameter estimates (if
\code{se = TRUE}); note that these are only approximate since they do not
account for the sequential nature of the estimation,}
\item{nobs}{number of observations,}
\item{logLik, pair.logLik}{log likelihood (overall and pairwise)}
\item{AIC, pair.AIC}{Aikaike's Informaton Criterion (overall and pairwise),}
\item{BIC, pair.BIC}{Bayesian's Informaton Criterion (overall and pairwise),}
\item{emptau}{matrix of empirical values of Kendall's tau,}
\item{p.value.indeptest}{matrix of p-values of the independence test.}
}
\description{
This function sequentially estimates the pair-copula parameters of a
d-dimensional R-vine copula model as specified by the corresponding
\code{\link[=RVineMatrix]{RVineMatrix()}} object.
}
\details{
The pair-copula parameter estimation is performed tree-wise, i.e., for each
R-vine tree the results from the previous tree(s) are used to calculate the
new copula parameters using \code{\link[=BiCopEst]{BiCopEst()}}.
}
\note{
For a comprehensive summary of the fitted model, use
\code{summary(object)}; to see all its contents, use \code{str(object)}.
}
\examples{

# define 5-dimensional R-vine tree structure matrix
Matrix <- c(5, 2, 3, 1, 4,
            0, 2, 3, 4, 1,
            0, 0, 3, 4, 1,
            0, 0, 0, 4, 1,
            0, 0, 0, 0, 1)
Matrix <- matrix(Matrix, 5, 5)

# define R-vine pair-copula family matrix
family <- c(0, 1, 3, 4, 4,
            0, 0, 3, 4, 1,
            0, 0, 0, 4, 1,
            0, 0, 0, 0, 3,
            0, 0, 0, 0, 0)
family <- matrix(family, 5, 5)

# define R-vine pair-copula parameter matrix
par <- c(0, 0.2, 0.9, 1.5, 3.9,
         0, 0, 1.1, 1.6, 0.9,
         0, 0, 0, 1.9, 0.5,
         0, 0, 0, 0, 4.8,
         0, 0, 0, 0, 0)
par <- matrix(par, 5, 5)

# define second R-vine pair-copula parameter matrix
par2 <- matrix(0, 5, 5)

# define RVineMatrix object
RVM <- RVineMatrix(Matrix = Matrix, family = family,
                   par = par, par2 = par2,
                   names = c("V1", "V2", "V3", "V4", "V5"))

# simulate a sample of size 300 from the R-vine copula model
set.seed(123)
simdata <- RVineSim(300, RVM)

# sequential estimation
summary(RVineSeqEst(simdata, RVM, method = "itau", se = TRUE))
summary(RVineSeqEst(simdata, RVM, method = "mle", se = TRUE))

}
\seealso{
\code{\link[=RVineMatrix]{RVineMatrix()}},
\code{\link[=BiCop]{BiCop()}},
\code{\link[=BiCopEst]{BiCopEst()}},
\code{\link[=plot.RVineMatrix]{plot.RVineMatrix()}},
\code{\link[=contour.RVineMatrix]{contour.RVineMatrix()}},
\code{\link[foreach:foreach]{foreach::foreach()}}
}
\author{
Ulf Schepsmeier, Jeffrey Dissmann, Thomas Nagler
}
