% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xGRviaGeneAnno.r
\name{xGRviaGeneAnno}
\alias{xGRviaGeneAnno}
\title{Function to conduct region-based enrichment analysis using nearby gene annotations}
\usage{
xGRviaGeneAnno(data.file, background.file = NULL,
format.file = c("data.frame", "bed", "chr:start-end", "GRanges"),
build.conversion = c(NA, "hg38.to.hg19", "hg18.to.hg19"), gap.max = 0,
GR.Gene = c("UCSC_knownGene", "UCSC_knownCanonical"), ontology =
c("GOBP",
"GOMF", "GOCC", "PS", "PS2", "SF", "Pfam", "DO", "HPPA", "HPMI",
"HPCM",
"HPMA", "MP", "EF", "MsigdbH", "MsigdbC1", "MsigdbC2CGP",
"MsigdbC2CPall",
"MsigdbC2CP", "MsigdbC2KEGG", "MsigdbC2REACTOME", "MsigdbC2BIOCARTA",
"MsigdbC3TFT", "MsigdbC3MIR", "MsigdbC4CGN", "MsigdbC4CM",
"MsigdbC5BP",
"MsigdbC5MF", "MsigdbC5CC", "MsigdbC6", "MsigdbC7", "DGIdb", "GTExV4",
"GTExV6", "CreedsDisease", "CreedsDiseaseUP", "CreedsDiseaseDN",
"CreedsDrug",
"CreedsDrugUP", "CreedsDrugDN", "CreedsGene", "CreedsGeneUP",
"CreedsGeneDN", "KEGG", "KEGGmetabolism", "KEGGgenetic",
"KEGGenvironmental",
"KEGGcellular", "KEGGorganismal", "KEGGdisease", "REACTOME",
"REACTOME_ImmuneSystem", "REACTOME_SignalTransduction", "CGL"),
size.range = c(10, 2000), min.overlap = 3, which.distance = NULL,
test = c("hypergeo", "fisher", "binomial"), p.tail = c("one-tail",
"two-tails"), p.adjust.method = c("BH", "BY", "bonferroni", "holm",
"hochberg", "hommel"), ontology.algorithm = c("none", "pc", "elim",
"lea"),
elim.pvalue = 0.01, lea.depth = 2, path.mode = c("all_paths",
"shortest_paths", "all_shortest_paths"), true.path.rule = F, verbose =
T,
RData.location = "http://galahad.well.ox.ac.uk/bigdata")
}
\arguments{
\item{data.file}{an input data file, containing a list of genomic
regions to test. If the input file is formatted as a 'data.frame'
(specified by the parameter 'format.file' below), the first three
columns correspond to the chromosome (1st column), the starting
chromosome position (2nd column), and the ending chromosome position
(3rd column). If the format is indicated as 'bed' (browser extensible
data), the same as 'data.frame' format but the position is 0-based
offset from chromomose position. If the genomic regions provided are
not ranged but only the single position, the ending chromosome position
(3rd column) is allowed not to be provided. If the format is indicated
as "chr:start-end", instead of using the first 3 columns, only the
first column will be used and processed. If the file also contains
other columns, these additional columns will be ignored. Alternatively,
the input file can be the content itself assuming that input file has
been read. Note: the file should use the tab delimiter as the field
separator between columns}

\item{background.file}{an input background file containing a list of
genomic regions as the test background. The file format is the same as
'data.file'. By default, it is NULL meaning all annotatable bases (ig
non-redundant bases covered by 'annotation.file') are used as
background. However, if only one annotation (eg only a transcription
factor) is provided in 'annotation.file', the background must be
provided}

\item{format.file}{the format for input files. It can be one of
"data.frame", "chr:start-end", "bed" or "GRanges"}

\item{build.conversion}{the conversion from one genome build to
another. The conversions supported are "hg38.to.hg19" and
"hg18.to.hg19". By default it is NA (no need to do so)}

\item{gap.max}{the maximum distance to nearby genes. Only those genes
no far way from this distance will be considered as nearby genes. By
default, it is 0 meaning that nearby genes are those overlapping with
genomic regions}

\item{GR.Gene}{the genomic regions of genes. By default, it is
'UCSC_knownGene', that is, UCSC known genes (together with genomic
locations) based on human genome assembly hg19. It can be
'UCSC_knownCanonical', that is, UCSC known canonical genes (together
with genomic locations) based on human genome assembly hg19.
Alternatively, the user can specify the customised input. To do so,
first save your RData file (containing an GR object) into your local
computer, and make sure the GR object content names refer to Gene
Symbols. Then, tell "GR.Gene" with your RData file name (with or
without extension), plus specify your file RData path in
"RData.location"}

\item{ontology}{the ontology supported currently. It can be "GOBP" for
Gene Ontology Biological Process, "GOMF" for Gene Ontology Molecular
Function, "GOCC" for Gene Ontology Cellular Component, "PS" for
phylostratific age information, "PS2" for the collapsed PS version
(inferred ancestors being collapsed into one with the known taxonomy
information), "SF" for SCOP domain superfamilies, "Pfam" for Pfam
domain families, "DO" for Disease Ontology, "HPPA" for Human Phenotype
Phenotypic Abnormality, "HPMI" for Human Phenotype Mode of Inheritance,
"HPCM" for Human Phenotype Clinical Modifier, "HPMA" for Human
Phenotype Mortality Aging, "MP" for Mammalian Phenotype, "EF" for
Experimental Factor Ontology (used to annotate GWAS Catalog genes),
Drug-Gene Interaction database ("DGIdb") for druggable categories,
tissue-specific eQTL-containing genes from GTEx ("GTExV4", "GTExV6p"
and "GTExV7"), crowd extracted expression of differential signatures
from CREEDS ("CreedsDisease", "CreedsDiseaseUP", "CreedsDiseaseDN",
"CreedsDrug", "CreedsDrugUP", "CreedsDrugDN", "CreedsGene",
"CreedsGeneUP" and "CreedsGeneDN"), KEGG pathways (including 'KEGG' for
all, 'KEGGmetabolism' for 'Metabolism' pathways, 'KEGGgenetic' for
'Genetic Information Processing' pathways, 'KEGGenvironmental' for
'Environmental Information Processing' pathways, 'KEGGcellular' for
'Cellular Processes' pathways, 'KEGGorganismal' for 'Organismal
Systems' pathways, and 'KEGGdisease' for 'Human Diseases' pathways),
'REACTOME' for REACTOME pathways or 'REACTOME_x' for its sub-ontologies
(where x can be 'CellCellCommunication', 'CellCycle',
'CellularResponsesToExternalStimuli', 'ChromatinOrganization',
'CircadianClock', 'DevelopmentalBiology', 'DigestionAndAbsorption',
'Disease', 'DNARepair', 'DNAReplication',
'ExtracellularMatrixOrganization', 'GeneExpression(Transcription)',
'Hemostasis', 'ImmuneSystem', 'Metabolism', 'MetabolismOfProteins',
'MetabolismOfRNA', 'Mitophagy', 'MuscleContraction', 'NeuronalSystem',
'OrganelleBiogenesisAndMaintenance', 'ProgrammedCellDeath',
'Reproduction', 'SignalTransduction', 'TransportOfSmallMolecules',
'VesicleMediatedTransport'), and the molecular signatures database
(Msigdb, including "MsigdbH", "MsigdbC1", "MsigdbC2CGP",
"MsigdbC2CPall", "MsigdbC2CP", "MsigdbC2KEGG", "MsigdbC2REACTOME",
"MsigdbC2BIOCARTA", "MsigdbC3TFT", "MsigdbC3MIR", "MsigdbC4CGN",
"MsigdbC4CM", "MsigdbC5BP", "MsigdbC5MF", "MsigdbC5CC", "MsigdbC6",
"MsigdbC7")}

\item{size.range}{the minimum and maximum size of members of each term
in consideration. By default, it sets to a minimum of 10 but no more
than 2000}

\item{min.overlap}{the minimum number of overlaps. Only those terms
with members that overlap with input data at least min.overlap (3 by
default) will be processed}

\item{which.distance}{which terms with the distance away from the
ontology root (if any) is used to restrict terms in consideration. By
default, it sets to 'NULL' to consider all distances}

\item{test}{the test statistic used. It can be "fisher" for using
fisher's exact test, "hypergeo" for using hypergeometric test, or
"binomial" for using binomial test. Fisher's exact test is to test the
independence between gene group (genes belonging to a group or not) and
gene annotation (genes annotated by a term or not), and thus compare
sampling to the left part of background (after sampling without
replacement). Hypergeometric test is to sample at random (without
replacement) from the background containing annotated and non-annotated
genes, and thus compare sampling to background. Unlike hypergeometric
test, binomial test is to sample at random (with replacement) from the
background with the constant probability. In terms of the ease of
finding the significance, they are in order: hypergeometric test >
fisher's exact test > binomial test. In other words, in terms of the
calculated p-value, hypergeometric test < fisher's exact test <
binomial test}

\item{p.tail}{the tail used to calculate p-values. It can be either
"two-tails" for the significance based on two-tails (ie both over- and
under-overrepresentation) or "one-tail" (by default) for the
significance based on one tail (ie only over-representation)}

\item{p.adjust.method}{the method used to adjust p-values. It can be
one of "BH", "BY", "bonferroni", "holm", "hochberg" and "hommel". The
first two methods "BH" (widely used) and "BY" control the false
discovery rate (FDR: the expected proportion of false discoveries
amongst the rejected hypotheses); the last four methods "bonferroni",
"holm", "hochberg" and "hommel" are designed to give strong control of
the family-wise error rate (FWER). Notes: FDR is a less stringent
condition than FWER}

\item{ontology.algorithm}{the algorithm used to account for the
hierarchy of the ontology. It can be one of "none", "pc", "elim" and
"lea". For details, please see 'Note' below}

\item{elim.pvalue}{the parameter only used when "ontology.algorithm" is
"elim". It is used to control how to declare a signficantly enriched
term (and subsequently all genes in this term are eliminated from all
its ancestors)}

\item{lea.depth}{the parameter only used when "ontology.algorithm" is
"lea". It is used to control how many maximum depth is used to consider
the children of a term (and subsequently all genes in these children
term are eliminated from the use for the recalculation of the
signifance at this term)}

\item{path.mode}{the mode of paths induced by vertices/nodes with input
annotation data. It can be "all_paths" for all possible paths to the
root, "shortest_paths" for only one path to the root (for each node in
query), "all_shortest_paths" for all shortest paths to the root (i.e.
for each node, find all shortest paths with the equal lengths)}

\item{true.path.rule}{logical to indicate whether the true-path rule
should be applied to propagate annotations. By default, it sets to
false}

\item{verbose}{logical to indicate whether the messages will be
displayed in the screen. By default, it sets to false for no display}

\item{RData.location}{the characters to tell the location of built-in
RData files. See \code{\link{xRDataLoader}} for details}
}
\value{
an object of class "eTerm", a list with following components:
\itemize{
\item{\code{term_info}: a matrix of nTerm X 4 containing snp/gene set
information, where nTerm is the number of terms, and the 4 columns are
"id" (i.e. "Term ID"), "name" (i.e. "Term Name"), "namespace" and
"distance"}
\item{\code{annotation}: a list of terms containing annotations, each
term storing its annotations. Always, terms are identified by "id"}
\item{\code{g}: an igraph object to represent DAG}
\item{\code{data}: a vector containing input data in consideration. It
is not always the same as the input data as only those mappable are
retained}
\item{\code{background}: a vector containing the background data. It is
not always the same as the input data as only those mappable are
retained}
\item{\code{overlap}: a list of overlapped snp/gene sets, each storing
snps overlapped between a snp/gene set and the given input data (i.e.
the snps of interest). Always, gene sets are identified by "id"}
\item{\code{fc}: a vector containing fold changes}
\item{\code{zscore}: a vector containing z-scores}
\item{\code{pvalue}: a vector containing p-values}
\item{\code{adjp}: a vector containing adjusted p-values. It is the p
value but after being adjusted for multiple comparisons}
\item{\code{or}: a vector containing odds ratio}
\item{\code{CIl}: a vector containing lower bound confidence interval
for the odds ratio}
\item{\code{CIu}: a vector containing upper bound confidence interval
for the odds ratio}
\item{\code{cross}: a matrix of nTerm X nTerm, with an on-diagnal cell
for the overlapped-members observed in an individaul term, and
off-diagnal cell for the overlapped-members shared betwene two terms}
\item{\code{call}: the call that produced this result}
}
}
\description{
\code{xGRviaGeneAnno} is supposed to conduct region-based enrichment
analysis for the input genomic region data (genome build h19), using
nearby gene annotations. To do so, nearby genes are first defined
within the maximum gap between genomic regions and gene location.
Enrichment analysis is based on either Fisher's exact test or
Hypergeometric test for estimating the significance of overlapped
nearby genes. Test background can be provided; by default, the
annotatable genes will be used.
}
\note{
The interpretation of the algorithms used to account for the hierarchy
of the ontology is:
\itemize{
\item{"none": does not consider the ontology hierarchy at all.}
\item{"lea": computers the significance of a term in terms of the
significance of its children at the maximum depth (e.g. 2). Precisely,
once snps are already annotated to any children terms with a more
signficance than itself, then all these snps are eliminated from the
use for the recalculation of the signifance at that term. The final
p-values takes the maximum of the original p-value and the recalculated
p-value.}
\item{"elim": computers the significance of a term in terms of the
significance of its all children. Precisely, once snps are already
annotated to a signficantly enriched term under the cutoff of e.g.
pvalue<1e-2, all these snps are eliminated from the ancestors of that
term).}
\item{"pc": requires the significance of a term not only using the
whole snps as background but also using snps annotated to all its
direct parents/ancestors as background. The final p-value takes the
maximum of both p-values in these two calculations.}
\item{"Notes": the order of the number of significant terms is: "none"
> "lea" > "elim" > "pc".}
}
}
\examples{
\dontrun{
# Load the XGR package and specify the location of built-in data
library(XGR)
RData.location <- "http://galahad.well.ox.ac.uk/bigdata_dev"

# Enrichment analysis for GWAS SNPs from ImmunoBase
## a) provide input data
data.file <- "http://galahad.well.ox.ac.uk/bigdata/ImmunoBase_GWAS.bed"

## b) perform DO enrichment analysis for nearby genes (with GWAS SNPs)
eTerm <- xGRviaGeneAnno(data.file=data.file, format.file="bed",
gap.max=0, ontology="DO", RData.location=RData.location)

## c) view enrichment results for the top significant terms
xEnrichViewer(eTerm)

## d) save enrichment results to the file called 'Regions2genes_enrichments.txt'
output <- xEnrichViewer(eTerm, top_num=length(eTerm$adjp),
sortBy="adjp", details=TRUE)
utils::write.table(output, file="Regions2genes_enrichments.txt",
sep="\\t", row.names=FALSE)

## e) barplot of significant enrichment results
bp <- xEnrichBarplot(eTerm, top_num=10, displayBy="fc")
print(bp)
}
}
\seealso{
\code{\link{xEnrichViewer}}, \code{\link{xEnricherGenes}}
}
