% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/abclass.R
\name{abclass}
\alias{abclass}
\alias{abclass.control}
\title{Angle-Based Classification}
\usage{
abclass(
  x,
  y,
  intercept = TRUE,
  weight = NULL,
  loss = c("logistic", "boost", "hinge-boost", "lum"),
  control = list(),
  ...
)

abclass.control(
  lambda = NULL,
  alpha = 0.5,
  nlambda = 50,
  lambda_min_ratio = NULL,
  grouped = TRUE,
  group_weight = NULL,
  group_penalty = c("lasso", "scad", "mcp"),
  dgamma = 1,
  nfolds = 0,
  stratified_cv = TRUE,
  alignment = c("fraction", "lambda"),
  lum_a = 1,
  lum_c = 1,
  boost_umin = -5,
  maxit = 1e+05,
  epsilon = 0.001,
  standardize = TRUE,
  varying_active_set = TRUE,
  verbose = 0,
  ...
)
}
\arguments{
\item{x}{A numeric matrix representing the design matrix.  No missing valus
are allowed.  The coefficient estimates for constant columns will be
zero.  Thus, one should set the argument \code{intercept} to \code{TRUE}
to include an intercept term instead of adding an all-one column to
\code{x}.}

\item{y}{An integer vector, a character vector, or a factor vector
representing the response label.}

\item{intercept}{A logical value indicating if an intercept should be
considered in the model.  The default value is \code{TRUE} and the
intercept is excluded from regularization.}

\item{weight}{A numeric vector for nonnegative observation weights. Equal
observation weights are used by default.}

\item{loss}{A character value specifying the loss function.  The available
options are \code{"logistic"} for the logistic deviance loss,
\code{"boost"} for the exponential loss approximating Boosting machines,
\code{"hinge-boost"} for hybrid of SVM and AdaBoost machine, and
\code{"lum"} for largin-margin unified machines (LUM).  See Liu, et
al. (2011) for details.}

\item{control}{A list of control parameters. See \code{abclass.control()}
for details.}

\item{...}{Other control parameters passed to \code{abclass.control}.}

\item{lambda}{A numeric vector specifying the tuning parameter \emph{lambda}
of elastic-net penalty.  A data-driven \emph{lambda} sequence will be
generated and used according to specified \code{alpha}, \code{nlambda}
and \code{lambda_min_ratio} if this argument is \code{NULL} by default.
The specified \code{lambda} will be sorted in decreasing order
internally and only the unique values will be kept.}

\item{alpha}{A numeric value in [0, 1] representing the mixing parameter
\emph{alpha} in elastic-net penalty.}

\item{nlambda}{A positive integer specifying the length of the internally
generated \emph{lambda} sequence.  This argument will be ignored if a
valid \code{lambda} is specified.  The default value is \code{50}.}

\item{lambda_min_ratio}{A positive number specifying the ratio of the
smallest lambda parameter to the largest lambda parameter.  The default
value is set to \code{1e-4} if the sample size is larger than the number
of predictors, and \code{1e-2} otherwise.}

\item{grouped}{A logicial value.  Experimental flag to apply group Lasso.}

\item{group_weight}{A numerical vector with nonnegative values representing
the adaptive penalty factors for grouped lasso.}

\item{group_penalty}{A character vector specifying the name of the group
penalty.}

\item{dgamma}{A positive number specifying the increment to the minimal
gamma parameter for group SCAD or group MCP.}

\item{nfolds}{A nonnegative integer specifying the number of folds for
cross-validation.  The default value is \code{0} and no cross-validation
will be performed if \code{nfolds < 2}.}

\item{stratified_cv}{A logical value indicating if the cross-validation
procedure should be stratified by the response label. The default value
is \code{TRUE}.}

\item{alignment}{A character vector specifying how to align the lambda
sequence used in the main fit with the cross-validation fits.  The
available options are \code{"fraction"} for allowing cross-validation
fits to have their own lambda sequences and \code{"lambda"} for using
the same lambda sequence of the main fit.  The option \code{"lambda"}
will be applied if a meaningful \code{lambda} is specified.  The default
value is \code{"fraction"}.}

\item{lum_a}{A positive number greater than one representing the parameter
\emph{a} in LUM, which will be used only if \code{loss = "lum"}.  The
default value is \code{1.0}.}

\item{lum_c}{A nonnegative number specifying the parameter \emph{c} in LUM,
which will be used only if \code{loss = "hinge-boost"} or \code{loss =
"lum"}.  The default value is \code{1.0}.}

\item{boost_umin}{A negative number for adjusting the boosting loss for the
internal majorization procedure.}

\item{maxit}{A positive integer specifying the maximum number of
iteration.  The default value is \code{10^5}.}

\item{epsilon}{A positive number specifying the relative tolerance that
determines convergence.  The default value is \code{1e-3}.}

\item{standardize}{A logical value indicating if each column of the design
matrix should be standardized internally to have mean zero and standard
deviation equal to the sample size.  The default value is \code{TRUE}.
Notice that the coefficient estimates are always returned on the
original scale.}

\item{varying_active_set}{A logical value indicating if the active set
should be updated after each cycle of coordinate-majorization-descent
algorithm.  The default value is \code{TRUE} for usually more efficient
estimation procedure.}

\item{verbose}{A nonnegative integer specifying if the estimation procedure
should print out intermediate steps/results.  The default value is
\code{0} for silent estimation procedure.}
}
\value{
The function \code{abclass()} returns an object of class
    \code{abclass} representing a trained classifier; The function
    \code{abclass.control()} returns an object of class {abclass.control}
    representing a list of control parameters.
}
\description{
Multi-category angle-based large-margin classifiers with regularization by
the elastic-net penalty.
}
\examples{
library(abclass)
set.seed(123)

## toy examples for demonstration purpose
## reference: example 1 in Zhang and Liu (2014)
ntrain <- 100 # size of training set
ntest <- 100  # size of testing set
p0 <- 5       # number of actual predictors
p1 <- 5       # number of random predictors
k <- 5        # number of categories

n <- ntrain + ntest; p <- p0 + p1
train_idx <- seq_len(ntrain)
y <- sample(k, size = n, replace = TRUE)         # response
mu <- matrix(rnorm(p0 * k), nrow = k, ncol = p0) # mean vector
## normalize the mean vector so that they are distributed on the unit circle
mu <- mu / apply(mu, 1, function(a) sqrt(sum(a ^ 2)))
x0 <- t(sapply(y, function(i) rnorm(p0, mean = mu[i, ], sd = 0.25)))
x1 <- matrix(rnorm(p1 * n, sd = 0.3), nrow = n, ncol = p1)
x <- cbind(x0, x1)
train_x <- x[train_idx, ]
test_x <- x[- train_idx, ]
y <- factor(paste0("label_", y))
train_y <- y[train_idx]
test_y <- y[- train_idx]

## Regularization through ridge penalty
model1 <- abclass(train_x, train_y, nlambda = 5, nfolds = 3,
                  loss = "logistic", alpha = 0, lambda_min_ratio = 1e-2)
pred1 <- predict(model1, test_x)
table(test_y, pred1)
mean(test_y == pred1) # accuracy

## groupwise regularization via group lasso
model2 <- abclass(train_x, train_y, nlambda = 5, nfolds = 3,
                  grouped = TRUE, loss = "boost")
pred2 <- predict(model2, test_x)
table(test_y, pred2)
mean(test_y == pred2) # accuracy
}
\references{
Zhang, C., & Liu, Y. (2014). Multicategory Angle-Based Large-Margin
Classification. \emph{Biometrika}, 101(3), 625--640.

Liu, Y., Zhang, H. H., & Wu, Y. (2011). Hard or soft classification?
large-margin unified machines. \emph{Journal of the American Statistical
Association}, 106(493), 166--177.
}
