#' fit an adaptive lasso with adaptive weights derived from CISL
#'
#' Compute the CISL procedure (see \code{cisl} for more details) to determine
#' adaptive penalty weights, then run an adaptive lasso with this penalty weighting.
#' BIC is used for the adaptive lasso for variable selection.
#' Can deal with very large sparse data matrices.
#' Intended for binary reponse only (option \code{family = "binomial"} is forced).
#' Depends on the \code{glmnet} function from the package \code{glmnet}.
#'
#' The CISL procedureis first implemented with its default value except for
#' \code{dfmax} and \code{nlambda} through parameters \code{cisl_dfmax} and
#' \code{cisl_nlambda}.
#' In addition, the \code{betaPos} parameter is set to FALSE in \code{cisl}.
#' For each covariate \eqn{i}, 100 values of the CISL quantity \eqn{\tau_i}
#' are estimated.
#' The adaptive weight for a given covariate \eqn{i} is defined by
#' \deqn{w_i = 1- 1/100 \sum_b=1, .., 100 indicator [ \tau^b_i >0  ]}
#' If \eqn{\tau_i} is the null vector, the associated adaptve weights in infinty.
#' If \eqn{\tau_i} is always positive, rather than "forcing" the variable into
#' the model, we set the corresponding adaptive weight to \eqn{1/100}.
#'
#' @param x Input matrix, of dimension nobs x nvars. Each row is an observation
#' vector. Can be in sparse matrix format (inherit from class
#' \code{"sparseMatrix"} as in package \code{Matrix}).
#' @param y Binary response variable, numeric.
#' @param cisl_nB \code{nB} option in \code{cisl} function
#' @param cisl_dfmax \code{dfmax} option in \code{cisl} function
#' @param cisl_nlambda \code{nlambda} option in \code{cisl} function
#' @param cisl_ncore \code{ncore} option in \code{cisl} function
#' @param maxp A limit on how many relaxed coefficients are allowed.
#' Default is 50, in \code{glmnet} option default is 'n-3', where 'n' is the sample size.
#' @param path Since \code{glmnet} does not do stepsize optimization, the Newton
#' algorithm can get stuck and not converge, especially with relaxed fits. With \code{path=TRUE},
#' each relaxed fit on a particular set of variables is computed pathwise using the original sequence
#' of lambda values (with a zero attached to the end). Default is \code{path=TRUE}.
#' @param betaPos Should the covariates selected by the procedure be
#' positively associated with the outcome ? Default is \code{TRUE}.
#' @param \dots Other arguments that can be passed to \code{glmnet}
#' from package \code{glmnet} other than \code{penalty.factor},
#' \code{family}, \code{maxp} and \code{path}.
#'
#' @return An object with S3 class \code{"adaptive"}.
#' \item{aws}{Numeric vector of penalty weights derived from CISL.
#' Length equal to nvars.}
#' \item{criterion}{Character, indicates which criterion is used with the
#' adaptive lasso for variable selection. For \code{adapt_cisl} function,
#' \code{criterion} is "bic".}
#' \item{beta}{Numeric vector of regression coefficients in the adaptive lasso.
#' If \code{criterion} = "cv" the regression coefficients are PENALIZED, if
#' \code{criterion} = "bic" the regression coefficients are UNPENALIZED.
#' Length equal to nvars. Could be NA if adaptive weights are all equal to infinity.}
#' \item{selected_variables}{Character vector, names of variable(s) selected
#' with this adaptive approach.
#' If \code{betaPos = TRUE}, this set is the covariates with a positive regression
#' coefficient in \code{beta}.
#' Else this set is the covariates with a non null regression coefficient in \code{beta}.
#' Covariates are ordering according to the p-values (two-sided if \code{betaPos = FALSE} ,
#' one-sided if \code{betaPos = TRUE}) in the classical multiple logistic regression
#' model that minimzes the BIC in the adaptive lasso.}
#' @examples
#'
#' set.seed(15)
#' drugs <- matrix(rbinom(100*20, 1, 0.2), nrow = 100, ncol = 20)
#' colnames(drugs) <- paste0("drugs",1:ncol(drugs))
#' ae <- rbinom(100, 1, 0.3)
#' acisl <- adapt_cisl(x = drugs, y = ae, cisl_nB = 50, maxp=10)
#'
#'
#' @author Emeline Courtois \cr Maintainer: Emeline Courtois
#' \email{emeline.courtois@@inserm.fr}
#' @export adapt_cisl


adapt_cisl <- function(x, y, cisl_nB = 100, cisl_dfmax = 50, cisl_nlambda = 250, cisl_ncore = 1,
                       maxp = 50, path = TRUE, betaPos = TRUE, ...){

  # CISL with cisl.betaPos = F -------
  cc <- cisl(x = x, y = y, nB = cisl_nB, dfmax = cisl_dfmax, nlambda = cisl_nlambda, ncore = cisl_ncore, betaPos = FALSE)

  # Derived apw -------
  apws <-  apply(cc$prob, 1, poidsCISL_v2)

  # lasso-bic on adaptive step ----
  # Adaptive step -----
  # lasso-bic
  ada.cisl <- lasso_bic(x = x, y = y, maxp = maxp, path = path,
                        betaPos = betaPos, penalty.factor = apws, ...)

  beta2 <- ada.cisl$beta
  select_var <- ada.cisl$selected_variables

  # Result -----
  res <- list()
  res$aws <- apws
  res$criterion <- "bic"
  res$beta <- beta2
  res$selected_variables <- select_var

  class(res) = "adaptive"

  return(res)


}
