// SPDX-FileCopyrightText: 2011-2012 Tasos Varoudis
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "shapegraph.h"

#include "axialpolygons.h"
#include "parsers/mapinfodata.h"
#include "tolerances.h"

#include "genlib/comm.h" // For communicator
#include "genlib/containerutils.h"
#include "genlib/readwritehelpers.h"

#include <cmath>
#include <float.h>
#include <time.h>

#ifndef _WIN32
#define _finite finite
#endif

////////////////////////////////////////////////////////////////////////////////////////////

ShapeGraph::ShapeGraph(const std::string &name, int type) : ShapeMap(name, type) {
    m_keyvertexcount = 0;
    m_hasgraph = true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////

void ShapeGraph::initialiseAttributesAxial() {
    m_attributes->clear();
    // note, expects these to be numbered 0, 1...
    m_attributes->insertOrResetLockedColumn(ShapeGraph::Column::CONNECTIVITY);
    m_attributes->insertOrResetLockedColumn(ShapeGraph::Column::LINE_LENGTH);
}

void ShapeGraph::makeConnections(const KeyVertices &keyvertices) {
    m_connectors.clear();
    m_links.clear();
    m_unlinks.clear();
    m_keyvertices.clear();

    // note, expects these to be numbered 0, 1...
    auto conn_col = m_attributes->getColumnIndex(ShapeGraph::Column::CONNECTIVITY);
    auto leng_col = m_attributes->getColumnIndex(ShapeGraph::Column::LINE_LENGTH);

    size_t i = 0;
    for (const auto &shape : m_shapes) {
        int key = shape.first;
        AttributeRow &row = m_attributes->getRow(AttributeKey(key));
        // all indices should match...
        m_connectors.push_back(Connector());
        m_connectors[i].m_connections =
            getLineConnections(key, TOLERANCE_B * __max(m_region.height(), m_region.width()));
        row.setValue(conn_col, float(m_connectors[i].m_connections.size()));
        row.setValue(leng_col, float(shape.second.getLine().length()));
        if (keyvertices.size()) {
            // note: depends on lines being recorded in same order as keyvertices...
            m_keyvertices.push_back(keyvertices[i]);
        }
        i++;
    }
}

/////////////////////////////////////////////////////////////////////////////////////////

bool ShapeGraph::outputMifPolygons(std::ostream &miffile, std::ostream &midfile) const {
    // take lines from lines layer and make into regions (using the axial polygons)
    std::vector<Line> lines;
    for (const auto &shape : m_shapes) {
        lines.push_back(shape.second.getLine());
    }
    AxialPolygons polygons;
    polygons.init(lines, m_region);

    std::vector<std::vector<Point2f>> newpolygons;
    polygons.makePolygons(newpolygons);

    MapInfoData mapinfodata;
    if (m_hasMapInfoData) {
        mapinfodata.m_coordsys = m_mapinfodata.m_coordsys;
        mapinfodata.m_bounds = m_mapinfodata.m_bounds;
    }
    mapinfodata.exportPolygons(miffile, midfile, newpolygons, m_region);

    return true;
}

void ShapeGraph::outputNet(std::ostream &netfile) const {
    double maxdim = __max(m_region.width(), m_region.height());
    Point2f offset = Point2f((maxdim - m_region.width()) / (2.0 * maxdim),
                             (maxdim - m_region.height()) / (2.0 * maxdim));
    if (isSegmentMap()) {
        netfile << "*Vertices " << m_shapes.size() * 2 << std::endl;
        int i = -1;
        for (const auto &shape : m_shapes) {
            i++;
            Line li = shape.second.getLine();
            Point2f p1 = li.start();
            Point2f p2 = li.end();
            p1.x = offset.x + (p1.x - m_region.bottom_left.x) / maxdim;
            p2.x = offset.x + (p2.x - m_region.bottom_left.x) / maxdim;
            p1.y = 1.0 - (offset.y + (p1.y - m_region.bottom_left.y) / maxdim);
            p2.y = 1.0 - (offset.y + (p2.y - m_region.bottom_left.y) / maxdim);
            netfile << (i * 2 + 1) << " \"" << i << "a\" " << p1.x << " " << p1.y << std::endl;
            netfile << (i * 2 + 2) << " \"" << i << "b\" " << p2.x << " " << p2.y << std::endl;
        }
        netfile << "*Edges" << std::endl;
        for (size_t i = 0; i < m_shapes.size(); i++) {
            netfile << (i * 2 + 1) << " " << (i * 2 + 2) << " 2" << std::endl;
        }
        netfile << "*Arcs" << std::endl;
        // this makes an assumption about which is the "start" and which is the "end"
        // it works for an automatically converted axial map, I'm not sure it works for others...
        for (size_t j = 0; j < m_connectors.size(); j++) {
            const Connector &conn = m_connectors[j];
            for (auto &segconn : conn.m_forward_segconns) {
                SegmentRef ref = segconn.first;
                float weight = segconn.second;
                netfile << (j * 2 + 1) << " " << (ref.ref * 2 + ((ref.dir == 1) ? 1 : 2)) << " "
                        << weight << std::endl;
            }
            for (auto &segconn : conn.m_back_segconns) {
                SegmentRef ref = segconn.first;
                float weight = segconn.second;
                netfile << (j * 2 + 2) << " " << (ref.ref * 2 + ((ref.dir == 1) ? 1 : 2)) << " "
                        << weight << std::endl;
            }
        }
    } else {
        netfile << "*Vertices " << m_shapes.size() << std::endl;
        int i = -1;
        for (const auto &shape : m_shapes) {
            i++;
            Point2f p = shape.second.getCentroid();
            p.x = offset.x + (p.x - m_region.bottom_left.x) / maxdim;
            p.y = 1.0 - (offset.y + (p.y - m_region.bottom_left.y) / maxdim);
            netfile << (i + 1) << " \"" << i << "\" " << p.x << " " << p.y << std::endl;
        }
        netfile << "*Edges" << std::endl;
        for (size_t j = 0; j < m_connectors.size(); j++) {
            const Connector &conn = m_connectors[j];
            for (size_t k = 0; k < conn.m_connections.size(); k++) {
                auto to = conn.m_connections[k];
                if (j < to) {
                    netfile << (j + 1) << " " << (to + 1) << " 1" << std::endl;
                }
            }
        }
    }
}

bool ShapeGraph::readShapeGraphData(std::istream &stream) {

    m_attributes->clear();
    m_connectors.clear();
    m_map_type = ShapeMap::EMPTYMAP;

    // note that keyvertexcount and keyvertices are different things! (length keyvertices not the
    // same as keyvertexcount!)
    stream.read((char *)&m_keyvertexcount, sizeof(m_keyvertexcount));
    int size;
    stream.read((char *)&size, sizeof(size));
    for (int i = 0; i < size; i++) {
        std::vector<int> tempVec;
        dXreadwrite::readIntoVector(stream, tempVec);
        m_keyvertices.push_back(std::set<int>(tempVec.begin(), tempVec.end()));
    }
    return true;
}

std::tuple<bool, bool, bool, int> ShapeGraph::read(std::istream &stream) {
    bool read = readShapeGraphData(stream);
    // now base class read:
    auto shapeMapReadResult = ShapeMap::read(stream);
    std::get<0>(shapeMapReadResult) = read && std::get<0>(shapeMapReadResult);

    return shapeMapReadResult;
}

bool ShapeGraph::writeShapeGraphData(std::ostream &stream) const {
    // note keyvertexcount and keyvertices are different things!  (length keyvertices not the same
    // as keyvertexcount!)
    stream.write((char *)&m_keyvertexcount, sizeof(m_keyvertexcount));
    auto size = m_keyvertices.size();
    stream.write((char *)&size, sizeof(static_cast<int>(size)));
    for (size_t i = 0; i < m_keyvertices.size(); i++) {
        dXreadwrite::writeVector(
            stream, std::vector<int>(m_keyvertices[i].begin(), m_keyvertices[i].end()));
    }
    return true;
}

bool ShapeGraph::write(std::ostream &stream, const std::tuple<bool, bool, int> &displayData) const {
    bool written = writeShapeGraphData(stream);
    // now simply run base class write:
    written = written & ShapeMap::write(stream, displayData);

    return written;
}

void ShapeGraph::writeAxialConnectionsAsDotGraph(std::ostream &stream) {
    const std::vector<Connector> &connectors = ShapeMap::getConnections();

    stream << "strict graph {" << std::endl;

    stream.precision(12);

    for (size_t i = 0; i < connectors.size(); i++) {
        const auto &connections = connectors[i].m_connections;
        for (auto connection : connections) {
            stream << "    " << i << " -- " << connection << std::endl;
        }
    }
    stream << "}" << std::endl;
}

void ShapeGraph::writeLinksUnlinksAsPairsCSV(std::ostream &stream, char delimiter) {
    stream.precision(12);

    stream << "refA" << delimiter << "refB" << delimiter << "link" << std::endl;

    for (auto &link : m_links) {
        stream << depthmapX::getMapAtIndex(m_shapes, static_cast<size_t>(link.a))->first
               << delimiter
               << depthmapX::getMapAtIndex(m_shapes, static_cast<size_t>(link.b))->first
               << delimiter << "1" << std::endl;
    }

    for (auto &unlink : m_unlinks) {
        stream << depthmapX::getMapAtIndex(m_shapes, static_cast<size_t>(unlink.a))->first
               << delimiter
               << depthmapX::getMapAtIndex(m_shapes, static_cast<size_t>(unlink.b))->first
               << delimiter << "0" << std::endl;
    }
}

void ShapeGraph::writeAxialConnectionsAsPairsCSV(std::ostream &stream) {
    const std::vector<Connector> &connectors = ShapeMap::getConnections();

    stream.precision(12);

    stream << "refA,refB" << std::endl;

    for (size_t i = 0; i < connectors.size(); i++) {
        auto &connections = connectors[i].m_connections;
        if (i != 0)
            stream << std::endl;
        for (auto iter = connections.begin(); iter != connections.end(); ++iter) {
            if (iter != connections.begin())
                stream << std::endl;
            stream << i << "," << *iter;
        }
    }
}

void ShapeGraph::writeSegmentConnectionsAsPairsCSV(std::ostream &stream) {
    const std::vector<Connector> &connectors = ShapeMap::getConnections();

    stream.precision(12);

    stream << "refA,refB,ss_weight,for_back,dir";

    // directed links
    for (size_t i = 0; i < connectors.size(); i++) {
        for (auto &segconn : connectors[i].m_forward_segconns) {
            stream << std::endl;
            stream << i << "," << segconn.first.ref << "," << segconn.second << "," << 0 // forward
                   << "," << int(segconn.first.dir);
        }

        for (auto &segconn : connectors[i].m_back_segconns) {
            stream << std::endl;
            stream << i << "," << segconn.first.ref << "," << segconn.second << "," << 1 // back
                   << "," << int(segconn.first.dir);
        }
    }
}

void ShapeGraph::unlinkAtPoint(const Point2f &unlinkPoint) {
    std::vector<Point2f> closepoints;
    std::vector<std::pair<size_t, size_t>> intersections;
    PixelRef pix = pixelate(unlinkPoint);
    std::vector<ShapeRef> &pix_shapes =
        m_pixel_shapes(static_cast<size_t>(pix.y), static_cast<size_t>(pix.x));
    auto iter = pix_shapes.begin();
    for (; iter != pix_shapes.end(); ++iter) {
        for (auto jter = iter; jter != pix_shapes.end(); ++jter) {
            auto aIter = m_shapes.find(int(iter->m_shape_ref));
            auto bIter = m_shapes.find(int(jter->m_shape_ref));
            auto a = static_cast<size_t>(std::distance(m_shapes.begin(), aIter));
            auto b = static_cast<size_t>(std::distance(m_shapes.begin(), bIter));
            auto &connections = m_connectors[size_t(a)].m_connections;
            if (aIter != m_shapes.end() && bIter != m_shapes.end() && aIter->second.isLine() &&
                bIter->second.isLine() &&
                std::find(connections.begin(), connections.end(), b) != connections.end()) {
                closepoints.push_back(intersection_point(aIter->second.getLine(),
                                                         bIter->second.getLine(), TOLERANCE_A));
                intersections.push_back(std::make_pair(a, b));
            }
        }
    }
    double mindist = -1.0;
    int minpair = -1;
    int j = 0;
    for (auto &closepoint : closepoints) {
        if (minpair == -1 || dist(unlinkPoint, closepoint) < mindist) {
            mindist = dist(unlinkPoint, closepoint);
            minpair = j;
        }
        j++;
    }
    if (minpair != -1) {
        auto &intersection = intersections[size_t(minpair)];
        unlinkShapes(intersection.first, intersection.second);
    } else {
        std::cerr << "eek!";
    }
}
////////////////////////////////////////////////////////////////////////////

// this unlink options was originally excised on the version 7 recode
// however, it is *very specific* to axial maps, and so have been reincluded here

void ShapeGraph::unlinkFromShapeMap(const ShapeMap &shapemap) {
    // used to make a shape map from every axial intersection,

    // find lines in rough vincinity of unlink point, and check for the closest
    // pair to unlink:

    const std::map<int, SalaShape> &polygons = shapemap.getAllShapes();
    for (const auto &polygon : polygons) {
        // just use the points:
        if (polygon.second.isPoint()) {
            unlinkAtPoint(polygon.second.getPoint());
        }
    }
}

///////////////////////////////////////////////////////////////////////////////

// Two ways to make a segment map

// Method 1: direct linkage of endpoints where they touch

void ShapeGraph::makeNewSegMap(Communicator *comm) {
    // now make a connection set from the ends of lines:
    struct LineConnector {
        const Line &m_line;
        Connector &m_connector;
        int m_index;
        LineConnector(const Line &line, Connector &connector, int index)
            : m_line(line), m_connector(connector), m_index(index) {}
    };

    std::vector<Connector> connectionset;
    for (auto &shape : m_shapes) {
        if (shape.second.isLine()) {
            connectionset.emplace_back();
        }
    }
    std::map<size_t, LineConnector> lineConnectors;
    auto connectionIter = connectionset.begin();
    int connectionIdx = 0;
    for (auto &shape : m_shapes) {
        if (shape.second.isLine()) {
            lineConnectors.insert(
                std::make_pair(shape.first, LineConnector(shape.second.getLine(), *connectionIter,
                                                          connectionIdx)));
            connectionIter++;
            connectionIdx++;
        }
    }

    time_t atime = 0;
    if (comm) {
        qtimer(atime, 0);
        comm->CommPostMessage(Communicator::NUM_RECORDS, lineConnectors.size());
    }

    double maxdim = __max(m_region.width(), m_region.height());

    size_t count = 0;
    for (auto &lineConnector_a : lineConnectors) {
        Connector &connectionset_a = lineConnector_a.second.m_connector;
        const Line &line_a = lineConnector_a.second.m_line;
        int idx_a = lineConnector_a.second.m_index;
        // n.b., vector() is based on t_start and t_end, so we must use t_start and t_end here and
        // throughout
        PixelRef pix1 = pixelate(line_a.t_start());
        std::vector<ShapeRef> &shapes1 =
            m_pixel_shapes(static_cast<size_t>(pix1.y), static_cast<size_t>(pix1.x));
        for (auto &shape : shapes1) {
            auto lineConnector_b = lineConnectors.find(shape.m_shape_ref);
            if (lineConnector_b != lineConnectors.end() &&
                idx_a < lineConnector_b->second.m_index) {

                Connector &connectionset_b = lineConnector_b->second.m_connector;
                const Line &line_b = lineConnector_b->second.m_line;
                int idx_b = lineConnector_b->second.m_index;

                Point2f alpha = line_a.vector();
                Point2f beta = line_b.vector();
                alpha.normalise();
                beta.normalise();
                if (approxeq(line_a.t_start(), line_b.t_start(), (maxdim * TOLERANCE_B))) {
                    float x = float(2.0 * acos(__min(__max(-dot(alpha, beta), -1.0), 1.0)) / M_PI);
                    depthmapX::addIfNotExists(connectionset_a.m_back_segconns, SegmentRef(1, idx_b),
                                              x);
                    depthmapX::addIfNotExists(connectionset_b.m_back_segconns, SegmentRef(1, idx_a),
                                              x);
                }
                if (approxeq(line_a.t_start(), line_b.t_end(), (maxdim * TOLERANCE_B))) {
                    float x = float(2.0 * acos(__min(__max(-dot(alpha, -beta), -1.0), 1.0)) / M_PI);
                    depthmapX::addIfNotExists(connectionset_a.m_back_segconns,
                                              SegmentRef(-1, idx_b), x);
                    depthmapX::addIfNotExists(connectionset_b.m_forward_segconns,
                                              SegmentRef(1, idx_a), x);
                }
            }
        }

        PixelRef pix2 = pixelate(line_a.t_end());
        std::vector<ShapeRef> &shapes2 =
            m_pixel_shapes(static_cast<size_t>(pix2.y), static_cast<size_t>(pix2.x));
        for (auto &shape : shapes2) {
            auto lineConnector_b = lineConnectors.find(shape.m_shape_ref);
            if (lineConnector_b != lineConnectors.end() &&
                idx_a < lineConnector_b->second.m_index) {

                Connector &connectionset_b = lineConnector_b->second.m_connector;
                const Line &line_b = lineConnector_b->second.m_line;
                int idx_b = lineConnector_b->second.m_index;

                Point2f alpha = line_a.vector();
                Point2f beta = line_b.vector();

                alpha.normalise();
                beta.normalise();
                if (approxeq(line_a.t_end(), line_b.t_start(), (maxdim * TOLERANCE_B))) {
                    float x = float(2.0 * acos(__min(__max(-dot(-alpha, beta), -1.0), 1.0)) / M_PI);
                    depthmapX::addIfNotExists(connectionset_a.m_forward_segconns,
                                              SegmentRef(1, idx_b), x);
                    depthmapX::addIfNotExists(connectionset_b.m_back_segconns,
                                              SegmentRef(-1, idx_a), x);
                }
                if (approxeq(line_a.t_end(), line_b.t_end(), (maxdim * TOLERANCE_B))) {
                    float x =
                        float(2.0 * acos(__min(__max(-dot(-alpha, -beta), -1.0), 1.0)) / M_PI);
                    depthmapX::addIfNotExists(connectionset_a.m_forward_segconns,
                                              SegmentRef(-1, idx_b), x);
                    depthmapX::addIfNotExists(connectionset_b.m_forward_segconns,
                                              SegmentRef(-1, idx_a), x);
                }
            }
        }

        if (comm) {
            if (qtimer(atime, 500)) {
                if (comm->IsCancelled()) {
                    throw Communicator::CancelledException();
                }
                comm->CommPostMessage(Communicator::CURRENT_RECORD, count);
            }
        }
        count++;
    }

    // initialise attributes now separated from making the connections
    makeSegmentConnections(connectionset);
}

// Method 2: Making a segment map (in two stages)

// One: take the original axial map and split it up
// (note: you need to start from an axial map,
//  but the map could have been created from a road-centre-line
//  graph or equivalent -- reason is that you might want to
//  preserve unlinks in your angular mapping)

// A "linetest" is used in order to use the test component to
// identify the original axial line this line segment is
// associated with

void ShapeGraph::makeSegmentMap(std::vector<Line> &lines, std::vector<Connector> &connectors,
                                double stubremoval) {
    // the first (key) pair is the line / line intersection, second is the pair of associated
    // segments for the first line
    std::map<OrderedIntPair, std::pair<int, int>> segmentlist;

    // this code relies on the polygon order being the same as the connections

    auto iter = m_shapes.begin();
    for (size_t i = 0; i < m_connectors.size(); i++) {
        auto shape = iter->second;
        int axialRef = iter->first;
        iter++;
        if (!shape.isLine()) {
            continue;
        }
        const Line &line = shape.getLine();
        std::vector<std::pair<double, int>> breaks; // this is a vector instead of a map because the
                                                    // original code allowed for duplicate keys
        int axis = line.width() >= line.height() ? XAXIS : YAXIS;
        // we need the breaks ordered from start to end of the line
        // this is automatic for XAXIS, but on YAXIS, need to know
        // if the line is ascending or decending
        int parity = (axis == XAXIS) ? 1 : line.sign();

        auto &connections = m_connectors[i].m_connections;
        for (size_t j = 0; j < connections.size(); j++) {
            // find the intersection point and add...
            // note: more than one break at the same place allowed
            auto shapeJ =
                depthmapX::getMapAtIndex(m_shapes, static_cast<size_t>(connections[j]))->second;
            if (i != connections[j] && shapeJ.isLine()) {
                breaks.push_back(std::make_pair(
                    parity * line.intersection_point(shapeJ.getLine(), axis, TOLERANCE_A),
                    connections[j]));
            }
        }
        std::sort(breaks.begin(), breaks.end());
        // okay, now we have a list from one end of the other of lines this line connects with
        Point2f lastpoint = line.start();
        int seg_a = -1, seg_b = -1;
        double neardist;
        // TOLERANCE_C is introduced as of 01.08.2008 although it is a fix to a bug first
        // found in July 2006.  It has been set "high" deliberately (1e-6 = a millionth of the line
        // height / width) in order to catch small errors made by operators or floating point errors
        // in other systems when drawing, for example, three axial lines intersecting
        if (stubremoval == 0.0) {
            // if 0, convert to tolerance
            stubremoval = TOLERANCE_C;
        }
        neardist = (axis == XAXIS) ? (line.width() * stubremoval) : (line.height() * stubremoval);
        double overlapdist =
            (axis == XAXIS) ? (line.width() * TOLERANCE_C) : (line.height() * TOLERANCE_C);
        //
        for (auto breaksIter = breaks.begin(); breaksIter != breaks.end();) {
            std::vector<int> keylist;
            if (seg_a == -1) {
                Point2f thispoint = line.point_on_line(parity * breaksIter->first, axis);
                if (fabs(parity * breaksIter->first - line.start()[axis]) < neardist) {
                    seg_a = -1;
                    lastpoint = thispoint;
                } else {
                    Line segment_a(line.start(), thispoint);
                    lines.push_back(segment_a);
                    connectors.push_back(Connector(axialRef));
                    seg_a = static_cast<int>(lines.size()) - 1;
                }
                lastpoint = thispoint;
            }
            //
            double here = parity * breaksIter->first;
            while (breaksIter != breaks.end() &&
                   fabs(parity * breaksIter->first - here) < overlapdist) {
                keylist.push_back(breaksIter->second);
                ++breaksIter;
            }
            //
            if (breaksIter == breaks.end() &&
                fabs(line.end()[axis] - parity * breaks.rbegin()->first) < neardist) {
                seg_b = -1;
            } else {
                Point2f thispoint;
                if (breaksIter != breaks.end()) {
                    thispoint = line.point_on_line(parity * breaksIter->first, axis);
                } else {
                    thispoint = line.end();
                }
                Line segment_b(lastpoint, thispoint);
                lines.push_back(segment_b);
                connectors.push_back(Connector(axialRef));
                seg_b = static_cast<int>(lines.size()) - 1;
                //
                lastpoint = thispoint;
            }
            //
            for (size_t j = 0; j < keylist.size(); j++) {
                //
                if (keylist[j] < static_cast<int>(i)) {
                    // other line already segmented, look up in segment list,
                    // and join segments together nicely
                    auto segIter =
                        segmentlist.find(OrderedIntPair(keylist[j], static_cast<int>(i)));

                    if (segIter !=
                        segmentlist.end()) { // <- if it isn't -1 something has gone badly wrong!
                        int seg_1 = segIter->second.first;
                        int seg_2 = segIter->second.second;
                        if (seg_a != -1) {
                            if (seg_1 != -1) {
                                Point2f alpha =
                                    lines[size_t(seg_a)].start() - lines[size_t(seg_a)].end();
                                Point2f beta =
                                    lines[size_t(seg_1)].start() - lines[size_t(seg_1)].end();
                                alpha.normalise();
                                beta.normalise();
                                float x = float(
                                    2.0 * acos(__min(__max(-dot(alpha, beta), -1.0), 1.0)) / M_PI);
                                depthmapX::addIfNotExists(
                                    connectors[size_t(seg_a)].m_forward_segconns,
                                    SegmentRef(-1, seg_1), x);
                                depthmapX::addIfNotExists(
                                    connectors[size_t(seg_1)].m_forward_segconns,
                                    SegmentRef(-1, seg_a), x);
                            }
                            if (seg_2 != -1) {
                                Point2f alpha =
                                    lines[size_t(seg_a)].start() - lines[size_t(seg_a)].end();
                                Point2f beta =
                                    lines[size_t(seg_2)].end() - lines[size_t(seg_2)].start();
                                alpha.normalise();
                                beta.normalise();
                                float x = float(
                                    2.0 * acos(__min(__max(-dot(alpha, beta), -1.0), 1.0)) / M_PI);
                                depthmapX::addIfNotExists(
                                    connectors[size_t(seg_a)].m_forward_segconns,
                                    SegmentRef(1, seg_2), x);
                                depthmapX::addIfNotExists(connectors[size_t(seg_2)].m_back_segconns,
                                                          SegmentRef(-1, seg_a), x);
                            }
                        }
                        if (seg_b != -1) {
                            if (seg_1 != -1) {
                                Point2f alpha =
                                    lines[size_t(seg_b)].end() - lines[size_t(seg_b)].start();
                                Point2f beta =
                                    lines[size_t(seg_1)].start() - lines[size_t(seg_1)].end();
                                alpha.normalise();
                                beta.normalise();
                                float x = float(
                                    2.0 * acos(__min(__max(-dot(alpha, beta), -1.0), 1.0)) / M_PI);
                                depthmapX::addIfNotExists(connectors[size_t(seg_b)].m_back_segconns,
                                                          SegmentRef(-1, seg_1), x);
                                depthmapX::addIfNotExists(
                                    connectors[size_t(seg_1)].m_forward_segconns,
                                    SegmentRef(1, seg_b), x);
                            }
                            if (seg_2 != -1) {
                                Point2f alpha =
                                    lines[size_t(seg_b)].end() - lines[size_t(seg_b)].start();
                                Point2f beta =
                                    lines[size_t(seg_2)].end() - lines[size_t(seg_2)].start();
                                alpha.normalise();
                                beta.normalise();
                                float x = float(
                                    2.0 * acos(__min(__max(-dot(alpha, beta), -1.0), 1.0)) / M_PI);
                                depthmapX::addIfNotExists(connectors[size_t(seg_b)].m_back_segconns,
                                                          SegmentRef(1, seg_2), x);
                                depthmapX::addIfNotExists(connectors[size_t(seg_2)].m_back_segconns,
                                                          SegmentRef(1, seg_b), x);
                            }
                        }
                    }
                } else {
                    // other line still to be segmented, add ourselves to segment list
                    // to be added later
                    segmentlist.insert(
                        std::make_pair(OrderedIntPair(static_cast<int>(i), keylist[j]),
                                       std::pair<int, int>(seg_a, seg_b)));
                }
            }
            if (seg_a != -1 && seg_b != -1) {
                depthmapX::addIfNotExists(connectors[size_t(seg_a)].m_forward_segconns,
                                          SegmentRef(1, seg_b), 0.0f);
                depthmapX::addIfNotExists(connectors[size_t(seg_b)].m_back_segconns,
                                          SegmentRef(-1, seg_a), 0.0f);
            }
            seg_a = seg_b;
        }
    }
}

void ShapeGraph::initialiseAttributesSegment() {
    m_attributes->clear();

    // note, expects these in alphabetical order to preserve numbering:
    m_attributes->insertOrResetLockedColumn(Column::AXIAL_LINE_REF);
    m_attributes->insertOrResetLockedColumn(Column::SEGMENT_LENGTH);
}

// now segments and connections are listed separately...
// put them together in a new map

void ShapeGraph::makeSegmentConnections(std::vector<Connector> &connectionset) {
    m_connectors.clear();

    // note, expects these in alphabetical order to preserve numbering:
    auto w_conn_col = m_attributes->getOrInsertColumn(Column::ANGULAR_CONNECTIVITY);
    auto uw_conn_col = m_attributes->getOrInsertLockedColumn(Column::CONNECTIVITY);

    auto ref_col = m_attributes->getColumnIndex(Column::AXIAL_LINE_REF);
    auto leng_col = m_attributes->getColumnIndex(Column::SEGMENT_LENGTH);

    int i = -1;
    for (const auto &shape : m_shapes) {
        i++;
        Connector &connector = connectionset[size_t(i)];
        AttributeRow &row = m_attributes->getRow(AttributeKey(shape.first));

        row.setValue(ref_col, float(connector.m_segment_axialref));
        row.setValue(leng_col, float(shape.second.getLine().length()));

        // all indices should match... (including lineset/connectionset versus m_shapes)
        m_connectors.push_back(connector);
        float total_weight = 0.0f;
        for (auto iter = connector.m_forward_segconns.begin();
             iter != connector.m_forward_segconns.end(); ++iter) {
            total_weight += iter->second;
        }
        for (auto iter = connector.m_back_segconns.begin(); iter != connector.m_back_segconns.end();
             ++iter) {
            total_weight += iter->second;
        }
        row.setValue(w_conn_col, float(total_weight));
        row.setValue(uw_conn_col,
                     float(connector.m_forward_segconns.size() + connector.m_back_segconns.size()));

        // free up connectionset as we go along:
        connectionset[size_t(i)] = Connector();
    }
}

// this pushes axial map values to a segment map
// the segment map is 'this', the axial map is passed:

void ShapeGraph::pushAxialValues(ShapeGraph &axialmap) {
    if (!m_attributes->hasColumn(Column::AXIAL_LINE_REF)) {
        // this should never happen
        // AT: I am converting this to throw an error
        throw depthmapX::RuntimeException("Axial line ref does not exist");
    }

    std::vector<size_t> colindices;
    for (size_t i = 0; i < axialmap.m_attributes->getNumColumns(); i++) {
        std::string colname = std::string("Axial ") + axialmap.m_attributes->getColumnName(i);
        colindices.push_back(m_attributes->getOrInsertColumn(colname));
    }
    for (auto iter = m_attributes->begin(); iter != m_attributes->end(); iter++) {
        int axialref = (int)iter->getRow().getValue(Column::AXIAL_LINE_REF);
        // P.K: The original code here got the index of the row, but the column
        // "Axial Line Ref" should actually contain keys, not indices
        AttributeRow &row = axialmap.m_attributes->getRow(AttributeKey(axialref));
        for (size_t k = 0; k < axialmap.m_attributes->getNumColumns(); k++) {
            float val = row.getValue(k);
            // need to look up the column index:
            iter->getRow().setValue(colindices[k], val);
        }
    }
}
