% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MixedEffectsSet.R
\name{MixedEffectsSet}
\alias{MixedEffectsSet}
\title{Create a set of mixed effects models}
\usage{
MixedEffectsSet(
  response_unit,
  covariate_units,
  parameter_names,
  predict_fn,
  model_specifications,
  predict_ranef,
  fixed_only = FALSE,
  descriptors = list(),
  covariate_definitions = list()
)
}
\arguments{
\item{response_unit}{A named list containing one element, with a name representing the response
variable and a value representing the units of the response variable
using the \code{units::as_units} function.}

\item{covariate_units}{A named list containing the covariate specifications, with names
representing the covariate name and the values representing the units of
the coavariate using the \code{units::as_units} function}

\item{parameter_names}{A character vector naming the columns in \code{model_specifications} that
represent the parameters}

\item{predict_fn}{A function that takes the covariate names as arguments and returns a
prediction of the response variable. This function should be vectorized.}

\item{model_specifications}{A dataframe such that each row of the dataframe provides model-level
descriptors and parameter estimates for that model. Models must be
uniquely identifiable using the descriptors. This is usually established
using the \code{load_parameter_frame()} function.}

\item{predict_ranef}{A function that predicts the random effects, takes any named covariates in
\code{covariate_units} as arguments}

\item{fixed_only}{A boolean value indicating if the model produces predictions using only
fixed effects. This is useful when publications do not provide sufficient
information to predict the random effects.}

\item{descriptors}{An optional named list of descriptors that describe the context of the
allometric model}

\item{covariate_definitions}{An optional named list of custom covariate definitions that will supersede
the definitions given by the variable naming system.}
}
\value{
An instance of MixedEffectsSet
}
\description{
A \code{MixedEffectsSet} represents a group of fixed-effects models that all have
the same functional structure. Fitting a large family of models (e.g., for
many different species) using the same functional structure is a common
pattern in allometric studies, and \code{MixedEffectsSet} facilitates the
installation of these groups of models by allowing the user to specify the
parameter estimates and descriptions in a dataframe or spreadsheet.
}
\details{
Because mixed-effects models already accommodate a grouping structure,
\code{MixedEffectsSet} tends to be a much rarer occurrence than \code{FixedEffectsSet}
and \code{MixedEffectsModel}.
}
\section{Slots}{

\describe{
\item{\code{parameters}}{A named list of parameters and their values}

\item{\code{predict_fn_populated}}{The prediction function populated with the
parameter values}

\item{\code{specification}}{A tibble::tbl_df of the model specification, which are
the parameters and the descriptors together}

\item{\code{predict_ranef}}{The function that predicts the random effects}

\item{\code{predict_ranef_populated}}{The function that predicts the random effects
populated with the fixed effect parameter estimates}

\item{\code{fixed_only}}{A boolean value indicating if the model produces predictions
using only fixed effects}

\item{\code{model_specifications}}{A \code{tibble::tbl_df} of model specifications, where
each row reprents one model identified with descriptors and containing the
parameter estimates.}
}}

\examples{
mixed_effects_set <- MixedEffectsSet(
  response_unit = list(
    vsia = units::as_units("ft^3")
  ),
  covariate_units = list(
    dsob = units::as_units("in")
  ),
  parameter_names = "a",
  predict_ranef = function(dsob, hst) {
    list(a_i = 1)
  },
  predict_fn = function(dsob) {
    (a + a_i) * dsob^2
  },
  model_specifications = tibble::tibble(a = c(1, 2))
)
}
