% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_covariates.R
\name{calculate_modis}
\alias{calculate_modis}
\title{Calculate MODIS product covariates in multiple CPU threads}
\usage{
calculate_modis(
  from = NULL,
  locs = NULL,
  locs_id = "site_id",
  radius = c(0L, 1000L, 10000L, 50000L),
  preprocess = process_modis_merge,
  name_covariates = NULL,
  subdataset = NULL,
  fun_summary = "mean",
  package_list_add = NULL,
  export_list_add = NULL,
  max_cells = 3e+07,
  geom = FALSE,
  ...
)
}
\arguments{
\item{from}{character. List of paths to MODIS/VIIRS files.}

\item{locs}{sf/SpatVector object. Unique locs where covariates
will be calculated.}

\item{locs_id}{character(1). Site identifier. Default is \code{"site_id"}}

\item{radius}{numeric. Radii to calculate covariates.
Default is \code{c(0, 1000, 10000, 50000)}.}

\item{preprocess}{function. Function to handle HDF files.}

\item{name_covariates}{character. Name header of covariates.
e.g., \code{"MOD_NDVIF_0_"}.
The calculated covariate names will have a form of
"\code{\{name_covariates\}\{zero-padded buffer radius in meters\}}",
e.g., 'MOD_NDVIF_0_50000' where 50 km radius circular buffer
was used to calculate mean NDVI value.}

\item{subdataset}{Indices, names, or search patterns for subdatasets.
Find detail usage of the argument in notes.}

\item{fun_summary}{character or function. Function to summarize
extracted raster values.}

\item{package_list_add}{character. A vector with package names to load
these in each thread. Note that \code{sf}, \code{terra}, \code{exactextractr},
\code{doParallel}, \code{parallelly} and \code{dplyr} are the default packages to be
loaded.}

\item{export_list_add}{character. A vector with object names to export
to each thread. It should be minimized to spare memory.}

\item{max_cells}{integer(1). Maximum number of cells to be read at once.
Higher values will expedite processing, but will increase memory usage.
Maximum possible value is \code{2^31 - 1}.
See \code{\link[exactextractr:exact_extract]{exactextractr::exact_extract}} for details.}

\item{geom}{FALSE/"sf"/"terra".. Should the function return with geometry?
Default is \code{FALSE}, options with geometry are "sf" or "terra". The
coordinate reference system of the \code{sf} or \code{SpatVector} is that of \code{from.}}

\item{...}{Arguments passed to \code{preprocess}.}
}
\value{
A data.frame or SpatVector with an attribute:
\itemize{
\item \code{attr(., "dates_dropped")}: Dates with insufficient tiles.
Note that the dates mean the dates with insufficient tiles,
not the dates without available tiles.
}
}
\description{
\code{calculate_modis} essentially runs \code{\link{calculate_modis_daily}} function
in each thread (subprocess). Based on daily resolution, each day's workload
will be distributed to each thread. With \code{product} argument,
the files are processed by a customized function where the unique structure
and/or characteristics of the products are considered.
}
\note{
Overall, this function and dependent routines assume that the file
system can handle concurrent access to the (network) disk by multiple
processes. File system characteristics, package versions, and hardware
settings and specification can affect the processing efficiency.
\code{locs} is expected to be convertible to \code{sf} object. \code{sf}, \code{SpatVector}, and
other class objects that could be converted to \code{sf} can be used.
Common arguments in \code{preprocess} functions such as \code{date} and \code{path} are
automatically detected and passed to the function. Please note that
\code{locs} here and \code{path} in \code{preprocess} functions are assumed to have a
standard naming convention of raw files from NASA.
The argument \code{subdataset} should be in a proper format
depending on \code{preprocess} function:
\itemize{
\item \code{process_modis_merge()}: Regular expression pattern.
e.g., \code{"^LST_"}
\item \code{process_modis_swath()}: Subdataset names.
e.g., \code{c("Cloud_Fraction_Day", "Cloud_Fraction_Night")}
\item \code{process_blackmarble()}: Subdataset number.
e.g., for VNP46A2 product, 3L.
Dates with less than 80 percent of the expected number of tiles,
which are determined by the mode of the number of tiles, are removed.
Users will be informed of the dates with insufficient tiles.
The result data.frame will have an attribute with the dates with
insufficient tiles.
}
}
\examples{
## NOTE: Example is wrapped in `\dontrun{}` as function requires a large
##       amount of data which is not included in the package.
\dontrun{
locs <- data.frame(lon = -78.8277, lat = 35.95013, id = "001")
locs <- terra::vect(locs, geom = c("lon", "lat"), crs = "EPSG:4326")
calculate_modis(
  from =
    list.files("./data", pattern = "VNP46A2.", full.names = TRUE),
  locs = locs,
  locs_id = "site_id",
  radius = c(0L, 1000L),
  preprocess = process_modis_merge,
  name_covariates = "cloud_fraction_0",
  subdataset = "Cloud_Fraction",
  fun_summary = "mean"
)
}
}
\seealso{
This function leverages the calculation of single-day MODIS
covariates:
\itemize{
\item \code{\link[=calculate_modis_daily]{calculate_modis_daily()}}
}

Also, for preprocessing, please refer to:
\itemize{
\item \code{\link[=process_modis_merge]{process_modis_merge()}}
\item \code{\link[=process_modis_swath]{process_modis_swath()}}
\item \code{\link[=process_blackmarble]{process_blackmarble()}}
}
}
