#' Modified AMMI Stability Value
#'
#' \code{MASV.AMMI} computes the Modified AMMI Stability Value (MASV) (Zali et
#' al., 2012; Please see \strong{Note}) from a modified formula of AMMI
#' Stability Value (ASV) (Purchase et al. 1997). This formula calculates AMMI
#' stability value considering all significant interaction principal components
#' (IPCs) in the AMMI model. Using MASV, the Simultaneous Selection Index for
#' Yield and Stability (SSI) is also calculated according to the argument
#' \code{ssi.method}.
#'
#' The Modified AMMI Stability Value
#' (\ifelse{html}{\out{<i>MASV</i>}}{\eqn{MASV}}) is computed as follows:
#'
#' \ifelse{html}{\out{<p style="text-align: center;"><i>MASV =
#' <big>&radic;[&sum;</big><sup>N'-1</sup><sub style="line-height: 1.8;
#' margin-left: -5ex;">n=1</sub>((<sup>SSIPC<sub>n</sub></sup> &frasl;
#' <sub>SSIPC<sub>n+1</sub></sub>) &times; PC<sub>1</sub>)<sup>2</sup> +
#' (PC<sub>N'</sub>)<sup>2</sup><big>]</big></i></p>}}{\deqn{MASV =
#' \sqrt{\sum_{n=1}^{N'-1}\left (\frac{SSIPC_{n}}{SSIPC_{n+1}} \times PC_{n}
#' \right )^2   + \left (PC_{N'}  \right )^2}}}
#'
#' Where, \ifelse{html}{\out{<i>SSIPC<sub>1</sub></i>}}{\eqn{SSIPC_{1}}},
#' \ifelse{html}{\out{<i>SSIPC<sub>2</sub></i>}}{\eqn{SSIPC_{2}}},
#' \ifelse{html}{\out{&hellip;}}{\eqn{\cdots}},
#' \ifelse{html}{\out{<i>SSIPC<sub>n</sub></i>}}{\eqn{SSIPC_{n}}} are the sum of
#' squares of the 1st, 2nd, ..., and \ifelse{html}{\out{<i>n</i>}}{\eqn{n}}th
#' IPC; and \ifelse{html}{\out{<i>PC<sub>1</sub></i>}}{\eqn{PC_{1}}},
#' \ifelse{html}{\out{<i>PC<sub>2</sub></i>}}{\eqn{PC_{2}}},
#' \ifelse{html}{\out{&hellip;}}{\eqn{\cdots}},
#' \ifelse{html}{\out{<i>PC<sub>n</sub></i>}}{\eqn{PC_{n}}} are the scores of
#' 1st, 2nd, ..., and \ifelse{html}{\out{<i>n</i>}}{\eqn{n}}th IPC.
#'
#' @note In Zali et al., (2012), the formula for both AMMI stability value (ASV)
#'   was found to be erroneous, when compared with the original publications
#'   (Purchase 1997; Purchase et al., 1999; Purchase et al., 2000).
#'
#'   \strong{ASV (Zali et al., 2012)} \ifelse{html}{\out{<p style="text-align:
#'   center;"><i>ASV = <big>&radic;[</big>(<sup>SSIPC<sub>1</sub></sup>
#'   <sub>&frasl; SSIPC<sub>2</sub></sub>) &times; (PC<sub>1</sub>)<sup>2</sup>
#'   + (PC<sub>2</sub>)<sup>2</sup><big>]</big></i></p>}}{\deqn{ASV =
#'   \sqrt{\left ( \frac{SSIPC_{1}}{SSIPC_{2}} \right ) \times (PC_{1})^2   +
#'   \left (PC_{2} \right )^2}}}
#'
#'   \strong{ASV (Purchase 1997; Purchase et al., 1999; Purchase et al., 2000)}
#'   \ifelse{html}{\out{<p style="text-align: center;"><i>ASV =
#'   <big>&radic;[</big>((<sup>SSIPC<sub>1</sub></sup> &frasl;
#'   <sub>SSIPC<sub>2</sub></sub>) &times; PC<sub>1</sub>)<sup>2</sup> +
#'   (PC<sub>2</sub>)<sup>2</sup><big>]</big></i></p>}}{\deqn{ASV = \sqrt{\left
#'   (\frac{SSIPC_{1}}{SSIPC_{2}} \times PC_{1}  \right )^2   + \left (PC_{2}
#'   \right )^2}}}
#'
#'   The authors believe that the proposed Modified AMMI stability value (MASV)
#'   in Zali et al., (2012) is also erroneous and have implemented the corrected
#'   one in \code{MASV.AMMI}.
#'
#'   \strong{MASV (Zali et al., 2012)} \ifelse{html}{\out{<p style="text-align:
#'   center;"><i>MASV = <big>&radic;[&sum;</big><sup>N'-1</sup><sub
#'   style="line-height: 1.8; margin-left:
#'   -5ex;">n=1</sub>(<sup>SSIPC<sub>n</sub></sup> &frasl;
#'   <sub>SSIPC<sub>n+1</sub></sub>) &times; (PC<sub>1</sub>)<sup>2</sup> +
#'   (PC<sub>N'</sub>)<sup>2</sup><big>]</big></i></p>}}{\deqn{MASV =
#'   \sqrt{\sum_{n=1}^{N'-1}\left ( \frac{SSIPC_{n}}{SSIPC_{n+1}} \right )
#'   \times (PC_{n})^2   + \left (PC_{N'}  \right )^2}}}
#'
#' @param model The AMMI model (An object of class \code{AMMI} generated by
#'   \code{\link[agricolae]{AMMI}}).
#' @param n The number of principal components to be considered for computation.
#'   The default value is the number of significant IPCs.
#' @param alpha Type I error probability (Significance level) to be considered
#'   to identify the number of significant IPCs.
#' @param ssi.method The method for the computation of simultaneous selection
#'   index. Either \code{"farshadfar"} or \code{"rao"} (See
#'   \code{\link[ammistability]{SSI}}).
#' @param a The ratio of the weights given to the stability components for
#'   computation of SSI when \code{method = "rao"} (See
#'   \code{\link[ammistability]{SSI}}).
#'
#' @return A data frame with the following columns:  \item{MASV}{The MASV
#'   values.} \item{SSI}{The computed values of simultaneous selection index for
#'   yield and stability.} \item{rMASV}{The ranks of MASV values.} \item{rY}{The
#'   ranks of the mean yield of genotypes.} \item{means}{The mean yield of the
#'   genotypes.}
#'
#'   The names of the genotypes are indicated as the row names of the data
#'   frame.
#'
#' @importFrom methods is
#' @importFrom stats aggregate
#' @importFrom agricolae AMMI
#' @export
#'
#' @references
#'
#' \insertRef{purchase_parametric_1997}{ammistability}
#'
#' \insertRef{purchase_use_1999}{ammistability}
#'
#' \insertRef{purchase_genotype_2000}{ammistability}
#'
#' \insertRef{zali_evaluation_2012}{ammistability}
#'
#' @seealso \code{\link[agricolae]{AMMI}}, \code{\link[agricolae]{index.AMMI}},
#'   \code{\link[ammistability]{SSI}}
#'
#' @examples
#' library(agricolae)
#' data(plrv)
#'
#' # AMMI model
#' model <- with(plrv, AMMI(Locality, Genotype, Rep, Yield, console = FALSE))
#'
#' # ANOVA
#' model$ANOVA
#'
#' # IPC F test
#' model$analysis
#'
#' # Mean yield and IPC scores
#' model$biplot
#'
#' # G*E matrix (deviations from mean)
#' array(model$genXenv, dim(model$genXenv), dimnames(model$genXenv))
#'
#' # With default n (N') and default ssi.method (farshadfar)
#' MASV.AMMI(model)
#'
#' # With n = 4 and default ssi.method (farshadfar)
#' MASV.AMMI(model, n = 4)
#'
#' # With default n (N') and ssi.method = "rao"
#' MASV.AMMI(model, ssi.method = "rao")
#'
#' # Changing the ratio of weights for Rao's SSI
#' MASV.AMMI(model, ssi.method = "rao", a = 0.43)
#'
MASV.AMMI <- function(model, n, alpha = 0.05,
                      ssi.method = c("farshadfar", "rao"), a = 1) {

  # Check model class
  if (!is(model, "AMMI")) {
    stop('"model" is not of class "AMMI"')
  }

  # Check alpha value
  if (!(0 < alpha && alpha < 1)) {
    stop('"alpha" should be between 0 and 1 (0 < alpha < 1)')
  }

  # Find number of significant IPCs according to F test
  if (missing(n) || is.null(n)) {
    n <- sum(model$analysis$Pr.F <= alpha, na.rm = TRUE)
  }

  # Check for n
  if (n %% 1 != 0 && length(n) != 1) {
    stop('"n" is not an integer vector of unit length')
  }

  # Check if n > N
  if (n > nrow(model$analysis)) {
    stop('"n" is greater than the number of IPCs in "model"')
  }

  ssi.method <- match.arg(ssi.method)

  # Fetch response (Yield)
  yresp <- setdiff(colnames(model$means), c("ENV", "GEN", "RESIDUAL"))

  A <- model$biplot
  A <- A[A[, 1] == "GEN", -c(1, 2)]
  A <- A[, 1:n] # Fetch only n IPCs

  MASV <- rep(0, nrow(A))

  for (i in seq_along(A)) {
    pc <- model$analysis[i, 4] / model$analysis[(i + 1), 4]
    MASV <- MASV + (A[, i] * pc)^2
    if ((i + 1) == max(seq_along(A))) break()
  }

  MASV <- MASV + (A[, max(seq_along(A))]^2)
  MASV <- sqrt(MASV)

  B <- model$means
  W <- aggregate(B[, yresp], by = list(model$means$GEN), FUN = mean, na.rm = TRUE)
  SSI_MASV <- SSI(y = W$x, sp = MASV, gen = W$Group.1,
                  method = ssi.method, a = a)
  ranking <- SSI_MASV
  colnames(ranking) <- c("MASV", "SSI", "rMASV", "rY", "means")

  return(ranking)
}
