% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orsf_pd.R
\name{orsf_pd_oob}
\alias{orsf_pd_oob}
\alias{orsf_pd_inb}
\alias{orsf_pd_new}
\title{ORSF partial dependence}
\usage{
orsf_pd_oob(
  object,
  pred_spec,
  pred_horizon = NULL,
  pred_type = "risk",
  expand_grid = TRUE,
  prob_values = c(0.025, 0.5, 0.975),
  prob_labels = c("lwr", "medn", "upr"),
  boundary_checks = TRUE,
  ...
)

orsf_pd_inb(
  object,
  pred_spec,
  pred_horizon = NULL,
  pred_type = "risk",
  expand_grid = TRUE,
  prob_values = c(0.025, 0.5, 0.975),
  prob_labels = c("lwr", "medn", "upr"),
  boundary_checks = TRUE,
  ...
)

orsf_pd_new(
  object,
  pred_spec,
  new_data,
  pred_horizon = NULL,
  pred_type = "risk",
  expand_grid = TRUE,
  prob_values = c(0.025, 0.5, 0.975),
  prob_labels = c("lwr", "medn", "upr"),
  boundary_checks = TRUE,
  ...
)
}
\arguments{
\item{object}{(\emph{orsf_fit}) a trained oblique random survival forest
(see \link{orsf}).}

\item{pred_spec}{(\emph{named list} or \emph{data.frame}).
\itemize{
\item If \code{pred_spec} is a named list,
Each item in the list should be a vector of values that will be used as
points in the partial dependence function. The name of each item in the
list should indicate which variable will be modified to take the
corresponding values.
\item If \code{pred_spec} is a \code{data.frame}, columns will
indicate variable names, values will indicate variable values, and
partial dependence will be computed using the inputs on each row.
}}

\item{pred_horizon}{(\emph{double}) a value or vector indicating the time(s)
that predictions will be calibrated to. E.g., if you were predicting
risk of incident heart failure within the next 10 years, then
\code{pred_horizon = 10}. \code{pred_horizon} can be \code{NULL} if \code{pred_type} is
\code{'mort'}, since mortality predictions are aggregated over all
event times}

\item{pred_type}{(\emph{character}) the type of predictions to compute. Valid
options are
\itemize{
\item 'risk' : probability of having an event at or before \code{pred_horizon}.
\item 'surv' : 1 - risk.
\item 'chf': cumulative hazard function
\item 'mort': mortality prediction
}}

\item{expand_grid}{(\emph{logical}) if \code{TRUE}, partial dependence will be
computed at all possible combinations of inputs in \code{pred_spec}. If
\code{FALSE}, partial dependence will be computed for each variable
in \code{pred_spec}, separately.}

\item{prob_values}{(\emph{numeric}) a vector of values between 0 and 1,
indicating what quantiles will be used to summarize the partial
dependence values at each set of inputs. \code{prob_values} should
have the same length as \code{prob_labels}. The quantiles are calculated
based on predictions from \code{object} at each set of values indicated
by \code{pred_spec}.}

\item{prob_labels}{(\emph{character}) a vector of labels with the same length
as \code{prob_values}, with each label indicating what the corresponding
value in \code{prob_values} should be labelled as in summarized outputs.
\code{prob_labels} should have the same length as \code{prob_values}.}

\item{boundary_checks}{(\emph{logical}) if \code{TRUE}, \code{pred_spec} will be vetted
to make sure the requested values are between the 10th and 90th
percentile in the object's training data. If \code{FALSE}, these checks are
skipped.}

\item{...}{Further arguments passed to or from other methods (not currently used).}

\item{new_data}{a \link{data.frame}, \link[tibble:tibble-package]{tibble}, or \link[data.table:data.table]{data.table} to compute predictions in.
Missing data are not currently allowed}
}
\value{
a \link[data.table:data.table]{data.table} containing
partial dependence values for the specified variable(s) at the
specified prediction horizon(s).
}
\description{
Compute partial dependence for an ORSF model.
Partial dependence (PD) shows the expected prediction from a model as a function of a single predictor or multiple predictors. The expectation is marginalized over the values of all other predictors, giving something like a multivariable adjusted estimate of the model's prediction.
You can compute partial dependence three ways using a random forest:
\itemize{
\item using in-bag predictions for the training data
\item using out-of-bag predictions for the training data
\item using predictions for a new set of data
}

See examples for more details
}
\section{Examples}{
Begin by fitting an ORSF ensemble:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(aorsf)

set.seed(329730)

index_train <- sample(nrow(pbc_orsf), 150) 

pbc_orsf_train <- pbc_orsf[index_train, ]
pbc_orsf_test <- pbc_orsf[-index_train, ]

fit <- orsf(data = pbc_orsf_train, 
            formula = Surv(time, status) ~ . - id,
            oobag_pred_horizon = 365.25 * 5)
}\if{html}{\out{</div>}}
\subsection{Three ways to compute PD and ICE}{

You can compute partial dependence and ICE three ways with \code{aorsf}:
\itemize{
\item using in-bag predictions for the training data

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pd_train <- orsf_pd_inb(fit, pred_spec = list(bili = 1:5))

pd_train
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##    pred_horizon  bili      mean        lwr       medn       upr
##           <num> <int>     <num>      <num>      <num>     <num>
## 1:      1826.25     1 0.2065186 0.01461416 0.09406926 0.8053158
## 2:      1826.25     2 0.2352372 0.02673697 0.12477942 0.8206148
## 3:      1826.25     3 0.2754197 0.04359767 0.17630939 0.8406553
## 4:      1826.25     4 0.3303309 0.09237920 0.24319095 0.8544871
## 5:      1826.25     5 0.3841395 0.15224112 0.30174988 0.8663482
}\if{html}{\out{</div>}}
\item using out-of-bag predictions for the training data

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pd_train <- orsf_pd_oob(fit, pred_spec = list(bili = 1:5))

pd_train
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##    pred_horizon  bili      mean        lwr       medn       upr
##           <num> <int>     <num>      <num>      <num>     <num>
## 1:      1826.25     1 0.2075896 0.01389732 0.09063976 0.7998756
## 2:      1826.25     2 0.2352634 0.02628113 0.12935779 0.8152149
## 3:      1826.25     3 0.2750782 0.04254451 0.18877830 0.8371582
## 4:      1826.25     4 0.3302680 0.08806724 0.24827784 0.8441472
## 5:      1826.25     5 0.3846734 0.14808075 0.29926304 0.8562432
}\if{html}{\out{</div>}}
\item using predictions for a new set of data

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pd_test <- orsf_pd_new(fit, 
                       new_data = pbc_orsf_test, 
                       pred_spec = list(bili = 1:5))

pd_test
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##    pred_horizon  bili      mean        lwr      medn       upr
##           <num> <int>     <num>      <num>     <num>     <num>
## 1:      1826.25     1 0.2541661 0.01581296 0.1912170 0.8103449
## 2:      1826.25     2 0.2824737 0.03054392 0.2304441 0.8413602
## 3:      1826.25     3 0.3205550 0.04959123 0.2736161 0.8495418
## 4:      1826.25     4 0.3743186 0.10474085 0.3501337 0.8619464
## 5:      1826.25     5 0.4258793 0.16727203 0.4032790 0.8626002
}\if{html}{\out{</div>}}
\item in-bag partial dependence indicates relationships that the model has
learned during training. This is helpful if your goal is to
interpret the model.
\item out-of-bag partial dependence indicates relationships that the model
has learned during training but using the out-of-bag data simulates
application of the model to new data. if you want to test your
model’s reliability or fairness in new data but you don’t have
access to a large testing set.
\item new data partial dependence shows how the model predicts outcomes
for observations it has not seen. This is helpful if you want to
test your model’s reliability or fairness.
}
}
}

