% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict_adjust.R
\name{predict_adjust}
\alias{predict_adjust}
\title{Adjust prediction intervals for multiple comparisons}
\usage{
predict_adjust(
  object,
  newdata,
  se.fit = FALSE,
  scale = NULL,
  df = Inf,
  interval = c("none", "confidence", "prediction"),
  level = 0.95,
  type = c("response", "terms"),
  method = "none",
  terms = NULL,
  na.action = stats::na.pass,
  pred.var = res.var/weights,
  weights = 1,
  ...
)
}
\arguments{
\item{object}{Object of class inheriting from \code{"lm"}}

\item{newdata}{An optional data frame in which to look for variables with
    which to predict.  If omitted, the fitted values are used.}

\item{se.fit}{A switch indicating if standard errors are required.}

\item{scale}{Scale parameter for std.err. calculation.}

\item{df}{Degrees of freedom for scale.}

\item{interval}{Type of interval calculation.  Can be abbreviated.}

\item{level}{Tolerance/confidence level.}

\item{type}{Type of prediction (response or model term).  Can be abbreviated.}

\item{method}{A character string indicating the type of
adjustment to make. The default choice is
\code{"none"}. The other available options are
\code{"bonferroni"}, \code{"wh"} (Working-Hotelling),
and \code{"scheffe"}.}

\item{terms}{If \code{type = "terms"}, which terms (default is all
    terms), a \code{\link{character}} vector.}

\item{na.action}{function determining what should be done with missing
    values in \code{newdata}.  The default is to predict \code{NA}.}

\item{pred.var}{the variance(s) for future observations to be assumed
    for prediction intervals.  See \sQuote{Details}.}

\item{weights}{variance weights for prediction.  This can be a numeric
    vector or a one-sided model formula.  In the latter case, it is
    interpreted as an expression evaluated in \code{newdata}.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
\code{predict_adjust} produces:

A vector of predictions if \code{interval = "none"}.

A matrix of predictions and bounds with
column names \code{fit}, \code{lwr}, and \code{upr} if
\code{interval} is set. For \code{type = "terms"} this is
a matrix with a column per term and may have an attribute
\code{"constant"}.

If \code{se.fit} is \code{TRUE}, a
list with the following components is returned:
\itemize{
 \item{\code{fit}}{: vector or matrix as above}
 \item{\code{se.fit}}{: standard error of predicted means}
 \item{\code{residual.scale}}{: residual standard deviations}
 \item{\code{df}}{: degrees of freedom for residual}
}
}
\description{
A function to produce adjusted confidence/prediction
intervals for predicted mean/new responses with a
family-wise confidence level of at least \code{level} for
\code{lm} objects (not applicable if no adjustment is
used). Internally, the function is a slight revision of
the code used in the \code{\link[stats]{predict.lm}}
function.
}
\details{
Let \code{a = 1 - level}. All intervals are computed
using the formula \code{prediction +/- m * epesd}, where
\code{m} is a multiplier and \code{epesd} is the
estimated standard deviation of the prediction error of
the \code{estimate}.

\code{method = "none"} (no correction) produces the
standard t-based confidence intervals with multiplier
\code{stats::qt(1 - a/2, df = object$df.residual)}.

\code{method = "bonferroni"} produces Bonferroni-adjusted
intervals that use the multiplier \code{m = stats::qt(1 -
a/(2 * k), df = object$df.residual)}, where \code{k} is
the number of intervals being produced.

The Working-Hotelling and Scheffe adjustments are distinct;
the Working-Hotelling typically is related to a multiple comparisons adjustment
for confidence intervals of the response mean while the Scheffe adjustment is typically
related to a multiple comparisons adjustment for prediction intervals
for a new response. However, references often uses these names
interchangeably, so we use them equivalently in this function.

\code{method = "wh"} (Working-Hotelling) or
\code{method = "scheffe"} and \code{interval =
"confidence"} produces Working-Hotelling-adjusted intervals that
use the multiplier \code{m = sqrt(p * stats::qf(level,
df1 = p, df2 = object$df.residual))}, where \code{p} is
the number of estimated coefficients in the model.

\code{method = "wh"} (Working-Hotelling) or
\code{method = "scheffe"} and \code{interval =
"prediction"} produces Scheffe-adjusted intervals that
use the multiplier \code{m = sqrt(k * stats::qf(level,
df1 = k, df2 = object$df.residual))}, where \code{k} is
the number of intervals being produced.
}
\examples{
fit <- lm(100/mpg ~ disp + hp + wt + am, data = mtcars)
newdata <- as.data.frame(rbind(
               apply(mtcars, 2, mean),
               apply(mtcars, 2, median)))
predict_adjust(fit, newdata = newdata,
               interval = "confidence",
               method = "none")
predict_adjust(fit, newdata = newdata,
               interval = "confidence",
               method = "bonferroni")
predict_adjust(fit, newdata = newdata,
               interval = "confidence",
               method = "wh")
predict_adjust(fit, newdata = newdata,
               interval = "prediction",
               method = "scheffe")
}
\references{
Bonferroni, C. (1936). Teoria statistica delle classi e
calcolo delle probabilita. Pubblicazioni del R Istituto
Superiore di Scienze Economiche e Commericiali di
Firenze, 8, 3-62.

Working, H., & Hotelling, H. (1929). Applications of the
theory of error to the interpretation of trends. Journal
of the American Statistical Association, 24(165A), 73-85.
doi:10.1080/01621459.1929.10506274

Kutner, M. H., Nachtsheim, C. J., Neter, J., & Li, W.
(2004). Applied Linear Statistical Models, 5th edition.
New York: McGraw-Hill/Irwin.
}
\seealso{
\code{\link[stats]{predict.lm}}
}
