% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/permute_profile.R
\name{permute_profile}
\alias{permute_profile}
\title{Permute soil horizon depths using boundary distinctness}
\usage{
permute_profile(
  p,
  n = 100,
  id = NULL,
  boundary.attr,
  min.thickness = 1,
  soildepth = NULL,
  new.idname = "pID"
)
}
\arguments{
\item{p}{A single-profile SoilProfileCollection}

\item{n}{Number of permutations to generate (default: 100)}

\item{id}{Over-rides \code{n}: a vector of (unique) profile IDs equal in length to number of permutations (\code{n}) to generate.}

\item{boundary.attr}{Horizon attribute containing numeric "standard deviations" reflecting boundary transition distinctness}

\item{min.thickness}{Minimum thickness of permuted horizons (default: 1)}

\item{soildepth}{Depth below which horizon depths are not permuted (default: NULL)}

\item{new.idname}{New column name to contain unique profile ID (default: pID)}
}
\value{
A SoilProfileCollection with n permutations of p.
}
\description{
"Perturbs" the \strong{boundary between horizons} using a standard deviation thickness. The thickness standard deviation corresponds roughly to the concept of "horizon boundary distinctness." This is arguably "easier" to parameterize from something like a single profile description where boundary distinctness classes (based on vertical distance of transition) are recorded for each horizon.
}
\details{
This method is most "believable" when used to \emph{gently} permute the data, on the order of moving boundaries a few centimeters in either direction. The nice thing about it is it can leverage semi-quantitative (ordered factor) levels of boundary distinctness/topography for the upper and lower boundary of individual horizons, given a set of assumptions to convert classes to a "standard deviation" (see example).

If you imagine a normal curve with its mean centered on the vertical (depth axis) at a RV horizon depth. By the Empirical Rule for Normal distribution, two "standard deviations" above or below that RV depth represent 95\% of the "volume" of the boundary.

So, a standard deviation of 1-2cm would yield a "boundary thickness" in the 3-5cm range ("clear" distinctness class).

Of course, boundaries are not symmetrical and this is at best an approximation for properties like organic matter, nutrients or salts that can have strong depth-dependence within horizons. Also, boundary topography is non-uniform. There are definitely ways to implement other distributions, but invokes more detailed assumptions about field data that are generally only semi-quantitative or are not available.

Future implementations may use boundary topography as a second hierarchical level (e.g. trig-based random functions), but think that distinctness captures the "uncertainty" about horizon separation at a specific "point" on the ground (or line in the profile quite well, and the extra variation may be hard to interpret, in general.
}
\examples{

# example with sp1 (using boundary distinctness)
data("sp1")
depths(sp1) <- id ~ top + bottom

# specify "standard deviation" for boundary thickness
#   consider a normal curve centered at boundary RV depth
# lookup table: ~maximum thickness of boundary distinctness classes, divided by 3
bound.lut <- c('V'=0.5,'A'=2,'C'=5,'G'=15,'D'=45) / 3

## V          A          C          G          D
## 0.1666667  0.6666667  1.6666667  5.0000000 15.0000000

sp1$bound_sd <- bound.lut[sp1$bound_distinct]

# hold any NA boundary distinctness constant
sp1$bound_sd[is.na(sp1$bound_sd)] <- 0

quantile(sp1$bound_sd, na.rm = TRUE)
p <- sp1[3]

# assume boundary sd is 1/12 midpoint of horizon depth
# (i.e. general relationship: SD increases (less well known) with depth)
sp1 <- mutate(sp1, midpt = (bottom - top) / 2 + top, bound_sd = midpt / 12)
quantile(sp1$bound_sd)
p <- sp1[1]

}
\seealso{
\code{\link[=random_profile]{random_profile()}} \code{\link[=sim]{sim()}} \code{\link[=hzDistinctnessCodeToOffset]{hzDistinctnessCodeToOffset()}}
}
\author{
Andrew G. Brown
}
