% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mixMunsell.R
\name{mixMunsell}
\alias{mixMunsell}
\title{Mix Munsell Colors via Spectral Library}
\usage{
mixMunsell(
  x,
  w = rep(1, times = length(x))/length(x),
  mixingMethod = c("reference", "exact", "estimate", "adaptive", "spectra"),
  n = 1,
  keepMixedSpec = FALSE,
  distThreshold = 0.025,
  ...
)
}
\arguments{
\item{x}{vector of colors in Munsell notation}

\item{w}{vector of proportions, can sum to any number}

\item{mixingMethod}{approach used to simulate a mixture:
\itemize{
\item \code{reference}  : simulate a subtractive mixture of pigments, selecting \code{n} closest reference spectra from \code{\link{munsell.spectra.wide}}
\item \code{exact}: simulate a subtractive mixture of pigments, color conversion via CIE1931 color-matching functions (see details)
\item \code{estimate} : closest Munsell chip to a weighted mean of CIELAB coordinates
\item \code{adaptive} : use reference spectra when possible, falling-back to weighted mean of CIELAB coordinates
}}

\item{n}{number of closest matching color chips (\code{mixingMethod = reference} only)}

\item{keepMixedSpec}{keep weighted geometric mean spectra, final result is a \code{list} (\code{mixingMethod = reference} only)}

\item{distThreshold}{spectral distance used to compute \code{scaledDistance}, default value is based on an analysis of spectral distances associated with adjacent Munsell color chips. This argument is only used with  \code{mixingMethod = 'reference'}.}

\item{...}{additional arguments to \code{\link{spec2Munsell}}}
}
\value{
A \code{data.frame} with the closest matching Munsell color(s):
\itemize{
\item \code{munsell}: Munsell notation of the n-closest spectra
\item \code{distance}: spectral (Gower) distance to the n-closest spectra
\item \code{scaledDistance}: spectral distance scaled by \code{distThreshold}
\item \code{mixingMethod}: method used for each mixture
}

When \code{keepMixedSpec = TRUE} then a \code{list}:
\itemize{
\item \code{mixed}: a \code{data.frame} containing the same elements as above
\item \code{spec}: spectra for the 1st closest match
}
}
\description{
Simulate mixing of colors in Munsell notation, similar to the way in which mixtures of pigments operate.
}
\details{
An accurate simulation of pigment mixtures ("subtractive" color mixtures) is incredibly complex due to factors that aren't easily measured or controlled: pigment solubility, pigment particle size distribution, water content, substrate composition, and physical obstruction to name a few. That said, it is possible to simulate reasonable, subtractive color mixtures given a reference spectra library (350-800nm) and some assumptions about pigment qualities and lighting. For the purposes of estimating a mixture of soil colors (these are pigments after all) we can relax these assumptions and assume a standard light source. The only missing piece is the spectral library for all Munsell chips in our color books.

Thankfully, \href{https://arxiv.org/ftp/arxiv/papers/1710/1710.06364.pdf}{Scott Burns has outlined the entire process}, and Paul Centore has provided a Munsell color chip \href{https://www.munsellcolourscienceforpainters.com}{reflectance spectra library}. The estimation of a subtractive mixture of soil colors can proceed as follows:
\enumerate{
\item look up the associated spectra for each color in \code{x}
\item compute the weighted (\code{w} argument) geometric mean of the spectra
\item convert the spectral mixture to the closest Munsell color via:
}
\itemize{
\item search for the closest \code{n} matching spectra in the reference library (\code{mixtureMethod = 'reference'})
\item direct conversion of spectra to closest Munsell color via \code{\link{spec2Munsell}} ( (\code{mixtureMethod = 'exact'}))
}
\enumerate{
\item suggest resulting Munsell chip(s) as the best candidate for a simulated mixture
}

Key assumptions include:
\itemize{
\item similar particle size distribution
\item similar mineralogy (i.e. pigmentation qualities)
\item similar water content.
}

For the purposes of estimating (for example) a "mixed soil color within the top 18cm of soil" these assumptions are usually valid. Again, these are estimates that are ultimately "snapped" to the nearest chip and not do not need to approach the accuracy of paint-matching systems.

A message is printed when \code{scaledDistance} is larger than 1.
}
\examples{

# try a couple different methods
cols <- c('10YR 6/2', '5YR 5/6', '10B 4/4')
mixMunsell(cols, mixingMethod = 'reference')
mixMunsell(cols, mixingMethod = 'exact')
mixMunsell(cols, mixingMethod = 'estimate')

# 2.5 values
cols <- c('10YR 2.5/2', '5YR 5/6')
mixMunsell(cols, mixingMethod = 'reference')
mixMunsell(cols, mixingMethod = 'exact')
mixMunsell(cols, mixingMethod = 'estimate')

}
\references{
Marcus, R.T. (1998). The Measurement of Color. In K. Nassau (Ed.), Color for Science, Art, and Technology (pp. 32-96). North-Holland.

\itemize{
\item{inspiration / calculations based on the work of Scott Burns: }{\url{https://arxiv.org/ftp/arxiv/papers/1710/1710.06364.pdf}}

\item{related discussion on Stack Overflow: }{\url{https://stackoverflow.com/questions/10254022/implementing-kubelka-munk-like-krita-to-mix-colours-color-like-paint/29967630#29967630}}

\item{spectral library source: }{\url{https://www.munsellcolourscienceforpainters.com/MunsellResources/SpectralReflectancesOf2007MunsellBookOfColorGlossy.txt}}

}
}
\seealso{
\code{\link{munsell.spectra}}
}
\author{
D.E. Beaudette
}
