#' Get place details including name, address, description, and other attributes
#'
#' The `/places/{placeId}` request returns details for a place.
#'
#' To request details, you use the `requested_fields` argument to specify the fields and the attributes you want from the Place, Address, Details and/or Location price groups.
#'
#' It is always recommended to specify the fields you want, however, you can also use `requested_fields=all` to return all of the attributes available. By default, The `place_id` attribute is always returned in addition to the other attributes you requested.
#'
#' The attributes available for places may vary. For example, opening hours may not be available (or applicable) for geographic places or landmarks.
#'
#' You will only be charged for attributes that contain valid values for the requested fields. If no data is available for the requested field, null or an empty collection is returned and you are not charged. You are only charged once if one or more attributes with valid values are returned from a price group.
#'
#' @param place_id a character vector of place IDs as generated by `within_extent()` or `near_point()`.
#' @param requested_fields Required. See [API Reference](https://developers.arcgis.com/rest/places/place-id-get/#requestedfields) for possible fields or refer to the [`arcgisplaces::fields`] vector.
#' @inheritParams category_details
#' @export
#' @examples
#' \dontrun{
#' place_ids <- c(
#'   "37f1062ae1c3d37511003e382b08ca32",
#'   "9cdd210841deedef0e3309bdd3fe47f1"
#' )
#'
#' res <- place_details(place_ids)
#' }
#' @references [API Documentation](https://developers.arcgis.com/rest/places/place-id-get/)
#' @returns an sf object
place_details <- function(
    place_id,
    requested_fields,
    icon = NULL,
    token = arc_token(),
    .progress = TRUE) {
  # place_id must be a character vector with no NA values
  check_character(place_id)

  # tokens are required
  obj_check_token(token)

  # check that icon isnt null
  if (!is.null(icon)) {
    # TODO change multiple = TRUE when vectorization occurs
    rlang::arg_match(icon, c("svg", "cim", "png"), multiple = FALSE)
  }

  # check request_fields are valid
  requested_fields <- rlang::arg_match(
    requested_fields,
    values = fields,
    multiple = TRUE
  )

  # get the places url
  base_url <- places_url()

  # create the base request
  # TODO: do this in a vectorized manner
  b_req <- arc_base_req(
    base_url, token,
    path = "places/",
    # TODO make requestedFields comma separated
    query = c("f" = "json", icon = icon)
  )

  all_reqs <- lapply(place_id, function(.x) {
    req <- httr2::req_url_path_append(b_req, .x)
    httr2::req_url_query(
      req,
      requestedFields = requested_fields,
      .multi = "comma"
    )
  })

  all_resps <- httr2::req_perform_parallel(
    all_reqs,
    on_error = "continue",
    progress = .progress
  )

  all_bodies <- httr2::resps_data(
    all_resps,
    httr2::resp_body_string
  )

  results <- rbind_results(
    parse_place_details(all_bodies)
  )

  # we do not include a bbox here because we do not know what it is
  # we assign afterwards
  results[["location"]] <- structure(
    results[["location"]],
    precision = 0L,
    crs = EPSG4326,
    class = c("sfc_POINT", "sfc")
  )

  # we use {wk} to recompute the bounding box and then assign
  # the bbox attribute here
  attr(results[["location"]], "bbox") <- compute_sfc_bbox(
    results[["location"]]
  )

  # create an sf object
  res <- structure(
    results,
    class = c("sf", "tbl", "data.frame"),
    sf_column = "location",
    agr = factor(
      c(place_id = NA_character_, name = NA_character_, description = NA_character_, street_address = NA_character_, extended = NA_character_, po_box = NA_character_, neighborhood = NA_character_, census_block_id = NA_character_, locality = NA_character_, designated_market_area = NA_character_, post_town = NA_character_, postcode = NA_character_, region = NA_character_, country = NA_character_, admin_region = NA_character_, drop_off = NA_character_, front_door = NA_character_, roof = NA_character_, road = NA_character_, categories = NA_character_, chains = NA_character_, email = NA_character_, fax = NA_character_, telephone = NA_character_, website = NA_character_, hours = NA_character_, icon_url = NA_character_, facebook_id = NA_character_, instagram = NA_character_, twitter = NA_character_, price = NA_character_, user = NA_character_),
      levels = c("constant", "aggregate", "identity")
    )
  )

  res
}
