#' return package dependencies from NAMESPACE file
#'
#' @param path path to NAMESPACE file
#' @param document Run function roxygenise of roxygen2 package
#' @param clean Logical. Whether to remove the original NAMESPACE before updating
#'
#' @return a vector
#' @export
#' @importFrom stringr str_match
#' @importFrom stats setNames na.omit
#' @importFrom utils read.table
#' @importFrom roxygen2 roxygenise
#'
#' @examples
#' tmpdir <- tempdir()
#' file.copy(system.file("dummypackage",package = "attachment"), tmpdir,
#'  recursive = TRUE)
#' dummypackage <- file.path(tmpdir, "dummypackage")
#' # browseURL(dummypackage)
#' att_from_namespace(path = file.path(dummypackage, "NAMESPACE"))

att_from_namespace <- function(path = "NAMESPACE", document = TRUE, clean = TRUE) {
  if (!file.exists(path)) {
    stop(path, " does not exists. ",
         "You may want to use 'attachment::att_amend_desc()' or 'roxygen2::roxygenise()' first.")}

  path <- normalizePath(path)
  if (isTRUE(document)) {
    message("Updating ", basename(dirname(path)), " documentation")
    # devtools::document(dirname(path))
    if (isTRUE(clean)) {
        file.remove(path)
        cat("# Generated by roxygen2: do not edit by hand
\n", file = path)
    }
    withr::with_dir(dirname(path), {
      roxygenise(dirname(path), roclets = NULL)
      roxygen2::load_pkgload(path = path)
    })
  }
  base <- try(readLines(path), silent = TRUE)
  base <- try(base[!grepl("^#|^$", base)], silent = TRUE)
  # base <- try(read.table(path)[["V1"]], silent = TRUE)
  if (!isTRUE(inherits(base, "try-error"))) {
    out <- na.omit(unique(c(
      unique(str_match(base, "importFrom\\(([[:alnum:]\\.]+),.*")[, 2]),
      unique(str_match(base, "import\\(([[:alnum:]\\.]+).*")[, 2])
  )))
  } else {
    message("att_from_namespace() failed,",
    " package were not retrieved from NAMESPACE")
    out <- NULL
  }
  c(out, NULL)
}
