% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sdsm.R
\name{sdsm}
\alias{sdsm}
\title{Extract backbone using the Stochastic Degree Sequence Model}
\usage{
sdsm(
  B,
  method = "RefinedNormal",
  alpha = NULL,
  signed = FALSE,
  fwer = "none",
  class = "original",
  narrative = FALSE,
  ...
)
}
\arguments{
\item{B}{An unweighted bipartite graph, as: (1) an incidence matrix in the form of a matrix, sparse \code{\link{Matrix}}, or dataframe; (2) an edgelist in the form of a two-column matrix, sparse \code{\link{Matrix}}, or dataframe; (3) an \code{\link{igraph}} object; (4) a \code{\link{network}} object.
Any rows and columns of the associated bipartite matrix that contain only zeros are automatically removed before computations.}

\item{method}{string: Specifies the method of the Poisson Binomial distribution computation used by the "ppbinom" function in \link[PoissonBinomial]{PoissonBinomial-Distribution}.
"RefinedNormal" gives quick, very accurate approximations, while "DivideFFT" gives the quickest exact computations.}

\item{alpha}{real: significance level of hypothesis test(s)}

\item{signed}{boolean: TRUE for a signed backbone, FALSE for a binary backbone (see details)}

\item{fwer}{string: type of familywise error rate correction to be applied; can be any method allowed by \code{\link{p.adjust}}.}

\item{class}{string: the class of the returned backbone graph, one of c("original", "matrix", "sparseMatrix", "igraph", "network", "edgelist").
If "original", the backbone graph returned is of the same class as \code{B}.}

\item{narrative}{boolean: TRUE if suggested text & citations should be displayed.}

\item{...}{optional arguments}
}
\value{
If \code{alpha} != NULL: Binary or signed backbone graph of class \code{class}.

If \code{alpha} == NULL: An S3 backbone object containing three matrices (the weighted graph, edges' upper-tail p-values,
edges' lower-tail p-values), and a string indicating the null model used to compute p-values, from which a backbone
can subsequently be extracted using \code{\link[=backbone.extract]{backbone.extract()}}. The \code{signed}, \code{fwer}, \code{class}, and \code{narrative} parameters
are ignored.
}
\description{
\code{sdsm} extracts the backbone of a bipartite projection using the Stochastic Degree Sequence Model.
}
\details{
The \code{sdsm} function compares an edge's observed weight in the projection \code{B*t(B)} to the distribution of weights
expected in a projection obtained from a random bipartite network where both the row vertex degrees and column
vertex degrees are \emph{approximately} fixed at their values in \code{B}. It uses the Bipartite Configuration Model \link{bicm}
to compute probabilities for the Poisson binomial distribution.

When \code{signed = FALSE}, a one-tailed test (is the weight stronger) is performed for each edge with a non-zero weight. It
yields a backbone that perserves edges whose weights are significantly \emph{stronger} than expected in the chosen null
model. When \code{signed = TRUE}, a two-tailed test (is the weight stronger or weaker) is performed for each every pair of nodes.
It yields a backbone that contains positive edges for edges whose weights are significantly \emph{stronger}, and
negative edges for edges whose weights are significantly \emph{weaker}, than expected in the chosen null model.
\emph{NOTE: Before v2.0.0, all significance tests were two-tailed and zero-weight edges were evaluated.}
}
\examples{
#A binary bipartite network of 30 agents & 75 artifacts; agents form three communities
B <- rbind(cbind(matrix(rbinom(250,1,.8),10),
                 matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.2),10)),
           cbind(matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.8),10),
                 matrix(rbinom(250,1,.2),10)),
           cbind(matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.2),10),
                 matrix(rbinom(250,1,.8),10)))

P <- B\%*\%t(B) #An ordinary weighted projection...
plot(igraph::graph_from_adjacency_matrix(P, mode = "undirected",
                                         weighted = TRUE, diag = FALSE)) #...is a dense hairball

bb <- sdsm(B, alpha = 0.05, narrative = TRUE, class = "igraph") #An SDSM backbone...
plot(bb) #...is sparse with clear communities
}
\references{
{Neal, Z. P. (2014). The backbone of bipartite projections: Inferring relationships from co-authorship, co-sponsorship, co-attendance, and other co-behaviors. \emph{Social Networks, 39}, 84-97. \doi{10.1016/j.socnet.2014.06.001}}
{Neal, Z. P., Domagalski, R., and Sagan, B. (2021). Comparing Alternatives to the Fixed Degree Sequence Model for Extracting the Backbone of Bipartite Projections. \emph{Scientific Reports}. \doi{10.1038/s41598-021-03238-3}}
}
