% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reconc.R
\name{reconc_BUIS}
\alias{reconc_BUIS}
\title{BUIS for Probabilistic Reconciliation of forecasts via conditioning}
\usage{
reconc_BUIS(
  S,
  base_forecasts,
  in_type,
  distr,
  num_samples = 20000,
  suppress_warnings = FALSE,
  seed = NULL
)
}
\arguments{
\item{S}{Summing matrix (n x n_bottom).}

\item{base_forecasts}{A list containing the base_forecasts, see details.}

\item{in_type}{A string or a list of length n. If it is a list the i-th element is a string with two possible values:
\itemize{
\item 'samples' if the i-th base forecasts are in the form of samples;
\item 'params'  if the i-th base forecasts are in the form of estimated parameters.
}

If it \code{in_type} is a string it is assumed that all base forecasts are of the same type.}

\item{distr}{A string or a list of length n describing the type of base forecasts. If it is a list the i-th element is a string with two possible values:
\itemize{
\item 'continuous' or 'discrete' if \code{in_type[[i]]}='samples';
\item 'gaussian', 'poisson' or 'nbinom' if \code{in_type[[i]]}='params'.
}

If \code{distr} is a string it is assumed that all distributions are of the same type.}

\item{num_samples}{Number of samples drawn from the reconciled distribution.}

\item{suppress_warnings}{Logical. If \code{TRUE}, no warnings about effective sample size
are triggered. If \code{FALSE}, warnings are generated. Default is \code{FALSE}. See Details.}

\item{seed}{Seed for reproducibility.}
}
\value{
A list containing the reconciled forecasts. The list has the following named elements:
\itemize{
\item \code{bottom_reconciled_samples}: a matrix (n_bottom x \code{num_samples}) containing the reconciled samples for the bottom time series;
\item \code{upper_reconciled_samples}: a matrix (n_upper x \code{num_samples}) containing the reconciled samples for the upper time series;
\item \code{reconciled_samples}: a matrix (n x \code{num_samples}) containing the reconciled samples for all time series.
}
}
\description{
Uses the Bottom-Up Importance Sampling algorithm to draw samples from the reconciled
forecast distribution, which is obtained via conditioning.
}
\details{
The parameter \code{base_forecast} is a list containing n elements where the i-th element depends on
the values of \code{in_type[[i]]} and \code{distr[[i]]}.

If \code{in_type[[i]]}='samples', then \code{base_forecast[[i]]} is a vector containing samples from the base forecast distribution.

If \code{in_type[[i]]}='params', then \code{base_forecast[[i]]} is a vector containing the estimated:
\itemize{
\item mean and sd for the Gaussian base forecast if \code{distr[[i]]}='gaussian', see \link[stats]{Normal};
\item lambda for the Poisson base forecast if \code{distr[[i]]}='poisson', see \link[stats]{Poisson};
\item mu and size for the negative binomial base forecast if \code{distr[[i]]}='nbinom', see \link[stats]{NegBinomial}.
}

See the description of the parameters \code{in_type} and \code{distr} for more details.

The order of the \code{base_forecast} list is given by the order of the time series in the summing matrix.

Warnings are triggered from the Importance Sampling step if:
\itemize{
\item weights are all zeros, then the upper is ignored during reconciliation;
\item the effective sample size is < 200;
\item the effective sample size is < 1\% of the sample size (\code{num_samples} if \code{in_type} is 'params' or the size of the base forecast if if \code{in_type} is 'samples').
}

Note that warnings are an indication that the base forecasts might have issues. Please check the base forecasts in case of warnings.
}
\examples{

library(bayesRecon)

# Create a minimal hierarchy with 2 bottom and 1 upper variable
rec_mat <- get_reconc_matrices(agg_levels=c(1,2), h=2)
S <- rec_mat$S


#1) Gaussian base forecasts

#Set the parameters of the Gaussian base forecast distributions
mu1 <- 2
mu2 <- 4
muY <- 9
mus <- c(muY,mu1,mu2)

sigma1 <- 2
sigma2 <- 2
sigmaY <- 3
sigmas <- c(sigmaY,sigma1,sigma2)

base_forecasts = list()
for (i in 1:nrow(S)) {
base_forecasts[[i]] = c(mus[[i]], sigmas[[i]])
}


#Sample from the reconciled forecast distribution using the BUIS algorithm
buis <- reconc_BUIS(S, base_forecasts, in_type="params",
                 distr="gaussian", num_samples=100000, seed=42)

samples_buis <- buis$reconciled_samples

#In the Gaussian case, the reconciled distribution is still Gaussian and can be
#computed in closed form
Sigma <- diag(sigmas^2)  #transform into covariance matrix
analytic_rec <- reconc_gaussian(S, base_forecasts.mu = mus,
                                base_forecasts.Sigma = Sigma)

#Compare the reconciled means obtained analytically and via BUIS
print(c(S \%*\% analytic_rec$bottom_reconciled_mean))
print(rowMeans(samples_buis))


#2) Poisson base forecasts

#Set the parameters of the Poisson base forecast distributions
lambda1 <- 2
lambda2 <- 4
lambdaY <- 9
lambdas <- c(lambdaY,lambda1,lambda2)

base_forecasts <- list()
for (i in 1:nrow(S)) {
 base_forecasts[[i]] = lambdas[i]
}

#Sample from the reconciled forecast distribution using the BUIS algorithm
buis <- reconc_BUIS(S, base_forecasts, in_type="params",
                          distr="poisson", num_samples=100000, seed=42)
samples_buis <- buis$reconciled_samples

#Print the reconciled means
print(rowMeans(samples_buis))

}
\references{
Zambon, L., Azzimonti, D. & Corani, G. (2024). \emph{Efficient probabilistic reconciliation of forecasts for real-valued and count time series}. \doi{10.1007/s11222-023-10343-y}.
}
\seealso{
\code{\link[=reconc_gaussian]{reconc_gaussian()}}
}
