#' Estimate Multiple Correlation Coefficients
#' (R-Squared and Adjusted R-Squared)
#' and Generate the Corresponding Sampling Distribution
#' Using Nonparametric Bootstrapping
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details R-squared (\eqn{R^{2}}) and
#'   adjusted R-squared (\eqn{\bar{R}^{2}})
#'   is estimated from bootstrap samples.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{R^{2}} and \eqn{\bar{R}^{2}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betanb` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{object}{The function argument `object`.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{R^{2}} and \eqn{\bar{R}^{2}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{R^{2}} and \eqn{\bar{R}^{2}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{R^{2}} and \eqn{\bar{R}^{2}}.}
#'     \item{fun}{Function used ("RSqMC").}
#'   }
#'
#' @inheritParams BetaNB
#'
#' @examples
#' # Fit the regression model
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#' # Generate the sampling distribution of sample covariances
#' # (use a large R, for example, R = 5000 for actual research)
#' nb <- NB(object, R = 50)
#' # Generate confidence intervals for standardized regression slopes
#' rsq <- RSqNB(nb)
#' # Methods --------------------------------------------------------
#' print(rsq)
#' summary(rsq)
#' coef(rsq)
#' vcov(rsq)
#' confint(rsq, level = 0.95)
#' @export
#' @family Beta Nonparametric Bootstrap Functions
#' @keywords betaNB rsq
RSqNB <- function(object) {
  stopifnot(
    methods::is(
      object,
      "nb"
    )
  )
  fun <- "RSqNB"
  est <- c(
    rsq = object$lm_process$summary_lm$r.squared,
    adj = object$lm_process$summary_lm$adj.r.squared
  )
  foo <- function(x) {
    rsq <- .RSqofSigma(
      sigmacap = x,
      k = object$lm_process$k
    )
    adj <- (
      1 - (
        1 - rsq
      ) * (
        (
          object$lm_process$n - 1
        ) / object$lm_process$df
      )
    )
    return(
      c(
        rsq = rsq,
        adj = adj
      )
    )
  }
  thetahatstar <- lapply(
    X = object$thetahatstar,
    FUN = foo
  )
  vcov <- stats::var(
    do.call(
      what = "rbind",
      args = thetahatstar
    )
  )
  colnames(vcov) <- rownames(vcov) <- names(est)
  out <- list(
    call = match.call(),
    object = object,
    thetahatstar = thetahatstar,
    jackknife = lapply(
      X = object$jackknife,
      FUN = foo
    ),
    vcov = vcov,
    est = est,
    fun = fun
  )
  class(out) <- c(
    "betanb",
    class(out)
  )
  return(
    out
  )
}
