% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gibbs_NPC.R
\name{gibbs_NPC}
\alias{gibbs_NPC}
\title{Gibbs sampler for Bayesian semiparametric inference with the corrected AR likelihood}
\usage{
gibbs_NPC(data, ar.order, Ntotal, burnin, thin, print_interval = 200,
  numerical_thresh = 1e-07, adaption.N = burnin, adaption.batchSize = 50,
  adaption.tar = 0.44, full_lik = F, rho.alpha = rep(1, ar.order),
  rho.beta = rep(1, ar.order), eta = T, M = 1, g0.alpha = 1,
  g0.beta = 1, k.theta = 0.01, tau.alpha = 0.001, tau.beta = 0.001,
  trunc_l = 0.1, trunc_r = 0.9, coars = F, kmax = 100 * coars + 500 *
  (!coars), L = max(20, length(data)^(1/3)))
}
\arguments{
\item{data}{numeric vector; NA values are interpreted as missing values and treated as random}

\item{ar.order}{order of the autoregressive model (integer > 0)}

\item{Ntotal}{total number of iterations to run the Markov chain}

\item{burnin}{number of initial iterations to be discarded}

\item{thin}{thinning number (postprocessing)}

\item{print_interval}{Number of iterations, after which a status is printed to console}

\item{numerical_thresh}{Lower (numerical pointwise) bound for the spectral density}

\item{adaption.N}{total number of iterations, in which the proposal variances (of rho) are adapted}

\item{adaption.batchSize}{batch size of proposal adaption for the rho_i's (PACF)}

\item{adaption.tar}{target acceptance rate for the rho_i's (PACF)}

\item{full_lik}{logical; if TRUE, the full likelihood for all observations is used; if FALSE, the partial likelihood for the last n-p observations}

\item{rho.alpha, rho.beta}{prior parameters for the rho_i's: 2*(rho-0.5)~Beta(rho.alpha,rho.beta), default is Uniform(-1,1)}

\item{eta}{logical variable indicating whether the model confidence eta 
should be included in the inference (eta=T) or fixed to 1 (eta=F)}

\item{M}{DP base measure constant (> 0)}

\item{g0.alpha, g0.beta}{parameters of Beta base measure of DP}

\item{k.theta}{prior parameter for polynomial degree k (propto exp(-k.theta*k*log(k)))}

\item{tau.alpha, tau.beta}{prior parameters for tau (inverse gamma)}

\item{trunc_l, trunc_r}{left and right truncation of Bernstein polynomial basis functions, 0<=trunc_l<trunc_r<=1}

\item{coars}{flag indicating whether coarsened or default bernstein polynomials are used (see Appendix E.1 in Ghosal and van der Vaart 2017)}

\item{kmax}{upper bound for polynomial degree of Bernstein-Dirichlet mixture (can be set to Inf, algorithm is faster with kmax<Inf due to pre-computation of basis functions, but values 500<kmax<Inf are very memory intensive)}

\item{L}{truncation parameter of DP in stick breaking representation}
}
\value{
list containing the following fields:

   \item{psd.median,psd.mean}{psd estimates: (pointwise) posterior median and mean}
   \item{psd.p05,psd.p95}{pointwise credibility interval}
   \item{psd.u05,psd.u95}{uniform credibility interval}
   \item{k,tau,V,W}{posterior traces of nonparametric correction}
   \item{rho}{posterior trace of model AR parameters (PACF parametrization)}
   \item{eta}{posterior trace of model confidence eta}
}
\description{
Obtain samples of the posterior of the corrected autoregressive likelihood in conjunction with a Bernstein-Dirichlet prior on the correction.
}
\details{
Partial Autocorrelation Structure (PACF, uniform prior) and the residual variance sigma2 (inverse gamma prior) is used as model parametrization.
A Bernstein-Dirichlet prior for c_eta with base measure Beta(g0.alpha, g0.beta) is used.
Further details can be found in the simulation study section in the referenced paper.
}
\examples{
\dontrun{

##
## Example 1: Fit a nonparametrically corrected AR(p) model to sunspot data:
##

# Use this variable to set the autoregressive model order
ar.order <- 2

data <- sqrt(as.numeric(sunspot.year))
data <- data - mean(data)

# If you run the example be aware that this may take several minutes
print("example may take some time to run")
mcmc <- gibbs_NPC(data=data, Ntotal=50000, burnin=30000, thin=4,
  ar.order=ar.order)

# Plot spectral estimates on log-scale (excluding the zero frequency).
N <- length(mcmc$psd.median)
pdgrm <- (abs(fft(data))^2 / (2*pi*length(data)))[1:N]
plot.ts(log(pdgrm[-1]), col="gray", 
  main=paste0("Sunspot NPC results on logarithmic scale for p=", ar.order))
lines(log(mcmc$psd.median[-1]))
lines(log(mcmc$psd.p05[-1]),lty=2)
lines(log(mcmc$psd.p95[-1]),lty=2)
lines(log(mcmc$psd.u05[-1]),lty=3)
lines(log(mcmc$psd.u95[-1]),lty=3)
legend(x="topright", legend=c("periodogram", "pointwise median", 
  "pointwise CI", "uniform CI"), lty=c(1,1,2,3), col=c("gray", 1, 1, 1))


##
## Example 2: Fit a nonparametrically corrected AR(p) model to high-peaked AR(1) data
##

# Use this variable to set the autoregressive model order
ar.order <- 1

n <- 256
data <- arima.sim(n=n, model=list(ar=0.95)) 
data <- data - mean(data)
psd_true <- psd_arma(pi*omegaFreq(n), ar=0.95, ma=numeric(0), sigma2=1)

# If you run the example be aware that this may take several minutes
print("example may take some time to run")
mcmc <- gibbs_NPC(data=data, Ntotal=50000, burnin=30000, thin=4,
  ar.order=ar.order)

# Plot spectral estimates
N <- length(mcmc$psd.median)
pdgrm <- (abs(fft(data))^2 / (2*pi*length(data)))[1:N]
plot.ts(pdgrm[-1], col="gray",
  main=paste0("AR(1) data NPC results for p=", ar.order))
lines(mcmc$psd.median[-1])
lines(mcmc$psd.p05[-1],lty=2)
lines(mcmc$psd.p95[-1],lty=2)
lines(mcmc$psd.u05[-1],lty=3)
lines(mcmc$psd.u95[-1],lty=3)
lines(psd_true[-1],col=2)
legend(x="topright", legend=c("periodogram", "true psd", 
  "pointwise median", "pointwise CI", "uniform CI"), lty=c(1,1,1,2,3), 
  col=c("gray", "red", 1, 1, 1))

# Compute the Integrated Absolute Error (IAE) of posterior median
cat("IAE=", mean(abs(mcmc$psd.median-psd_true)[-1]) , sep="")
}
}
\references{
C. Kirch, R. Meyer et al.
\emph{Beyond Whittle: Nonparametric Correction of a Parametric Likelihood With a Focus on Bayesian Time Series Analysis}
<arXiv:1701.04846>

S. Ghosal and A. van der Vaart (2017)
\emph{Fundamentals of Nonparametric Bayesian Inference} <DOI:10.1017/9781139029834>
}
