\name{bipartite.projection.informationloss}
\alias{bipartite.projection.informationloss}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Generate monopartite projections and calculate the associated loss of information
}
\description{
Monopartite projections are generated by calling \code{bipartite.projection} from the igraph-package, for which subsequently information loss measurements are calculated.
}
\usage{
bipartite.projection.informationloss(input_graph, vType = c(FALSE, TRUE), measures = c("graph.dH","vertex.dH","edge.dH","covLoss"), mlps = 0, logB = exp(1), parall = FALSE, preschedule = FALSE, cores = 1, verbose = FALSE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{input_graph}{
a bipartite graph from which monopartite projections and the corresponding information loss measurements are to be calculated.
}
  \item{vType}{
logical vector, indicating according to which vertex subset the projection and measurements should be done.  See also function \code{bipartite.projection} from the \pkg{igraph}--package.
}
  \item{measures}{
string or vector of strings enumerating the desired information loss measures.  Change in uncertainty for graph, vertices, and edges are selected via \verb{"graph.dh"}, \verb{"vertex.dh"}, and \verb{"edge.dh"}, respectively, while the loss of coverage for edges is selected via \verb{"covLoss"}.  All four measures can be selected at once via \verb{"complete"}.
}
  \item{mlps}{
numerical, sets the minimum size of linkage patterns that are considered during calculation of change in uncertainty.  Defaults to \verb{0}.  See details below.
}
  \item{logB}{
numerical, sets the base of the logarithm used for the entropy--based measures \verb{"graph.dH"}, \verb{"vertex.dH"}, and \verb{"edge.dH"}.  Defaults to \verb{exp(1)}.
}
  \item{parall}{
logical, switch for parallel computation, strongly recommended if multiple CPUs or cores are available.  Defaults to \verb{FALSE}.
}
  \item{preschedule}{
logical, additional option for parallel processing.  Defaults to \verb{FALSE}, for more information see details below and \code{\link[multicore]{mclapply}}.
}
  \item{cores}{
numerical, the number of cores across which calculations are distributed if parallel execution has been selected.
}
  \item{verbose}{
logical, switch for verbose mode.
}
}
\details{
For calculating the entropy--based measures \verb{graph.dH}, \verb{vertex.dH}, and \verb{edge.dH}, the number of linkage patterns (cliques) underlying a monopartite projection has to be calculated.  If during projection omitted vertices can have only one neighbour in the bipartite graph, \verb{mlps} should be set to \verb{1}.  If, in addition, retained vertices can (theoretically) have no neighbour at all, \verb{mlps} should be set to \verb{0} (which is the  default).  If none of the prior two conditions holds true, \verb{mlps} should be set to \verb{2}.

When calculating the loss of coverage for edges, subsequently for each vertex with edges the average loss of coverage over all its adjacent edges will be determined automatically and added as vertex attribute.

Concerning parameter \verb{preschedule}, \verb{TRUE} is better for short computations or large number of vertices and edges, \verb{FALSE} is better for jobs that have high variance of completion time and not too many vertices and edges.  As cases with large numbers of vertices and edges as well as a high variance of completion time are not unlikely, the optimal choice should to be determined for each case individually.
}
\value{
List of one or two monopartite (one--mode) graphs, depending on the vertex subset selection via \verb{vType}, results of selected information loss measurements are added as attributes to the corresponding entities (graph, vertices, and/or edges).
  \item{\code{H.before} }{Uncertainty before graph projection, attribute of graph, vertices, and edges}
  \item{\code{H.after} }{Uncertainty after graph projection, attribute of graph, vertices, and edges}
  \item{\code{H.delta} }{Change in uncertainty after graph projection, attribute of graph, vertices, and edges}
  \item{\code{covLoss} }{Loss of coverage of underlying linkage patterns, attribute of edges}
  \item{\code{avCovLoss} }{Average loss of coverage of adjacent edges, attribute of vertices, calculated automatically if loss of coverage for edges is measured}
}
\references{
Vogt, I. and Mestres, J. (in preparation) Assessing information loss associated with bipartite network projection.
}
\author{
Ingo Vogt <\email{bigraph.vogt@gmail.com}>
}
\note{
%%  ~~further notes~~
}

\section{Warning }{Due to its computational complexity please apply calculations of changes in uncertainty with care, and opt for parallel calculation.  Apart from total graph size, large linkage degrees of vertices omitted during projection combined with a tendency of neighbourhood overlap might be critical and result in a computation time of a several days' duration.}

\seealso{
\code{\link[igraph]{bipartite.projection}}, \code{\link[multicore]{mclapply}}
}
\examples{
data(g)

## Calculate monopartite projection and apply all information loss measures
proj <- bipartite.projection.informationloss(g, vType=TRUE, measures="complete", verbose=TRUE)

## Graph measurements
proj[[1]]$H.delta

## Vertex measurements
summary(V(proj[[1]])$H.delta)
summary(V(proj[[1]])$avCovLoss)

## Edge measurements
summary(E(proj[[1]])$H.delta)
summary(E(proj[[1]])$covLoss)

## The function is currently defined as
function(input_graph,vType=c(FALSE,TRUE),measures=c("graph.dH","vertex.dH","edge.dH","covLoss"),mlps=0,logB=exp(1),parall=FALSE,preschedule=FALSE,cores=1,verbose=FALSE) {
	## set floating point precision
	options(digits=22)
	## check if input graph is bipartite
	bipartite <- is.bipartite(input_graph)
	if (bipartite[[1]]==TRUE){
		V(input_graph)$type <- bipartite$type
		## get incidence matrix for input graph
		incMatrix <- get.incidence(input_graph,type=bipartite$type)
		## if there is no vertex attribute 'id' in the input graph, take the numbering given by igraph object
		if (length(set_intersection(as.set(list.vertex.attributes(input_graph)),as.set("id")))==0){
			V(input_graph)$id <- V(input_graph)
		}
		## projecting input graph for each selected vertex set
		projected_graphs <- list()
		for (vtype in vType) {
			if (verbose) cat("\nVertex set: ",vtype,"\n")
			if (vtype==FALSE){
				bpp <- bipartite.projection(input_graph)$proj1
			}
			else {
				if (vtype==TRUE){
					bpp <- bipartite.projection(input_graph)$proj2
				}
			}
			if (verbose) cat(" ",length(V(bpp))," Vertices\n ",length(E(bpp))," Edges\n")
			## copy 'id' attrbutes from input graph to projected graph for remaining vertex set
			V(bpp)$id <- V(input_graph)[type==vtype]$id
			V(bpp)$name <- V(input_graph)[type==vtype]$name
			## calculate information loss according to selected measures
			for (measure in measures) {
				## change in uncertainty for entire graph
				if ((measure=="complete")|(measure=="graph.dH")) {
					if (verbose) cat('\n >> Calculating graph-based information loss')
					if (vtype==TRUE) {
						bpp <- graph.dH(bpp,t(incMatrix),mlps=mlps,logB=logB,parall=parall,preschedule=preschedule,cores=cores,verbose=verbose)
					}
					else {
						bpp <- graph.dH(bpp,incMatrix,mlps=mlps,logB=logB,parall=parall,preschedule=preschedule,cores=cores,,verbose=verbose)
					}
				}
				## change in uncertainty for each vertex
				if ((measure=="complete")|(measure=="vertex.dH")) {
					if (verbose) cat('\n >> Calculating vertex-based information loss')
					if (vtype==TRUE) {
						bpp <- vertex.dH(bpp,t(incMatrix),mlps,logB,parall=parall,preschedule=preschedule,cores=cores,verbose=verbose)
					}
					else {
						bpp <- vertex.dH(bpp,incMatrix,mlps,logB,parall=parall,preschedule=preschedule,cores=cores,verbose=verbose)
					}
				}
				## change in uncertainty for each edge
				if ((measure=="complete")|(measure=="edge.dH")) {
					if (verbose) cat('\n >> Calculating edge-based information loss')
					if (vtype==TRUE) {
						bpp <- edge.dH(bpp,t(incMatrix),mlps,logB,parall=parall,preschedule=preschedule,cores=cores,verbose=verbose)
					}
					else {
						bpp <- edge.dH(bpp,incMatrix,mlps,logB,parall=parall,preschedule=preschedule,cores=cores,verbose=verbose)
					}
				}
				## loss of coverage for each edge
				if ((measure=="complete")|(measure=="covLoss")) {
					if (verbose) cat('\n >> Calculating edge-based loss of coverage')
					if (vtype==TRUE) {
						bpp <- covLoss(bpp,t(incMatrix),parall=parall,verbose=verbose)
					}
					else {
						bpp <- covLoss(bpp,incMatrix,parall=parall,verbose=verbose)
					}
					if (verbose) cat('\n >> Calculating average loss of coverage for vertices\n\n')
					bpp <- vertex.avCovLoss(bpp)
				}
			}
			projected_graphs[[length(projected_graphs)+1]] <- bpp
		}
		return(projected_graphs)
	}
	else {
		cat("Input graph is not bipartite")
	}
  }
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ graph }
\keyword{ igraph }
\keyword{ biGraph }
\keyword{ bipartite projection }

