% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/find_optimal_n.R
\name{find_optimal_n}
\alias{find_optimal_n}
\title{Search for an optimal number of clusters in a list of partitions}
\usage{
find_optimal_n(
  partitions,
  metrics_to_use = "all",
  criterion = "elbow",
  step_quantile = 0.99,
  step_levels = NULL,
  step_round_above = TRUE,
  metric_cutoffs = c(0.5, 0.75, 0.9, 0.95, 0.99, 0.999),
  n_breakpoints = 1,
  plot = TRUE
)
}
\arguments{
\item{partitions}{a \code{bioregion.partition.metrics} object (output from
\code{\link[=partition_metrics]{partition_metrics()}} or a \code{data.frame} with the first two
columns named "K" (partition name) and "n_clusters" (number of clusters) and
the following columns containing evaluation metrics (numeric values)}

\item{metrics_to_use}{character string or vector of character strings
indicating upon which metric(s) in \code{partitions} the optimal number of
clusters should be calculated. Defaults to \code{"all"} which means all
metrics available in \code{partitions} will be used}

\item{criterion}{character string indicating the criterion to be used to
identify optimal number(s) of clusters. Available methods currently include
\code{"elbow"},
\code{"increasing_step"}, \code{"decreasing_step"}, \code{"cutoff"},
\code{"breakpoints"}, \code{"min"} or
\code{"max"}. Default is \code{"elbow"}. See details.}

\item{step_quantile}{if \code{"increasing_step"} or \code{"decreasing_step"},
specify here the quantile
of differences between two consecutive k to be used as the cutoff to
identify the most important steps in \code{eval_metric}}

\item{step_levels}{if \code{"increasing_step"} or \code{"decreasing_step"}, specify
here the number of largest steps to keep as cutoffs.}

\item{step_round_above}{a \code{boolean} indicating if the optimal number of
clusters should be picked above or below the identified steps. Indeed, each
step will correspond to a sudden increase or decrease between partition X &
partition X+1: should the optimal partition be X+1
(\code{step_round_above = TRUE}) or X (\code{step_round_above = FALSE}?
Defaults to \code{TRUE}}

\item{metric_cutoffs}{if \code{criterion = "cutoff"}, specify here the cutoffs
of \code{eval_metric} at which the number of clusters should be extracted}

\item{n_breakpoints}{specify here the number of breakpoints to look for in
the curve. Defaults to 1}

\item{plot}{a boolean indicating if a plot of the first \code{eval_metric}
should be drawn with the identified optimal numbers of cutoffs}
}
\value{
a \code{list} of class \code{bioregion.optimal.n} with three elements:
\itemize{
\item{\code{args}: input arguments
}
\item{\code{evaluation_df}: the input evaluation data.frame appended with
\code{boolean} columns identifying the optimal numbers of clusters
}
\item{\code{optimal_nb_clusters}: a list containing the optimal number(s)
of cluster(s) for each metric specified in \code{"metrics_to_use"}, based on
the chosen \code{criterion}
}
\item{\code{plot}: if requested, the plot will be stored in this slot}}
}
\description{
This function aims at optimizing one or several criteria on a set of
ordered partitions. It is usually applied to find one (or several) optimal
number(s) of clusters on, for example, a hierarchical tree to cut, or a
range of partitions obtained from k-means or PAM. Users are advised to be
careful if applied in other cases (e.g., partitions which are not ordered in
an increasing or decreasing sequence, or partitions which are not related
to each other).
}
\details{
\loadmathjax

This function explores the relationship evaluation metric ~ number of
clusters, and a criterion is applied to search an optimal number of
clusters.

\strong{Please read the note section about the following criteria.}

Foreword:

Here we implemented a set of criteria commonly found in the literature or
recommended in the bioregionalisation literature. Nevertheless, we also
advocate to move
beyond the "Search one optimal number of clusters" paradigm, and consider
investigating "multiple optimal numbers of clusters". Indeed, using only one
optimal number of clusters may simplify the natural complexity of biological
datasets, and, for example, ignore the often hierarchical / nested nature of
bioregionalisations. Using multiple partitions likely avoids this
oversimplification bias and may convey more information.
See, for example, the reanalysis of Holt et al. (2013)
by \insertCite{Ficetola2017}{bioregion}, where they used deep, intermediate
and shallow cuts.

Following this rationale, several of the criteria implemented here can/will
return multiple "optimal" numbers of clusters, depending on user choices.

\strong{Criteria to find optimal number(s) of clusters}
\itemize{
\item{\code{elbow}:
This method consists in finding one elbow in the evaluation metric curve, as
is commonly done in clustering analyses. The idea is to approximate the
number of clusters at which the evaluation metric no longer increments.It is
based on a fast method finding the maximum distance between the curve and a
straight line linking the minimum and maximum number of points. The code we
use here is based on code written by Esben Eickhardt available here
\url{https://stackoverflow.com/questions/2018178/finding-the-best-trade-off-point-on-a-curve/42810075#42810075}.
The code has been modified to work on both increasing and decreasing
evaluation metrics.}
\item{\code{increasing_step} or \code{decreasing_step}:
This method consists in identifying clusters at the most important changes,
or steps, in the evaluation metric. The objective can be to either look for
largest increases (\code{increasing_step}) or largest decreases
\code{decreasing_step}. Steps are calculated based on the pairwise differences
between partitions. Therefore, this is relative to the distribution of
differences in the evaluation metric over the tested partitions. Specify
\code{step_quantile} as the quantile cutoff above which steps will be selected as
most important (by default, 0.99, i.e. the largest 1\\% steps will be
selected).Alternatively, you can also choose to specify the number of top
steps to keep, e.g. to keep the largest three steps, specify
\code{step_level = 3}. Basically this method will emphasize the most important
changes in the evaluation metric as a first approximation of where important
cuts can be chosen.

**Please note that you should choose between \code{increasing_step} and
\code{decreasing_step} depending on the nature of your evaluation metrics. For
example, for metrics that are monotonously decreasing (e.g., endemism
metrics \code{"avg_endemism" & "tot_endemism"}) with the number of clusters
should n_clusters, you should choose \code{decreasing_step}. On the contrary, for
metrics that are monotonously increasing with the number of clusters (e.g.,
\code{"pc_distance"}), you should choose \code{increasing_step}. **
}
\item{\code{cutoffs}:
This method consists in specifying the cutoff value(s) in the evaluation
metric from which the number(s) of clusters should be derived. This is the
method used by \insertCite{Holt2013}{bioregion}. Note, however, that the
cut-offs suggested by Holt et al. (0.9, 0.95, 0.99, 0.999) may be only
relevant at very large spatial scales, and lower cut-offs should be
considered at finer spatial scales.
}
\item{\code{breakpoints}:
This method consists in finding break points in the curve using a segmented
regression. Users have to specify the number of expected break points in
\code{n_breakpoints} (defaults to 1). Note that since this method relies on a
regression model, it should probably not be applied with a low number of
partitions.}

\item{\code{min} & \code{max}:
Picks the optimal partition(s) respectively at the minimum or maximum value
of the evaluation metric.}
}
}
\note{
Please note that finding the optimal number of clusters is a procedure
which normally requires decisions from the users, and as such can hardly be
fully automatized. Users are strongly advised to read the references
indicated below to look for guidance on how to choose their optimal
number(s) of clusters. Consider the "optimal" numbers of clusters returned
by this function as first approximation of the best numbers for your
bioregionalisation.
}
\examples{
comat <- matrix(sample(0:1000, size = 500, replace = TRUE, prob = 1/1:1001),
20, 25)
rownames(comat) <- paste0("Site",1:20)
colnames(comat) <- paste0("Species",1:25)

comnet <- mat_to_net(comat)

dissim <- dissimilarity(comat, metric = "all")

# User-defined number of clusters
tree1 <- hclu_hierarclust(dissim,
                          n_clust = 2:15,
                          index = "Simpson")
tree1

a <- partition_metrics(tree1,
                   dissimilarity = dissim,
                   net = comnet,
                   species_col = "Node2",
                   site_col = "Node1",
                   eval_metric = c("tot_endemism",
                                   "avg_endemism",
                                   "pc_distance",
                                   "anosim"))
                                   
find_optimal_n(a)
find_optimal_n(a, criterion = "increasing_step")
find_optimal_n(a, criterion = "decreasing_step")
find_optimal_n(a, criterion = "decreasing_step",
               step_levels = 3) 
find_optimal_n(a, criterion = "decreasing_step",
               step_quantile = .9) 
find_optimal_n(a, criterion = "decreasing_step",
               step_levels = 3) 
find_optimal_n(a, criterion = "decreasing_step",
               step_levels = 3)                 
find_optimal_n(a, criterion = "breakpoints")             

}
\references{
\insertRef{Castro-Insua2018}{bioregion}

\insertRef{Ficetola2017}{bioregion}

\insertRef{Holt2013}{bioregion}

\insertRef{Kreft2010}{bioregion}

\insertRef{Langfelder2008}{bioregion}
}
\author{
Boris Leroy (\email{leroy.boris@gmail.com}),
Maxime Lenormand (\email{maxime.lenormand@inrae.fr}) and
Pierre Denelle (\email{pierre.denelle@gmail.com})
}
