\name{expSurv}
\alias{expSurv}
\alias{predict.resexpSurv}
\alias{plot.resexpSurv}
\title{Computation of expected survival based on a prediction model}
\description{
Based on a prediction model, this function computes expected survival for patients with associated confidence intervals. The returned object can be plotted to obtain a meaningful graphical visualization.
}
\usage{
expSurv(res, traindata, method, ci.level = .95, boot = FALSE, nboot, smooth = TRUE,
  pct.group = 4, time, trace = TRUE, ncores = 1)

\method{predict}{resexpSurv}(object, newdata, ...)

\method{plot}{resexpSurv}(x, method, pr.group, print.ci = TRUE, 
  xlim, ylim, xlab, ylab, ...)
}
\arguments{
  \item{res}{an object of class '\code{resBMsel}' generated by \code{\link{BMsel}}.}
  \item{traindata}{the \code{data.frame} used to compute the \code{res} object (training set).}
  \item{method}{selection method to compute. If missing, all methods contained in \code{res} are computed.}
  \item{ci.level}{the nominal level for the two-sided confidence interval (CI) of the survival probability.}
  \item{boot}{logical value: \code{TRUE} = boostraped CI, \code{FALSE} = analytical CI.}
  \item{nboot}{number of bootstrap replicates (only used when \code{boot}=\code{TRUE}).}
  \item{smooth}{logical value indicating if smoothed B-splines should be computed.}
  \item{pct.group}{number or percentile of the prognostic-risk groups. If a single number is provided, all the groups must be defined according to Cox (1957). If percentiles are provided, the sum must be 1 (e.g. 0.164, 0.336, 0.336, 0.164).}
  \item{time}{single time point to estimate the expected survival probabilities.}
  \item{trace}{logical parameter indicating if messages should be printed.}
  \item{ncores}{number of CPUs used (for the bootstrap CI).}
  \item{object, x}{an object of class '\code{resexpSurv}' generated by \code{\link{expSurv}}.}
  \item{newdata}{\code{\link{data.frame}} containing new patients data. \code{newdata} must have the same variables as \code{traindata}.}
  \item{pr.group}{parameter for the \code{plot()} indicating the number of the prognostic-risk group for which the plot will be printed.}
  \item{print.ci}{logical parameter for the \code{plot()} indicating if CI will be printed.}
  \item{xlim, ylim, xlab, ylab}{usual parameters for \link{plot}.}
  \item{...}{other paramaters for \link{predict} or \link{plot}.}
}
\details{
   Using an object of class '\code{resBMsel}' generated by \code{\link{BMsel}}, \code{expSurv} computes expected survival at a given \code{time} and constructs confidence intervals thereof either with an analytical (\code{boot} = \code{FALSE}) or non-parametric bootstrap approach (\code{boot} = \code{TRUE}). Smoothed B-splines (logical option \code{smooth}) and categorization of the prognostic score into risk groups (using the option \code{pct.group}) may be used to obtain a meaningful graphical visualization. Predictions for new patients (\code{newdata} data frame) can be computed using \code{predict()}. Graphical visualization can be obtained using \code{plot()}.
}
\value{
  A \code{list} of length three containing the expected survival (\code{surv}) and their corresponding confidence intervals (\code{lower} and \code{upper}). Each element of the list contains a \code{matrix} of dimension number of patients x number of implemented methods. 
}
\author{Nils Ternes, Federico Rotolo, and Stefan Michiels\cr
Maintainer: Nils Ternes \email{nils.ternes@yahoo.com}
}
\examples{
########################################
# Simulated data set
########################################

## Low calculation time
  set.seed(654321)
  sdata <- simdata(
    n = 500, p = 20, q.main = 3, q.inter = 0,
    prob.tt = 0.5, alpha.tt = 0,
    beta.main = -0.8,
    b.corr = 0.6, b.corr.by = 4,
    m0 = 5, wei.shape = 1, recr = 4, fu = 2,
    timefactor = 1)

  resBM <- BMsel(
    data = sdata, 
    method = c("lasso", "lasso-pcvl"), 
    inter = FALSE, 
    folds = 5)
  
  esurv <- expSurv(
    res = resBM,
    traindata = sdata,
    boot = FALSE,
    time = 5,
    trace = TRUE)
  plot(esurv, method = "lasso-pcvl")

\dontrun{
## Moderate calculation time
  set.seed(123456)
  sdata <- simdata(
    n = 500, p = 100, q.main = 5, q.inter = 5,
    prob.tt = 0.5, alpha.tt = -0.5,
    beta.main = c(-0.5, -0.2), beta.inter = c(-0.7, -0.4),
    b.corr = 0.6, b.corr.by = 10,
    m0 = 5, wei.shape = 1, recr = 4, fu = 2,
    timefactor = 1,
    active.inter = c("bm003", "bm021", "bm044", "bm049", "bm097"))

  resBM <- BMsel(
    data = sdata, 
    method = c("lasso", "lasso-pcvl"), 
    inter = TRUE, 
    folds = 5)
  
  esurv <- expSurv(
    res = resBM,
    traindata = sdata,
    boot = TRUE,
    nboot = 100,
    smooth = TRUE,
    pct.group = 4,
    time = 5,
    ncores = 5)
  plot(esurv, method = "lasso", pr.group = 3)
}  

########################################
# Breast cancer data set
########################################

\dontrun{
  data(Breast)
  dim(Breast)

  set.seed(123456)
  resBM <-  BMsel(
    data = Breast,
    x = 4:ncol(Breast),
    y = 2:1,
    tt = 3,
    inter = FALSE,
    std.x = TRUE,
    folds = 5,
    method = c("lasso", "lasso-pcvl"))

  esurv <- expSurv(
    res = resBM,
    traindata = Breast,
    boot = FALSE,
    smooth = TRUE,
    time = 4,
    trace = TRUE
  )
  plot(esurv, method = "lasso")

}

########################################
########################################
}


\keyword{survival}
\keyword{estimations}
\keyword{confidence intervals}
