linear_predictor <- function(draws, i = NULL) {
  # compute the linear predictor (eta) for brms models
  # Args:
  #   draws: a list generated by extract_draws containing
  #          all required data and posterior samples
  #   i: An optional vector indicating the observation(s) 
  #      for which to compute eta. If NULL, eta is computed 
  #      for all all observations at once.
  # Returns:
  #   Usually an S x N matrix where S is the number of samples
  #   and N is the number of observations or length of i if specified. 
  if (length(i) == 1L && is.categorical(draws$f) && !isTRUE(draws$old_cat)) {
    # new categorical models are using multivariate syntax
    # isTRUE(.) is mandatory as draws$old_cat may be NULL
    nobs <- draws$data$N_trait * (draws$data$ncat - 1)
    i <- seq(i, nobs, draws$data$N_trait)
  }
  N <- ifelse(!is.null(i), length(i), draws$data$N) 
  
  eta <- matrix(0, nrow = draws$nsamples, ncol = N)
  if (!is.null(draws$b)) {
    eta <- eta + fixef_predictor(X = p(draws$data$X, i), b = draws$b)  
  }
  if (!is.null(draws$data$offset)) {
    eta <- eta + matrix(rep(p(draws$data$offset, i), draws$nsamples), 
                        ncol = N, byrow = TRUE)
  }
  # incorporate monotonous effects
  for (j in seq_along(draws$bm)) {
    eta <- eta + monef_predictor(Xm = p(draws$data$Xm[, j], i), 
                                 bm = as.vector(draws$bm[[j]]), 
                                 simplex = draws$simplex[[j]])
  }
  # incorporate random effects
  group <- names(draws$r)
  for (j in seq_along(group)) {
    eta <- eta + ranef_predictor(Z = p(draws$Z[[group[j]]], i), 
                                 r = draws$r[[group[j]]]) 
  }
  if (!is.null(draws$arr)) {
    eta <- eta + fixef_predictor(X = p(draws$data$Yarr, i), b = draws$arr)
  }
  if ((!is.null(draws$ar) || !is.null(draws$ma)) && !use_cov(draws$autocor)) {
    # only run when ARMA effects were modeled as part of eta
    if (!is.null(i)) {
      stop("Pointwise evaluation is not yet implemented for ARMA models.",
           call. = FALSE)
    }
    eta <- arma_predictor(standata = draws$data, ar = draws$ar, 
                          ma = draws$ma, eta = eta, link = draws$f$link)
  }
  if (is.ordinal(draws$f)) {
    if (!is.null(draws$p)) {
      eta <- cse_predictor(Xp = p(draws$data$Xp, i), p = draws$p, 
                           eta = eta, ncat = draws$data$max_obs)
    } else {
      eta <- array(eta, dim = c(dim(eta), draws$data$max_obs - 1))
    } 
    for (k in 1:(draws$data$max_obs - 1)) {
      if (draws$f$family %in% c("cumulative", "sratio")) {
        eta[, , k] <-  draws$Intercept[, k] - eta[, , k]
      } else {
        eta[, , k] <- eta[, , k] - draws$Intercept[, k]
      }
    }
  } else if (is.categorical(draws$f)) {
    if (isTRUE(draws$old_cat)) {
      # deprecated as of brms > 0.8.0
      if (!is.null(draws$p)) {
        eta <- cse_predictor(Xp = p(draws$data$X, i), p = draws$p, 
                             eta = eta, ncat = draws$data$max_obs)
      } else {
        eta <- array(eta, dim = c(dim(eta), draws$data$max_obs - 1))
      }
    } else {
      ncat1 <- draws$data$ncat - 1 
      eta <- array(eta, dim = c(nrow(eta), ncol(eta) / ncat1, ncat1))
    }
  }
  eta
}

nonlinear_predictor <- function(draws, i = NULL) {
  # compute the non-linear predictor (eta) for brms models
  # Args:
  #   draws: a list generated by extract_draws containing
  #          all required data and posterior samples
  #   i: An optional vector indicating the observation(s) 
  #      for which to compute eta. If NULL, eta is computed 
  #      for all all observations at once.
  # Returns:
  #   Usually an S x N matrix where S is the number of samples
  #   and N is the number of observations or length of i if specified. 
  nlmodel_list <- list()
  nlpars <- names(draws$nonlinear)
  for (j in seq_along(nlpars)) {
    nlmodel_list[[nlpars[j]]] <- 
      linear_predictor(draws$nonlinear[[nlpars[j]]], i = i)
  }
  nlmodel_list[names(draws$C)] <- p(draws$C, i, row = FALSE)
  # evaluate non-linear predictor
  out <- try(with(nlmodel_list, eval(draws$nlform)), silent = TRUE)
  if (is(out, "try-error")) {
    if (grepl("could not find function", out)) {
      out <- rename(out, "Error in eval(expr, envir, enclos) : ", "")
      stop(paste0(out, "Most likely this is because you used a Stan ",
                  "function in the non-linear model formula that ",
                  "is not defined in R. Currently, you have to write ",
                  "this function yourself making sure that it is ",
                  "vectorized. I apologize for the inconvenience."),
           call. = FALSE)
    } else {
      out <- rename(out, "^Error :", "", fixed = FALSE)
      stop(out, call. = FALSE)
    }
  }
  out
}

fixef_predictor <- function(X, b) {
  # compute eta for fixed effects
  # Args:
  #   X: fixed effects design matrix
  #   b: fixed effects samples
  stopifnot(is.matrix(X))
  stopifnot(is.matrix(b))
  tcrossprod(b, X)
}

monef_predictor <- function(Xm, bm, simplex) {
  # compute eta for monotonous effects
  # Args:
  #   Xm: a vector of data for the monotonous effect
  #   bm: montonous effects samples
  #   simplex: matrix of samples of the simplex
  #            corresponding to bm
  stopifnot(is.vector(Xm))
  stopifnot(is.matrix(simplex))
  bm <- as.vector(bm)
  for (i in 2:ncol(simplex)) {
    # compute the cumulative representation of the simplex 
    simplex[, i] <- simplex[, i] + simplex[, i - 1]
  }
  simplex <- cbind(0, simplex)
  bm * simplex[, Xm + 1]
}

ranef_predictor <- function(Z, r) {
  # compute eta for random effects
  # Args:
  #   Z: sparse random effects design matrix
  #   r: random effects samples
  # Returns: 
  #   linear predictor for random effects
  Matrix::as.matrix(Matrix::tcrossprod(r, Z))
}

arma_predictor <- function(standata, eta, ar = NULL, ma = NULL, 
                           link = "identity") {
  # compute eta for ARMA effects
  # ToDo: use C++ for this function
  # Args:
  #   standata: the data initially passed to Stan
  #   eta: previous linear predictor samples
  #   ar: autoregressive samples (can be NULL)
  #   ma: moving average samples (can be NULL)
  #   link: the link function as character string
  # Returns:
  #   new linear predictor samples updated by ARMA effects
  S <- nrow(eta)
  Kar <- ifelse(is.null(ar), 0, ncol(ar))
  Kma <- ifelse(is.null(ma), 0, ncol(ma))
  K <- max(Kar, Kma, 1)
  Ks <- 1:K
  Y <- link(standata$Y, link)
  N <- length(Y)
  tg <- c(rep(0, K), standata$tg)
  E <- array(0, dim = c(S, K, K + 1))
  e <- matrix(0, nrow = S, ncol = K)
  zero_mat <- e
  zero_vec <- rep(0, S)
  for (n in 1:N) {
    if (Kma) {
      # add MA effects
      eta[, n] <- eta[, n] + rowSums(ma * E[, 1:Kma, K])
    }
    e[, K] <- Y[n] - eta[, n]
    if (n < N) {
      I <- which(n < N & tg[n + 1 + K] == tg[n + 1 + K - Ks])
      E[, I, K + 1] <- e[, K + 1 - I]
    }
    if (Kar) {
      # add AR effects
      eta[, n] <- eta[, n] + rowSums(ar * E[, 1:Kar, K])
    }
    # allows to keep the object size of e and E small
    E <- abind(E[, , 2:(K + 1), drop = FALSE], zero_mat)
    if (K > 1) {
      e <- cbind(e[, 2:K, drop = FALSE], zero_vec)
    }
  }
  eta
}

cse_predictor <- function(Xp, p, eta, ncat) {
  # add category specific effects to eta
  # Args:
  #   Xp: category specific design matrix 
  #   p: category specific effects samples
  #   ncat: number of categories
  #   eta: linear predictor matrix
  # Returns: 
  #   linear predictor including category specific effects as a 3D array
  stopifnot(is.matrix(Xp))
  stopifnot(is.matrix(p))
  ncat <- max(ncat)
  eta <- array(eta, dim = c(dim(eta), ncat - 1))
  indices <- seq(1, (ncat - 1) * ncol(Xp), ncat - 1) - 1
  Xp <- t(Xp)
  for (k in 1:(ncat - 1)) {
    eta[, , k] <- eta[, , k] + p[, indices + k, drop = FALSE] %*% Xp
  }
  eta
}

get_eta <- function(i, draws, ordinal = FALSE) {
  # extract the linear predictor of observation i from draws
  # Args:
  #   i: a vector (typically of length 1) indicating the
  #      observation for which to extract eta
  #   draws: a list generated by extract_draws
  #   ordinal: does draws$eta have 3 dimensions?
  if (!is.null(draws$eta)) {
    if (ordinal) {
      draws$eta[, i, , drop = FALSE]  
    } else {
      draws$eta[, i]  
    }
  } else if (!is.null(draws$nonlinear)) {
    nonlinear_predictor(draws, i = i)
  } else {
    linear_predictor(draws, i = i)
  }
}